/*
 * nodoka_rc_style.c
 * This file is part of gtk-nodoka-engine
 *
 * Copyright (C) 2007, 2008 - Martin Sourada, Daniel Geiger
 *
 * gtk-nodoka-engine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * gtk-nodoka-engine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gtk-nodoka-engine; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */ 

#include "nodoka_style.h"
#include "nodoka_rc_style.h"

#include "animation.h"

static void nodoka_rc_style_init (NodokaRcStyle * style);
#ifdef HAVE_ANIMATION
static void nodoka_rc_style_finalize (GObject * object);
#endif
static void nodoka_rc_style_class_init (NodokaRcStyleClass * klass);
static GtkStyle *nodoka_rc_style_create_style (GtkRcStyle * rc_style);
static guint nodoka_rc_style_parse (GtkRcStyle * rc_style,
									GtkSettings * settings, GScanner * scanner);
static void nodoka_rc_style_merge (GtkRcStyle * dest, GtkRcStyle * src);


static GtkRcStyleClass *parent_class;

GType nodoka_type_rc_style = 0;

enum
{
	TOKEN_SCROLLBARCOLOR = G_TOKEN_LAST + 1,
	TOKEN_CONTRAST,
	TOKEN_HILIGHT_RATIO,
	TOKEN_ROUNDNESS,
	TOKEN_MENUBARSTYLE,
	TOKEN_TOOLBARSTYLE,
	TOKEN_LISTVIEWHEADERSTYLE,
	TOKEN_LISTVIEWSTYLE,
	TOKEN_SCROLLBARSTYLE,
	TOKEN_ANIMATION,
	TOKEN_GRADIENTS,
	TOKEN_STRIPES,
	TOKEN_BULLETCOLOR,
	TOKEN_FOCUSINNER,
	TOKEN_FOCUSFILL,

	/* stuff to ignore */
	TOKEN_SQUAREDSTYLE,

	TOKEN_TRUE,
	TOKEN_FALSE
};

static struct
{
	const gchar *name;
	guint token;
}
theme_symbols[] =
{
	{
	"scrollbar_color", TOKEN_SCROLLBARCOLOR},
	{
	"contrast", TOKEN_CONTRAST},
	{
	"hilight_ratio", TOKEN_HILIGHT_RATIO},
	{
	"roundness", TOKEN_ROUNDNESS},
	{
	"menubarstyle", TOKEN_MENUBARSTYLE},
	{
	"toolbarstyle", TOKEN_TOOLBARSTYLE},
	{
	"listviewheaderstyle", TOKEN_LISTVIEWHEADERSTYLE},
	{
	"listviewstyle", TOKEN_LISTVIEWSTYLE},
	{
	"scrollbarstyle", TOKEN_SCROLLBARSTYLE},
	{
	"animation", TOKEN_ANIMATION},
	{
	"gradients", TOKEN_GRADIENTS},
	{
	"stripes", TOKEN_STRIPES},
	{
	"bullet_color", TOKEN_BULLETCOLOR},
	{
	"focus_inner", TOKEN_FOCUSINNER},
	{
	"focus_fill", TOKEN_FOCUSFILL},
		/* stuff to ignore */
	{
	"squaredstyle", TOKEN_SQUAREDSTYLE},
	{
	"TRUE", TOKEN_TRUE},
	{
	"FALSE", TOKEN_FALSE}
};


void
nodoka_rc_style_register_type (GTypeModule * module)
{
	static const GTypeInfo object_info = {
		sizeof (NodokaRcStyleClass),
		(GBaseInitFunc) NULL,
		(GBaseFinalizeFunc) NULL,
		(GClassInitFunc) nodoka_rc_style_class_init,
		NULL,					/* class_finalize */
		NULL,					/* class_data */
		sizeof (NodokaRcStyle),
		0,						/* n_preallocs */
		(GInstanceInitFunc) nodoka_rc_style_init,
		NULL
	};

	nodoka_type_rc_style = g_type_module_register_type (module,
														GTK_TYPE_RC_STYLE,
														"NodokaRcStyle",
														&object_info, 0);
}

static void
nodoka_rc_style_init (NodokaRcStyle * nodoka_rc)
{
	nodoka_rc->flags = 0;

	nodoka_rc->has_scrollbar_color = FALSE;
	nodoka_rc->contrast = 1.0;
	nodoka_rc->hilight_ratio = 0.909090909;
	nodoka_rc->roundness = 3;
	nodoka_rc->menubarstyle = 0;
	nodoka_rc->toolbarstyle = 1;
	nodoka_rc->listviewheaderstyle = 1;
	nodoka_rc->listviewstyle = 0;
	nodoka_rc->scrollbarstyle = 0;
	nodoka_rc->animation = FALSE;
	nodoka_rc->gradients = TRUE;
	nodoka_rc->stripes = FALSE;
	nodoka_rc->bullet_color.red = 0;
	nodoka_rc->bullet_color.green = 23808;
	nodoka_rc->bullet_color.blue = 49664;
	nodoka_rc->focus_inner = TRUE;
	nodoka_rc->focus_fill = FALSE;	
}

#ifdef HAVE_ANIMATION
static void
nodoka_rc_style_finalize (GObject * object)
{
	/* cleanup all the animation stuff */
	nodoka_animation_cleanup ();

	if (G_OBJECT_CLASS (parent_class)->finalize != NULL)
		G_OBJECT_CLASS (parent_class)->finalize (object);
}
#endif


static void
nodoka_rc_style_class_init (NodokaRcStyleClass * klass)
{
	GtkRcStyleClass *rc_style_class = GTK_RC_STYLE_CLASS (klass);
#ifdef HAVE_ANIMATION
	GObjectClass *g_object_class = G_OBJECT_CLASS (klass);
#endif

	parent_class = g_type_class_peek_parent (klass);

	rc_style_class->parse = nodoka_rc_style_parse;
	rc_style_class->create_style = nodoka_rc_style_create_style;
	rc_style_class->merge = nodoka_rc_style_merge;

#ifdef HAVE_ANIMATION
	g_object_class->finalize = nodoka_rc_style_finalize;
#endif
}

static guint
theme_parse_boolean (GtkSettings * settings,
					 GScanner * scanner, gboolean * retval)
{
	guint token;

	/* Skip 'ANIMATION' */
	token = g_scanner_get_next_token (scanner);

	token = g_scanner_get_next_token (scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token (scanner);
	if (token == TOKEN_TRUE)
		*retval = TRUE;
	else if (token == TOKEN_FALSE)
		*retval = FALSE;
	else
		return TOKEN_TRUE;

	return G_TOKEN_NONE;
}

static guint
theme_parse_color (GtkSettings * settings, GScanner * scanner, GdkColor * color)
{
	guint token;

	/* Skip 'blah_color' */
	token = g_scanner_get_next_token (scanner);

	token = g_scanner_get_next_token (scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	return gtk_rc_parse_color (scanner, color);
}

static guint
theme_parse_ratio (GtkSettings * settings, GScanner * scanner, double *ratio)
{
	guint token;

	/* Skip 'ratio' */
	token = g_scanner_get_next_token (scanner);

	token = g_scanner_get_next_token (scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token (scanner);
	if (token != G_TOKEN_FLOAT)
		return G_TOKEN_FLOAT;

	*ratio = scanner->value.v_float;

	return G_TOKEN_NONE;
}

static guint
theme_parse_int (GtkSettings * settings, GScanner * scanner, guint8 * style)
{
	guint token;

	/* Skip '*style' */
	token = g_scanner_get_next_token (scanner);

	token = g_scanner_get_next_token (scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token (scanner);
	if (token != G_TOKEN_INT)
		return G_TOKEN_INT;

	*style = scanner->value.v_int;

	return G_TOKEN_NONE;
}

static guint
nodoka_gtk2_rc_parse_dummy (GtkSettings * settings,
							GScanner * scanner, gchar * name)
{
	guint token;

	/* Skip option */
	token = g_scanner_get_next_token (scanner);

	/* print a warning. Isn't there a way to get the string from the scanner? */
	g_scanner_warn (scanner,
					"Nodoka configuration option \"%s\" is not supported and will be ignored.",
					name);

	/* equal sign */
	token = g_scanner_get_next_token (scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	/* eat whatever comes next */
	token = g_scanner_get_next_token (scanner);

	return G_TOKEN_NONE;
}

static guint
nodoka_rc_style_parse (GtkRcStyle * rc_style,
					   GtkSettings * settings, GScanner * scanner)
{
	static GQuark scope_id = 0;
	NodokaRcStyle *nodoka_style = NODOKA_RC_STYLE (rc_style);

	guint old_scope;
	guint token;
	guint i;

	/* Set up a new scope in this scanner. */

	if (!scope_id)
		scope_id = g_quark_from_string ("nodoka_theme_engine");

	/* If we bail out due to errors, we *don't* reset the scope, so the
	 * error messaging code can make sense of our tokens.
	 */
	old_scope = g_scanner_set_scope (scanner, scope_id);

	/* Now check if we already added our symbols to this scope
	 * (in some previous call to nodoka_rc_style_parse for the
	 * same scanner.
	 */

	if (!g_scanner_lookup_symbol (scanner, theme_symbols[0].name))
	{
		g_scanner_freeze_symbol_table (scanner);
		for (i = 0; i < G_N_ELEMENTS (theme_symbols); i++)
			g_scanner_scope_add_symbol (scanner, scope_id,
										theme_symbols[i].name,
										GINT_TO_POINTER (theme_symbols[i].
														 token));
		g_scanner_thaw_symbol_table (scanner);
	}

	/* We're ready to go, now parse the top level */
	token = g_scanner_peek_next_token (scanner);
	while (token != G_TOKEN_RIGHT_CURLY)
	{
		switch (token)
		{
		case TOKEN_SCROLLBARCOLOR:
			token =
				theme_parse_color (settings, scanner,
								   &nodoka_style->scrollbar_color);
			nodoka_style->flags |= NDK_FLAG_SCROLLBAR_COLOR;
			nodoka_style->has_scrollbar_color = TRUE;
			break;
		case TOKEN_CONTRAST:
			token =
				theme_parse_ratio (settings, scanner, &nodoka_style->contrast);
			nodoka_style->flags |= NDK_FLAG_CONTRAST;
			break;
		case TOKEN_HILIGHT_RATIO:
			token =
				theme_parse_ratio (settings, scanner,
								   &nodoka_style->hilight_ratio);
			nodoka_style->flags |= NDK_FLAG_HILIGHT_RATIO;
			break;
		case TOKEN_ROUNDNESS:
			token =
				theme_parse_int (settings, scanner, &nodoka_style->roundness);
			nodoka_style->flags |= NDK_FLAG_ROUNDNESS;
			break;
		case TOKEN_MENUBARSTYLE:
			token =
				theme_parse_int (settings, scanner,
								 &nodoka_style->menubarstyle);
			nodoka_style->flags |= NDK_FLAG_MENUBARSTYLE;
			break;
		case TOKEN_TOOLBARSTYLE:
			token =
				theme_parse_int (settings, scanner,
								 &nodoka_style->toolbarstyle);
			nodoka_style->flags |= NDK_FLAG_TOOLBARSTYLE;
			break;
		case TOKEN_LISTVIEWHEADERSTYLE:
			token =
				theme_parse_int (settings, scanner,
								 &nodoka_style->listviewheaderstyle);
			nodoka_style->flags |= NDK_FLAG_LISTVIEWHEADERSTYLE;
			break;
		case TOKEN_LISTVIEWSTYLE:
			token =
				theme_parse_int (settings, scanner,
								 &nodoka_style->listviewstyle);
			nodoka_style->flags |= NDK_FLAG_LISTVIEWSTYLE;
			break;
		case TOKEN_SCROLLBARSTYLE:
			token =
				theme_parse_int (settings, scanner,
								 &nodoka_style->scrollbarstyle);
			nodoka_style->flags |= NDK_FLAG_SCROLLBARSTYLE;
			break;
		case TOKEN_ANIMATION:
			token =
				theme_parse_boolean (settings, scanner,
									 &nodoka_style->animation);
			nodoka_style->flags |= NDK_FLAG_ANIMATION;
			break;
		case TOKEN_GRADIENTS:
			token =
				theme_parse_boolean (settings, scanner,
									 &nodoka_style->gradients);
			nodoka_style->flags |= NDK_FLAG_GRADIENTS;
			break;
		case TOKEN_STRIPES:
			token =
				theme_parse_boolean (settings, scanner, &nodoka_style->stripes);
			nodoka_style->flags |= NDK_FLAG_STRIPES;
			break;
		case TOKEN_BULLETCOLOR:
			token =
				theme_parse_color (settings, scanner, &nodoka_style->bullet_color);
			nodoka_style->flags |= NDK_FLAG_BULLETCOLOR;
			break;
		case TOKEN_FOCUSINNER:
			token =
				theme_parse_boolean (settings, scanner,
									 &nodoka_style->focus_inner);
			nodoka_style->flags |= NDK_FLAG_FOCUSINNER;
			break;
		case TOKEN_FOCUSFILL:
			token =
				theme_parse_boolean (settings, scanner,
									 &nodoka_style->focus_fill);
			nodoka_style->flags |= NDK_FLAG_FOCUSFILL;
			break;

			/* stuff to ignore */
		case TOKEN_SQUAREDSTYLE:
			token =
				nodoka_gtk2_rc_parse_dummy (settings, scanner, "squaredstyle");
			break;

		default:
			g_scanner_get_next_token (scanner);
			token = G_TOKEN_RIGHT_CURLY;
			break;
		}

		if (token != G_TOKEN_NONE)
			return token;

		token = g_scanner_peek_next_token (scanner);
	}

	g_scanner_get_next_token (scanner);

	g_scanner_set_scope (scanner, old_scope);

	return G_TOKEN_NONE;
}

static void
nodoka_rc_style_merge (GtkRcStyle * dest, GtkRcStyle * src)
{
	NodokaRcStyle *dest_w, *src_w;
	NodokaRcFlags flags;

	parent_class->merge (dest, src);

	if (!NODOKA_IS_RC_STYLE (src))
		return;

	src_w = NODOKA_RC_STYLE (src);
	dest_w = NODOKA_RC_STYLE (dest);

	flags = (~dest_w->flags) & src_w->flags;

	if (flags & NDK_FLAG_SCROLLBAR_COLOR)
	{
		dest_w->has_scrollbar_color = TRUE;
		dest_w->scrollbar_color = src_w->scrollbar_color;
	}
	if (flags & NDK_FLAG_CONTRAST)
		dest_w->contrast = src_w->contrast;
	if (flags & NDK_FLAG_HILIGHT_RATIO)
		dest_w->hilight_ratio = src_w->hilight_ratio;
	if (flags & NDK_FLAG_ROUNDNESS)
		dest_w->roundness = src_w->roundness;
	if (flags & NDK_FLAG_MENUBARSTYLE)
		dest_w->menubarstyle = src_w->menubarstyle;
	if (flags & NDK_FLAG_TOOLBARSTYLE)
		dest_w->toolbarstyle = src_w->toolbarstyle;
	if (flags & NDK_FLAG_LISTVIEWHEADERSTYLE)
		dest_w->listviewheaderstyle = src_w->listviewheaderstyle;
	if (flags & NDK_FLAG_LISTVIEWSTYLE)
		dest_w->listviewstyle = src_w->listviewstyle;
	if (flags & NDK_FLAG_SCROLLBARSTYLE)
		dest_w->scrollbarstyle = src_w->scrollbarstyle;
	if (flags & NDK_FLAG_ANIMATION)
		dest_w->animation = src_w->animation;
	if (flags & NDK_FLAG_GRADIENTS)
		dest_w->gradients = src_w->gradients;
	if (flags & NDK_FLAG_STRIPES)
		dest_w->stripes = src_w->stripes;
	if (flags & NDK_FLAG_BULLETCOLOR)
		dest_w->bullet_color = src_w->bullet_color;
	if (flags & NDK_FLAG_FOCUSINNER)
		dest_w->focus_inner = src_w->focus_inner;
	if (flags & NDK_FLAG_FOCUSFILL)
		dest_w->focus_fill = src_w->focus_fill;

	dest_w->flags |= src_w->flags;
}


/* Create an empty style suitable to this RC style
 */
static GtkStyle *
nodoka_rc_style_create_style (GtkRcStyle * rc_style)
{
	return GTK_STYLE (g_object_new (NODOKA_TYPE_STYLE, NULL));
}
