/*
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013 Apple Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#pragma once

#include <wtf/RefCountDebugger.h>

namespace WTF {

// This base class holds the non-template methods and attributes.
// The RefCounted class inherits from it reducing the template bloat
// generated by the compiler (technique called template hoisting).
class WTF_EMPTY_BASE_CLASS RefCountedBase : public RefCountDebugger {
public:
    void ref() const
    {
        willRef(m_refCount);
        ++m_refCount;
    }

    bool hasOneRef() const { return m_refCount == 1; }
    unsigned refCount() const { return m_refCount; }

protected:
    RefCountedBase() = default;

    ~RefCountedBase()
    {
        willDestroy(m_refCount);
        RELEASE_ASSERT(m_refCount == 1);
    }

    // Returns true if the pointer should be deleted.
    bool derefBase() const
    {
        willDeref(m_refCount);

        unsigned tempRefCount = m_refCount - 1;
        if (!tempRefCount) {
            willDelete();
            return true;
        }

        m_refCount = tempRefCount;
        return false;
    }

private:
    mutable unsigned m_refCount { 1 };
};

template<typename T> class RefCounted : public RefCountedBase {
    WTF_MAKE_NONCOPYABLE(RefCounted);
    WTF_DEPRECATED_MAKE_FAST_ALLOCATED(RefCounted);
public:
    void deref() const
    {
        if (derefBase())
            delete const_cast<T*>(static_cast<const T*>(this));
    }

protected:
    RefCounted() = default;
    ~RefCounted() = default;
} SWIFT_RETURNED_AS_UNRETAINED_BY_DEFAULT;

template<typename T>
inline void ref(T* obj)
{
    obj->ref();
}

template<typename T>
inline void deref(T* obj)
{
    obj->deref();
}

} // namespace WTF

using WTF::RefCounted;
