open OUnit2

open Mirage_crypto.Uncommon
open Mirage_crypto_pk

open Test_common

(*
# CAVS 11.2
# "SigGen" information for "dsa2_values"
# Mod sizes selected: L=1024, N=160, SHA-1 L=1024, N=160, SHA-224 L=1024, N=160, SHA-256 L=1024, N=160, SHA-384 L=1024, N=160, SHA-512 L=2048, N=224, SHA-1 L=2048, N=224, SHA-224 L=2048, N=224, SHA-256 L=2048, N=224, SHA-384 L=2048, N=224, SHA-512 L=2048, N=256, SHA-1 L=2048, N=256, SHA-224 L=2048, N=256, SHA-256 L=2048, N=256, SHA-384 L=2048, N=256, SHA-512 L=3072, N=256, SHA-1 L=3072, N=256, SHA-224 L=3072, N=256, SHA-256 L=3072, N=256, SHA-384 L=3072, N=256, SHA-512
# Generated on Tue Aug 16 11:21:08 2011
 *)

let dsa_test ~priv ~msg ?k ~r ~s ~hash _ =
  let hmsg = Digestif.(digest_string hash msg |> to_raw_string hash) in
  let (r', s') = Dsa.sign ~mask:`No ~key:priv ?k hmsg in
  assert_oct_equal ~msg:"computed r" r r' ;
  assert_oct_equal ~msg:"computed s" s s' ;
  (* now with masking *)
  let (r', s') = Dsa.sign ~key:priv ?k hmsg in
  assert_oct_equal ~msg:"computed r (masked)" r r' ;
  assert_oct_equal ~msg:"computed s (masked)" s s' ;
  let pub = Dsa.pub_of_priv priv in
  assert_bool "verify of given r, s"
    (Dsa.verify ~key:pub (r, s) hmsg) ;
  assert_bool "verify of computed r, s"
    (Dsa.verify ~key:pub (r', s') hmsg)

let params ~p ~q ~g = vx p, vx q, vx g

let priv_of f ~p ~q ~gg ~x ~y =
  match Dsa.priv ~fips:true ~p:(f p) ~q:(f q) ~gg:(f gg) ~x:(f x) ~y:(f y) () with
  | Ok dsa -> dsa
  | Error (`Msg m) -> invalid_arg "bad DSA private key %s" m

let priv_of_cs  = priv_of Z_extra.of_octets_be
let priv_of_hex = priv_of (fun cs -> vx cs |> Z_extra.of_octets_be)

let case_of ~domain ~hash ~x ~y ~k ~r ~s ~msg =
  let (p, q, gg) = domain in
  let priv   = priv_of_cs ~p ~q ~gg ~x:(vx x) ~y:(vx y)
  and (r, s) = vx r, vx s
  and k      = Z_extra.of_octets_be (vx k)
  and msg    = vx msg in
  dsa_test ~priv ~msg ~k ~r ~s ~hash

let sha1_cases =
  (*[mod = L=1024, N=160, SHA-1] *)
  let domain =
    params ~p:"a8f9cd201e5e35d892f85f80e4db2599a5676a3b1d4f190330ed3256b26d0e80a0e49a8fffaaad2a24f472d2573241d4d6d6c7480c80b4c67bb4479c15ada7ea8424d2502fa01472e760241713dab025ae1b02e1703a1435f62ddf4ee4c1b664066eb22f2e3bf28bb70a2a76e4fd5ebe2d1229681b5b06439ac9c7e9d8bde283"
~q:"f85f0f83ac4df7ea0cdf8f469bfeeaea14156495"
~g:"2b3152ff6c62f14622b8f48e59f8af46883b38e79b8c74deeae9df131f8b856e3ad6c8455dab87cc0da8ac973417ce4f7878557d6cdf40b35b4a0ca3eb310c6a95d68ce284ad4e25ea28591611ee08b8444bd64b25f3f7c572410ddfb39cc728b9c936f85f419129869929cdb909a6a3a99bbe089216368171bd0ba81de4fe33"
  in
  let case = case_of ~domain ~hash:Digestif.sha1 in

  [ case
~msg:"3b46736d559bd4e0c2c1b2553a33ad3c6cf23cac998d3d0c0e8fa4b19bca06f2f386db2dcff9dca4f40ad8f561ffc308b46c5f31a7735b5fa7e0f9e6cb512e63d7eea05538d66a75cd0d4234b5ccf6c1715ccaaf9cdc0a2228135f716ee9bdee7fc13ec27a03a6d11c5c5b3685f51900b1337153bc6c4e8f52920c33fa37f4e7"
~x:"c53eae6d45323164c7d07af5715703744a63fc3a"
~y:"313fd9ebca91574e1c2eebe1517c57e0c21b0209872140c5328761bbb2450b33f1b18b409ce9ab7c4cd8fda3391e8e34868357c199e16a6b2eba06d6749def791d79e95d3a4d09b24c392ad89dbf100995ae19c01062056bb14bce005e8731efde175f95b975089bdcdaea562b32786d96f5a31aedf75364008ad4fffebb970b"
~k:"98cbcc4969d845e2461b5f66383dd503712bbcfa"
~r:"50ed0e810e3f1c7cb6ac62332058448bd8b284c0"
~s:"c6aded17216b46b7e4b6f2a97c1ad7cc3da83fde" ;

case
~msg:"d2bcb53b044b3e2e4b61ba2f91c0995fb83a6a97525e66441a3b489d9594238bc740bdeea0f718a769c977e2de003877b5d7dc25b182ae533db33e78f2c3ff0645f2137abc137d4e7d93ccf24f60b18a820bc07c7b4b5fe08b4f9e7d21b256c18f3b9d49acc4f93e2ce6f3754c7807757d2e1176042612cb32fc3f4f70700e25"
~x:"e65131d73470f6ad2e5878bdc9bef536faf78831"
~y:"29bdd759aaa62d4bf16b4861c81cf42eac2e1637b9ecba512bdbc13ac12a80ae8de2526b899ae5e4a231aef884197c944c732693a634d7659abc6975a773f8d3cd5a361fe2492386a3c09aaef12e4a7e73ad7dfc3637f7b093f2c40d6223a195c136adf2ea3fbf8704a675aa7817aa7ec7f9adfb2854d4e05c3ce7f76560313b"
~k:"87256a64e98cf5be1034ecfa766f9d25d1ac7ceb"
~r:"a26c00b5750a2d27fe7435b93476b35438b4d8ab"
~s:"61c9bfcb2938755afa7dad1d1e07c6288617bf70" ;

case
~msg:"d5431e6b16fdae31481742bd394758beb8e24f31947e19b7ea7b458521882270c1f43192aa050f4485145af8f3f9c5142d68b85018d2ec9cb7a37ba12ed23e73b95fd680fba3c61265e9f5a0a027d70fad0c8aa08a3cbfbe99018d0045386173e5fae225faebe0cef5dd45910f400a86c2be4e15252a16de4120a267be2b594d"
~x:"20bcabc6d9347a6e79b8e498c60c44a19c73258c"
~y:"23b4f404aa3c575e550bb320fdb1a085cd396a10e5ebc6771da62f037cab19eacd67d8222b6344038c4f7af45f5e62b55480cbe2111154ca9697ca76d87b56944138084e74c6f90a05cf43660dff8b8b3fabfcab3f0e4416775fdf40055864be102b4587392e77752ed2aeb182ee4f70be4a291dbe77b84a44ee34007957b1e0"
~k:"7d9bcfc9225432de9860f605a38d389e291ca750"
~r:"3f0a4ad32f0816821b8affb518e9b599f35d57c2"
~s:"ea06638f2b2fc9d1dfe99c2a492806b497e2b0ea" ;

case
~msg:"85662b697550e4915c29e338b624b912845d6d1a920d9e4c1604dd47d692bc7c0ffb95ae614e852bebaf1573758ad01c713cac0b476e2f121745a3cfeeffb2441ff6abfb9bbeb98aa634ca6ff541947dcc9927659d44f95c5ff9170fdc3c86473cb601ba31b487fe5936bac5d9c632cbcc3db06246ba01c55a038d797fe3f6c3"
~x:"52d1fbe687aa0702a51a5bf9566bd51bd569424c"
~y:"6bc36cb3fa61cecc157be08639a7ca9e3de073b8a0ff23574ce5ab0a867dfd60669a56e60d1c989b3af8c8a43f5695d503e3098963990e12b63566784171058eace85c728cd4c08224c7a6efea75dca20df461013c75f40acbc23799ebee7f3361336dadc4a56f305708667bfe602b8ea75a491a5cf0c06ebd6fdc7161e10497"
~k:"960c211891c090d05454646ebac1bfe1f381e82b"
~r:"3bc29dee96957050ba438d1b3e17b02c1725d229"
~s:"0af879cf846c434e08fb6c63782f4d03e0d88865" ;

case
~msg:"87b6e75b9f8e99c4dd62adb693dd5890edff1bd0028f4ef849df0f1d2ce6b181fc3a55aea6d0a1f0aecab8ed9e248a00e96be794a7cfba1246efb710ef4b37471cef0a1bcf55cebc8d5ad071612bd237efedd5102362db07a1e2c7a6f15e09fe64ba42b60a2628d869ae05ef611fe38d9ce15eeec9bb3decc8dc17809f3b6e95"
~x:"c86a54ec5c4ec63d7332cf43ddb082a34ed6d5f5"
~y:"014ac746d3605efcb8a2c7dae1f54682a262e27662b252c09478ce87d0aaa522d7c200043406016c0c42896d21750b15dbd57f9707ec37dcea5651781b67ad8d01f5099fe7584b353b641bb159cc717d8ceb18b66705e656f336f1214b34f0357e577ab83641969e311bf40bdcb3ffd5e0bb59419f229508d2f432cc2859ff75"
~k:"6c445cee68042553fbe63be61be4ddb99d8134af"
~r:"637e07a5770f3dc65e4506c68c770e5ef6b8ced3"
~s:"7dfc6f83e24f09745e01d3f7ae0ed1474e811d47" ;

case
~msg:"2259eead2d6bbc76d49213ea0dc8b7350a97699f22341044c3940782364ac9ea683179a438a5ea45998df97c2972dae03851f5be23fa9f04182e79ddb2b56dc8652393ecb27f3f3b7c8a8d761a86b3b8f4d41a07b4be7d02fddefc42b928124a5a45b9f4609042209b3a7f585bd514cc39c00effcc42c7fe70fa83edf8a32bf4"
~x:"aee6f213b9903c8069387e64729a08999e5baf65"
~y:"0fe74045d7b0d472411202831d4932396f242a9765e92be387fd81bbe38d845054528b348c03984179b8e505674cb79d88cc0d8d3e8d7392f9aa773b29c29e54a9e326406075d755c291fcedbcc577934c824af988250f64ed5685fce726cff65e92d708ae11cbfaa958ab8d8b15340a29a137b5b4357f7ed1c7a5190cbf98a4"
~k:"e1704bae025942e2e63c6d76bab88da79640073a"
~r:"83366ba3fed93dfb38d541203ecbf81c363998e2"
~s:"1fe299c36a1332f23bf2e10a6c6a4e0d3cdd2bf4" ;

case
~msg:"219e8df5bf881590430ece608250f7670dc56537249302429e28ecfeb9ceaaa54910a69490f765f3df82e8b01cd7d76e561d0f6ce226ef3cf752cada6febdc5bf00d67947f92d420516b9e37c96c8f1f2da0b075097c3bda758a8d91bd2ebe9c75cf147f254c256963b33b67d02b6aa09e7d7465d038e50195ece4189b41e768" ~x:"699f1c07aa458c6786e770b40197235fe49cf21a" ~y:"3a41b0678ff3c4dde20fa39772bac31a2f18bae4bedec9e12ee8e02e30e556b1a136013bef96b0d30b568233dcecc71e485ed75c922afb4d0654e709bee84993792130220e3005fdb06ebdfc0e2df163b5ec424e836465acd6d92e243c86f2b94b26b8d73bd9cf722c757e0b80b0af16f185de70e8ca850b1402d126ea60f309" ~k:"5bbb795bfa5fa72191fed3434a08741410367491" ~r:"579761039ae0ddb81106bf4968e320083bbcb947" ~s:"503ea15dbac9dedeba917fa8e9f386b93aa30353" ;

case
~msg:"2da79d067885eb3ccf5e293ae3b1d8225322203abb5adfde3b0f53bbe24c4fe001541e1183d870a997f1f9460100b5d711923180154345287a0214cf1cac37b7a47dfbb2a0e8ce4916f94ebd6fa54e315b7a8eb5b63cd954c5ba05c1bf7e33a4e8a151f32d2877b01729c1ad0e7c01bb8ae723c995183803e45636520ea38ca1" ~x:"d6e08c20c82949ddba93ea81eb2fea8c595894dc" ~y:"56f7272210f316c51af8bfc45a421fd4e9b1043853271b7e79f40936f0adcf262a86097aa86e19e6cb5307685d863dba761342db6c973b3849b1e060aca926f41fe07323601062515ae85f3172b8f34899c621d59fa21f73d5ae97a3deb5e840b25a18fd580862fd7b1cf416c7ae9fc5842a0197fdb0c5173ff4a4f102a8cf89" ~k:"6d72c30d4430959800740f2770651095d0c181c2" ~r:"5dd90d69add67a5fae138eec1aaff0229aa4afc4" ~s:"47f39c4db2387f10762f45b80dfd027906d7ef04" ;

case
~msg:"ba30d85be357e7fb29f8a07e1f127baaa24b2ee027f64cb5efeec6aaeabcc7345c5d556ebf4bdc7a61c77c7b7ea43c73babc18f7b4807722da239e45ddf249849cbbfe3507112ebf87d7ef560c2e7d391ed8424f8710cea41685143e3006f81b68fbb4d5f9644c7cd10f7092ef2439b8d18c0df655e00289372a4166385d640c" ~x:"50018482864c1864e9db1f04bde8dbfd3875c76d" ~y:"0942a5b7a72ab116ead29308cf658dfe3d55d5d61afed9e3836e64237f9d6884fdd827d2d5890c9a41ae88e7a69fc9f345ade9c480c6f08cff067c183214c227236cedb6dd1283ca2a602574e8327510221d4c27b162143b7002d8c726916826265937b87be9d5ec6d7bd28fb015f84e0ab730da7a4eaf4ef3174bf0a22a6392" ~k:"df3a9348f37b5d2d4c9176db266ae388f1fa7e0f" ~r:"448434b214eee38bde080f8ec433e8d19b3ddf0d" ~s:"0c02e881b777923fe0ea674f2621298e00199d5f" ;

case
~msg:"83499efb06bb7ff02ffb46c278a5e92630ac5bc3f9e53dd2e78ff15e368c7e31aad77cf771f35fa02d0b5f135208a4afdd867bb2ec26ea2e7dd64cdef237508a38b27f39d8b22d45cac5a68a90b6ea76058645f6356a9344d36f00ec6652eaa4e9bae7b694f9f1fc8c6c5e86fadc7b27a219b5c1b2ae80a725e5f61165fe2edc" ~x:"ae56f66b0a9405b9cca54c60ec4a3bb5f8be7c3f" ~y:"a01542c3da410dd57930ca724f0f507c4df43d553c7f69459939685941ceb95c7dcc3f175a403b359621c0d4328e98f15f330a63865baf3e7eb1604a0715e16eed64fd14b35d3a534259a6a7ddf888c4dbb5f51bbc6ed339e5bb2a239d5cfe2100ac8e2f9c16e536f25119ab435843af27dc33414a9e4602f96d7c94d6021cec" ~k:"8857ff301ad0169d164fa269977a116e070bac17" ~r:"8c2fab489c34672140415d41a65cef1e70192e23" ~s:"3df86a9e2efe944a1c7ea9c30cac331d00599a0e" ;

case
~msg:"f23ee79eb4fce5cbf3b08d65a1803d2e3e191d3580a44d177d8ff069f90784d012ca5746e6dd6638dfe8413f1db3d8fe282c2160f5dd96607dd63d610f791dfc10abad18721587101cec8a2a12913cfbada3a5b7593958b9bfa6e9af3af5d71ff17ec72aaaeecaaffc5d174e629a090297e94cdfe988d9bf6c80827c23df5137" ~x:"a62079b4f45772bf17b85d7560e3be4e521439eb" ~y:"229a26dcaff29ed1a7264ed0f77d676239b9ba1ef4778e7dd640e8aa6fabdc1f1bd3f582e211bd01c26b3d9d3bffe7199f9ed45d764cd9d0e844b385cb34e6de22370ebc6ba41db409d63f50c1ac09bed00cdc2b7c55223c596b7e133ba25ba9e78f33502f8dd52f32a667a7683e504047817963238d9629a918a0ceebaad518" ~k:"c01acd36910f2f2bff608386b81c35a0a7c0b378" ~r:"8d388ec7f2863dd5b7c99ac93505d1580bf2e0c7" ~s:"76ae9317696d37f2d8bd61c47733e9455b61d347" ;

case
~msg:"6836255e6e659de4ffb535892d466a3bea09693e587eb5bd50f44f8a22f11697057d68660bc6562400d587baac1c19d330ff794a70df5300a5211c72541a56d0ff2af02a278ed2db1df94ccb2026d3138b2d924245021ee835d3c17b0b3b7677def85611227f6ce2913e7cb446a479b95acfd0105c25e4656fbc56c2a10a22b3" ~x:"7861e82e66b6caea54b159c59c887ec27b2e915f" ~y:"a7bbc35423510edfebf79c4e2e56986f2806d11116bcae90a716f05dcbfc46dcbfebe2ec946c40f9cc8c1a7439cdd04e270122ec1c3baca83811a9f1bdaed9b1172150af1c8ce1c5d502dfe5f4e8467e50605087a8f8c5f45ca672fd049eec0a06f5e01f782d51f3ba56404bf1388065552fc87ad21ac0fa4027a145d0b0d9e6" ~k:"4e22cfa2e8ca2b33a9fd91ff4837fc205864e8b1" ~r:"c0ab43d309a5e94b6ef4db9943306e6d966fc9b5" ~s:"07ec5aa1928f19fc3a420f29b935bac46124c0e2" ;

case
~msg:"4b0845c99db348294f1d83166b27f448ec29ab7965464477f45444f44672a409ddcafaf35e91faf401eca7498e3268caa2d96bf1aa840c0e1ed43a5ab60888fcf02b2f8a2c89daa598adf0b7d2dace9210efd41ab496a1e73a182da430c1d043e249a1289c91809c8c7298cfdbb0ae438b00936c283a0ec2d79cdc81c9ce3c2d" ~x:"6f2d3b09fae6910dd975870db3a2c19d97169491" ~y:"541a9c45e165d3d91e71bb1370d7c032c360322aa15e99d8c1c16ea35c8c193224a06467ab77a65478c467b3f20cb0c5fdb8c84cefa69566a594a2aa54c3a948ebc1ea7e6c3d28d380cbd01740634c96b76d6a03cc6eba0afa7226f23fc10a18b0b6f97270dfa038160960b5b839ba66af50fda07245810e80d38b6693e8a9ce" ~k:"8588557c12ec6fe176b0be7bbd8b482ad78f1fef" ~r:"44286019c1d53103980616940c028bad3217f78d" ~s:"4b372bf527c515f58025699a45f2021ef18e11b9" ;

case
~msg:"4597c1ca0b0764be31fa73ccc589116cc8d0a31605f2550eb37fa569b2496c4f34321d61bb8e49f858c8671b7437fc15f269dd2d4146470b817dfe3069225ddd3cd4a6c977fb6cfc0d43264a7bf6659283e140e4c89ab2e8a4d0ede6274961d655bd79c7e47880a741fb0180c325b5b7d2f7b8a57aed52d0206a83bb69a9d7a4" ~x:"49f6c1ac8e639bcc99b2d9d1f1e325713f29b97c" ~y:"5315adf90e196946be6b04c5414da1fafd98b0d17c3a39000a00091b7b56574b1ecd026eabb25be9ecd0ad691df2b7bf7eecd6ad95bb4d7d17ac747060ee7e3eb5c6fb7557cf7e8003a620e43e587d0612854472c3ad851839f744159411a33876aec365eb0491dec80ba14cba2d11dec42af4a4bf9c99312a2ae7e5462a2adf" ~k:"8ef1c5976ac8caf74df65d9ecdbe78a6490bc220" ~r:"90d547712bc0cebbd3ebd18a63d9b92a03953050" ~s:"34ea6176b4c63043295f129a4895e14ee5816563" ;

case
~msg:"18c62a40b52347a473f57aa668eebb4484beb5f10fdc51779e6770106c0d122eb6356ae53a3379e270edca39015da3005770c7b2a5afd11217993153ff43a0b26db01aa2a493de061492a0aa3f229b5abd1aff29395e31b063504eb35620219ba29997f92a52e1b2e6ff207480fd13d58ff0290eec5aabf23b84943eea20a43c" ~x:"396cbe3e71d74f6db795c38d49c32d78eab03397" ~y:"3b738246f9e38cebf4542ced3fc0c0096aeb9e9a3ad928f4dd4745d875fe6e20fb65556d06696432ecffd55b334940c6e23c903f0aa4a1335f7394c55070586baac86c38cc198ebaf15401259528c55192e9298d2a0c8914daf2ad00259fe72555c3c0442e38c1e6e5020928c6e6571a0a98f6f485e43791ae8aaab180461fa4" ~k:"04bfe51616f5c244d2e01648362f5bbe5fa73501" ~r:"29b7c0f90d624f8d587efd3f49f97da70f6e63e7" ~s:"222a3d9ffca0dcf57937e89c92538e32e7a8680f"
]

let sha224_cases =
  (* [mod = L=1024, N=160, SHA-224] *)
  let domain = params ~p:"8b9b32f5ba38faad5e0d506eb555540d0d7963195558ca308b7466228d92a17b3b14b8e0ab77a9f3b2959a09848aa69f8df92cd9e9edef0adf792ce77bfceccadd9352700ca5faecf181fa0c326db1d6e5d352458011e51bd3248f4e3bd7c820d7e0a81932aca1eba390175e53eada197223674e3900263e90f72d94e7447bff"
~q:"bc550e965647fb3a20f245ec8475624abbb26edd"
~g:"11333a931fba503487777376859fdc12f7c687b0948ae889d287f1b7a712ad220ae4f1ce379d0dbb5c9abf419621f005fc123c327e5055d1850634c36d397e689e111d598c1c3636b940c84f42f436846e8e7fcad9012ceda398720f32fffd1a45ab6136ce417069207ac140675b8f86dd063915ae6f62b0cec729fbd509ac17"
  in
  let case = case_of ~domain ~hash:Digestif.sha224 in

  [ case
~msg:"fb2128052509488cad0745ed3e6312850dd96ddaf791f1e624e22a6b9beaa65319c325c78ef59cacba0ccfa722259f24f92c17b77a8f6d8e97c93d880d2d8dbbbedcf6acefa06b0e476ca2013d0394bd90d56c10626ef43cea79d1ef0bc7ac452bf9b9acaef70325e055ac006d34024b32204abea4be5faae0a6d46d365ed0d9" ~x:"6e2e31bbfc670944d7a7120e39a981520614d8a8" ~y:"7e339f3757450390160e02291559f30bed0b2d758c5ccc2d8d456232bb435ae49de7e7957e3aad9bfdcf6fd5d9b6ee3b521bc2229a8421dc2aa59b9952345a8fc1de49b348003a9b18da642d7f6f56e3bc665131ae9762088a93786f7b4b72a4bcc308c67e2532a3a5bf09652055cc26bf3b18833598cffd7011f2285f794557" ~k:"8cb35d255505a4c41421e562d10827266aa68663" ~r:"afee719e7f848b54349ccc3b4fb26065833a4d8e" ~s:"734efe992256f31325e749bc32a24a1f957b3a1b" ;

case
~msg:"02971e0cdd48ae2331db9c6285e9880e96104fa7a9f378dfea718e63efe98352fe4d35a2bc94b3a888cfb88b8b7d9f6c8c54e48613f32c9946ffe6e9a4b7108ececdda41bc151b3d8724b61f5b83a4e27476914387b0488e41be54f63aa773175eb373a3641e6e7950eee8faf048a841f107d30cf9be268493231545d8984694" ~x:"0b448f49a085a52a03d7f668a1d6fb87f2e221ac" ~y:"633bb757b3c0e3b7867bf845301ea4e39f75c9759c223f46ce266d406b9df5db501fb826b6e61cba6104c604458c90799f2a36ab51166d0e83b770840624fedc35ebfb9853419e7e09b32b4bd652da4f1ce973ac2620c966b61e35a3f216439a8de1a104f172e1b6e2878112a66c34d16a9aef3ac24a34af5edbf39818a3e2ef" ~k:"4481a4be9db6821e3b0a08c9c82603631971a682" ~r:"92c65e07462d668b06dd45b608784965897838bc" ~s:"2e40adf41cafb8048c793c7092a7e823515b6cfa" ;

case
~msg:"062e82fb43236ee17ebfaa3d363b9b873d0fe41444c74cef7f7e3bd81f723fd90fd148a28e997585413695113757758aa4dd275f70b375f8903c7be46e3a3ad3190cd04971abd2f1db192ef0d2b98bbb80181a721a5809928b5bca5c118a2911132ad233cd27c7e41adfccfeb4e952874bfa819661182975e44d37c61734759c" ~x:"a4a25a8bb1c2ba69f9611939b591032b96333fa3" ~y:"3b0a091dfca05dce61e9f05b15b07487d2e3ea4f568dc9ac752d42c0aa771ae0ccc372ce9774fb9fd16a30cb3759bb1989488ce85db7cdfa506476acec644c21168f2db1f36efe0230c6fb8f1f2ae4eaf1799d5e29e212467b11bfbc1eebed142d7a017262cd8735e3e29d8e0c4a6e766c07d7aa9f8d176f536087bfecf4c414" ~k:"a7135820910f041b27321534a17bb1f33ac51aca" ~r:"ba554124874d06a6cef62740e15821ccddbfe6f3" ~s:"5962be757d75b0f17d15482ebb595ca4e9fbfe22" ;

case
~msg:"4fca074844eae247d19c06e92032ae8e773043e2e1f45d400e9dcebbde5d65e7c1423b0390161991c026f38a0e2bfeef40dae18741737b1d535ab46b566a1b672fc22dec86747a7c7638fa65047f1ede36ad43f6aedf51b5bf2979adf4d9a94ed802a29de5603b704770b32c8b946a32e1b6054cd70c3add025cc9371d1e404d" ~x:"1f15cafca282083e82d7e54258647b2914418986" ~y:"40b638c94b1e719a337d83358699c70cd867ff888c655a5f5a1de8732d058bf027d4747efe3b8dedca3276de5a58f136ed35cff03030f672da65c71f18e58278ddfc7b9b50a248eff9236874ee3cb0d0a35b7b2ee185b139ea84eed7bffc5094ab8743a75374bc36c7d69d5f3e6fe5f3ef1f9358f00a3c5892fff41ed6afee3b" ~k:"128ab9677c7ade5e1c02a8427650ff054db6390e" ~r:"651a389d8ca50d6e3273cabbe71cd84cccd02361" ~s:"3401fe47b3812daa8c020c9bd42609cbebdfa728" ;

case
~msg:"4d9630fe058998ca5b80ae62f3f73dc85bee291509843ac00240d13d55251ae53b37794783b97d53e042cab26f8c84de0a70f5b43051fbefb3e43f08f5d2e8aad9e2de2717412dbb902acc8849adc04d06fed8c1421c4cfe8b81ee7f5ac5d4f0c0b68e80b6f88fd3c7d5b32022572b0a681bd2d4df2d047b0b23b6887145afe1" ~x:"1485f719b8be77c78829baa0d2c322df60174476" ~y:"727b6528357d6705c20d31358f641934fdec63cc66df98837d2f378164e15fa0842207acf3220c8023a9f4f8d2057165b3c849eaeb5376e3fad11785f1d0261779aaedd53b1e52798007eb4c8e83b1ff321b620d883391a14fa47fec4901d96ec232eabb4a0bb4453368fef5176c67135649979d3214d3fb67a1319ac54feb01" ~k:"8f4cc1254c787ec8cbf54405105f7ef83ffdeee0" ~r:"9ca3e433504c557ba1aac66469781175cdfb4ad5" ~s:"72145dfa5279dd82ae99604d16a2b8df71b95320" ;

case
~msg:"62b9d601e30b42a279c7e04df3ca8d8140a55cd5876c7e9181c73575e4c4f921a94e4e2d0bdd7ba98600d652e5df5be9464e7a9011ab486960f69d57ece1d2c4af9324457c1e3d83fba4265beb47407e4761dbc949d5bd67fee4a476a4d5a93d77acda96a221a0a31e0f024b3f0b8234c015238f3258daa085ae9f4e1aa7b1cc" ~x:"43c76a9a00045cdfb2e7927b5c8730e006423c05" ~y:"5f6dfb064caddf644af399e33a672565766761d55ac0b84bead42c3980e7e396043744361778f04dcb698e4563853420fecacd594af828f57df541d9e4de899d61f04f6379c1c96246d152369395242a1c2e70eef8f35417a0ffdb039282516ce21b85687904c511087f113e5142f027f1179712edcbce27939ab15ec49c085f" ~k:"5e4b5e4595e31397422c7a4487ae51051289be61" ~r:"331920a7b79e3cfa7638e409d9702aafd08fbec6" ~s:"071d06e6cd301515f37b60690afa219fe5083d96" ;

case
~msg:"0006e09c20376442e689bf2d34268fd69109c1301ea66cbe90394cc0f41f94822c28845819b9a98764d2f7262e98891487ff55b05bd69e18b7cad41bd98e137566b6041c739db11f78e567cac02f33f140d19a4805002545375daebfd7dcbea33242e73c8e269149d7eb9db9f9006e17acb736b5e977645ab651b81225c5e543" ~x:"16f89d97dd3b31c191495173ae0e145c6ce185d6" ~y:"1b1f725664d75bdcb2a5a4c653061c460799dd48bf1e6b03e13c71d83e3fdb506fa94e6cafb5dbdead88a33d23d4e9287b4707e1fba871b97c9a48f930cdccba0dc06a4f0a8bfbb4e14d0b4d5a0871fa1341caec7bc08138713121d419769f31203508df71947265644fdc6137d8e466c8cb0ce985340cb2e279b4ce9315a772" ~k:"475b5aa12ff77d49e4c8171f80d3d1f15147ed12" ~r:"b6aa833b825184729af308f81bf5e58e2d7e9284" ~s:"5453b4b2e3fc802b2f977d0cf6eb7f5c16673fa3" ;

case
~msg:"e04a71f2b5c176a0db17a983a17dec588c00f42c9aa3026b5eb440f07a2140c2ed84024e0531ea7788dfeaa91883fb6a9841c17dcfd312968adb00e556bc7eb3021f57b7a16894fa4fe12ec93dfd494a0a1c693d6ade154ef648c05552da41224d4922d1861d9f7671b8ce6ce448e895ea0eed25802e3350ec08ae79f2d61e0f" ~x:"3eda44e3c38380df7a4f47d8e1024596238bcef1" ~y:"687e16309b06817b93236dd990faef9e232eb81cb9c7d6dae4fdd4f8e7ca933e185b1da622d7c7fa35e0a3906f915ded14ba96d6035b46bd6ca5fe172af94e081fb0b9a9583a458bd206c1e87f97a57d00d6eade19ec56ac2e9bbd8c15df356ee7b12c91311a38fc3315cfde9ff462ca6adff2808b3f8e805ee915ae885ca957" ~k:"aeaa655b6febfec50b05562c3f358865533e4736" ~r:"14892b1ec7fc716c75a17f7ad2e41ec6faa78836" ~s:"72cc56a9890e8bdf1a53d3acc6f89137264f9ff8" ;

case
~msg:"5e8eb96b5c6ad75d3dab1e28bb2ce751ecc31611a019e8d4b561c7e4533cc7ab73bd9de931e8c54c51c5711e6c276a8ed92f4bb457ddf28233da2ca3e3013c56e3cd2bc61d4d4e0e22cf6361304e56d68b315ca5d3fcc472a7eef8cca575204dd084a21a99ba67fddbf90df7c6c658761734bce13c3d22d80b6fb9bece551492" ~x:"0b55f99ad958a766eaf5ac20a127a4df1b946bae" ~y:"50b0f7605911bce6ed5ecff1e3c1816fbbf03a1479a0820603ffa715aef9ffbccbd067579cbbc8c87c392e85bbe929a0b5e1059faae6f9121df49c66a049a98a90d84c70a21312bf837f4723993d0ec0ac4c2a7ffb9d400957b39fb83e951ef41362452cf458d784c43fe822ea7a7abbea0a6998321a93819d2d282c7884f5c2" ~k:"9e9b9afb43a7157761f6c2011138d2f65ac1cba9" ~r:"7399b120d4bfbd6dc4064d2f3f8f0ca5c362b2d8" ~s:"2302d81d7ebb2417eef45d88941b070ecab11cab" ;

case
~msg:"da91c692cdb0a59562e2b664dcfe7554ac589d57f82246c4a8a3f9573bf47b257eb8f93447c1ebab13dce53d6f4416fb2c6c36303ed97885cf7a6caef055f7e3145ef3838c31877fad7a8883ffc84ebd973f8c06d17cdd339bb3371f9d3d4f2d9f0b80ae2bcc878b4af78f845eac4f2aacee6a9451daf814a44e927bb5428820" ~x:"ac701252c773ba36711b9731afdc077c5d3f9271" ~y:"678b3958ed24fc84942054f49d9e6f27bbac7de3a4a52af9ffcb9ce6c1fb8bdd99db0e80c868ac547c4cfc782de7ebcf6943b2e46433c670178de0104bd6fc25dc3054db9c48c12706e1dea35e163be36a4ab721950c028b0546f1719ff2edd81b2b7974fb9b121224ccfaabc47e9e629a97bc6ba42691ca3f649ccac47d0f1e" ~k:"6cabf2c0e2890b2b393da3ea6aac2782216efa73" ~r:"6f1579edcf437584d3e939fa5b002eee83e3b614" ~s:"71208a87a4cf2b3a9b65477773b0096d452dae60" ;

case
~msg:"0f2edc87f4d2942c4693b064a511b93f790c60dc149a1b0b7041af5183bc0f42234134b284270e4c7e53614f7ecfe711de0efb28336d0bb359c86e8be8839f583211e9174832b3d41ee6d21864ac6186fd1db920dda65b25966c5951ab8a2050dda87d1d72e3032852ad43da9fb430e850022b4bb6cc9cb90e428f3a5ca32a62" ~x:"588f40e3eb813cd22a41c9cdeadb6895a348db3c" ~y:"3a978e9022a8f7a0caa91f275bf9cf7648e1b9a31a0702d8acdbf59affb5467fb07a8f7e5b4c86775ac4efb609b946f05a3f13034db94acc64057f906d1854910de538f84367181c618e96c3f922547d408ee6408b7a70acedc75de8ae445c5d4dd5def4a352d2528234070cc720700c14ce12d2f36990d36b29d7b00596e34b" ~k:"8ac2fe7bcd690a7239d294b22725b818d262a446" ~r:"b6ea9cdb211c4560b3d592e93af6d5f133b64b9b" ~s:"6242e45a472fa8147cb5253dbddebae31ef31e4e" ;

case
~msg:"d12fc1983e0095e9e2b6b8743fb34386cc4821540e3efe1a29f84cf7e63e2a0668d551f912ad2221b5a3d6b9ebd12136def5e6690e1d32aae919f9f1cf5d24d62a46a9a9a604bae11b9c0866350367204a920b589a317ddfbb877f9fad6b0d3629af9635da46933151c0d9a20aaabddd3df5d049659b2860ddb8b20963261ea0" ~x:"67cd81c7d6ac2d8bd44ef26297ac02ecba41f73f" ~y:"10b7b14ad29fb34d7a39f3e953051f456a0cd1233ef54d90a4adc82dfbd9fa7a85628f11039632b47ba9daeca6e463ec4644f5e2a2a4bf95d392e8c9c9f287a20ba45a198815ca0e9ba854d7f3c79d9037fa1417724fb7f02799b1c2b2bcc79d64367b90c06d1789dcc6de57ca19fcefafc04fcce29c8f495ed564f5d9a112ca" ~k:"1341e376e8919e01991e5e48b8e0c7255929b3d2" ~r:"360617965f65a68abcb83dbf2d886a1a10ca05de" ~s:"71abb6acbf7e653d2ebc3cb7149b51cc0c92fba8" ;

case
~msg:"87a6dfb8487f16f6fef1d68bc31469ac210ea55387965bb4458ca0d00d6c46858be28a019ce914c39c2479f321f0252ca4a8bd681a5b358a093fc8341c31bc47c618403f93322b443084ce5818490b74e83c3866b8164bbcf79bf82539f428c9351c40b10d773cbe1cbaa8c9800a6dcf38d85515e2dff5d4f8a965ecaef37e38" ~x:"22bbb8468f3e90768d347cb3492f64db2a23f721" ~y:"75ef5d5f67022426f531e9b8ca9115921d5a5c446bcdf1af701b605bae687dff8d1e7b3c4f8b289537eb09a7461d6688a3711974371a5b73a2082e991410118666ccd94f444977d0c89ba36162de023aa519037a6ba6305417dad3f2dc38756a40046491e8ee80c4f147825b8c021b5d09a2422d39d7c4abc395f6c2d7903c66" ~k:"9c609e56c19f74ddc46eb2e2cfe26b1519ff0d1b" ~r:"5409cd62f5539306ae8c936082eef932c6505c39" ~s:"07c0ccb30ec90b1481409cbfa2f5de6cfaf1efc5" ;

case
~msg:"a332b38e642bcad8bd271f776fff24a731724a43400c1614f5e21296db04f725eebad28d62e20ca3f7f18328a76b8092d97b632bb78718f0f2f9ecc7c12cc36b505959917b5c54312ad4717be84fa840b9f06de005c792af3e9ea72b7ae2e3423d07c781c9c2553f899554a0d8dec9a285c1ee25160fa278489474a0e4379516" ~x:"bbb1854e9b0942cb5d1eb71e8cc6fc7e0f4cfcb5" ~y:"41cc1d6d9e0cf5f158dab599114f3ee4738f197cf2c956b6bb0ddd6dfdcf5e4db399aacc16c538948c4b50de85bad6d916dbc415bad2f6737023fc7063c133bd0c4231d6b33ce813c0d6024d1315269571b2554bbb2edf2a99108a4359e8e23bf8a143bfc538ab9f8842cd4e925968f49ac56a02e3f067e26001e5207bcb56d4" ~k:"336e458fc213c0b2775537ae61decc034ccb1d32" ~r:"a16a7308a6824d929b6a9a3bdb280d151a6eed81" ~s:"7a42addab7ddb98000286044d9993d5cf818f2b1" ;

case
~msg:"79b144d50e0047596cf06bfcb3e9ce3959ec4b8cc9ba01434fc3f68f47c868cea048b990e62cd7a50eee288b35ae62aa797924c9dcab76409b869b33de28885e62f17db7a7758973482968b9f960eb2dba84ae85101aa6c6141b3f0839a4185a4c496eae876ecdc45627330d36f01a67cbb7faef834357330aac36c7c6f47ac9" ~x:"754b24ea5c8cb8e88e370074e79cb62605530018" ~y:"74db7460c51919a9e87b430d105d86362ee4acd9682bf6c9fe87d9956c2f5ff17d95930ccc12f7e92d8bcb6af5f7ef1848da8d15c9152082477de99594781b998daafbf8ae4af23772125c19e166421f806bd0fbeac365076ecd9e15432ad4ac2523418f6e410cbfcbc5a71a0edf22e694a67d14b9cfc9722bc4bd8c43e22a91" ~k:"1b50341e94f4498b92cce4d17ab9d4016fb2e074" ~r:"021a3de98c3da698b477b4c3d50b2169e65f5e91" ~s:"afd764318dd0fee04fd6b07f550320789cd9bfa5"
]



let sha256_cases =
  (* [mod = L=1024, N=160, SHA-256] *)
  let domain = params ~p:"cba13e533637c37c0e80d9fcd052c1e41a88ac325c4ebe13b7170088d54eef4881f3d35eae47c210385a8485d2423a64da3ffda63a26f92cf5a304f39260384a9b7759d8ac1adc81d3f8bfc5e6cb10efb4e0f75867f4e848d1a338586dd0648feeb163647ffe7176174370540ee8a8f588da8cc143d939f70b114a7f981b8483"
~q:"95031b8aa71f29d525b773ef8b7c6701ad8a5d99"
~g:"45bcaa443d4cd1602d27aaf84126edc73bd773de6ece15e97e7fef46f13072b7adcaf7b0053cf4706944df8c4568f26c997ee7753000fbe477a37766a4e970ff40008eb900b9de4b5f9ae06e06db6106e78711f3a67feca74dd5bddcdf675ae4014ee9489a42917fbee3bb9f2a24df67512c1c35c97bfbf2308eaacd28368c5c"
  in
  let case = case_of ~domain ~hash:Digestif.sha256 in

[ case ~msg:"812172f09cbae62517804885754125fc6066e9a902f9db2041eeddd7e8da67e4a2e65d0029c45ecacea6002f9540eb1004c883a8f900fd84a98b5c449ac49c56f3a91d8bed3f08f427935fbe437ce46f75cd666a0707265c61a096698dc2f36b28c65ec7b6e475c8b67ddfb444b2ee6a984e9d6d15233e25e44bd8d7924d129d" ~x:"2eac4f4196fedb3e651b3b00040184cfd6da2ab4" ~y:"4cd6178637d0f0de1488515c3b12e203a3c0ca652f2fe30d088dc7278a87affa634a727a721932d671994a958a0f89223c286c3a9b10a96560542e2626b72e0cd28e5133fb57dc238b7fab2de2a49863ecf998751861ae668bf7cad136e6933f57dfdba544e3147ce0e7370fa6e8ff1de690c51b4aeedf0485183889205591e8
" ~k:"85976c5610a74959531040a5512b347eac587e48" ~r:"76683a085d6742eadf95a61af75f881276cfd26a" ~s:"3b9da7f9926eaaad0bebd4845c67fcdb64d12453" ;

case ~msg:"c1b1f1472f08df38a52a55ba55827ba3b7cdd6beded904fcd52610c899eda3c61682656873bbfaab0d907495dacf458ea3450afd93be967a37434d412b6325669ad84b4eaa278a24870ecc2df0da13ad526a9e6669958d4e52dbfba2803ae9ae135d0c0acca86a04c42ba9cafb09b7af96347188880b086169ebdf9f1f5f3173" ~x:"1a220585a989ef2c12bbfa9fc0d258713556fe38" ~y:"99187498534f313dc7cd7f3a48d62b2335bcdc36f0dc98dbf845dc6085c267474c36fdfca38854219830e614bbcab2bb9decb81e86124bd78f86d471bd84be06ac1f0f41fe5b4b3740b2107e0c9c48f81e31e9bf550d96564dd380ca47a11d72f0d0a3275f075f95bbd59869c14dc912a1cbcf01db9fb7f71015cc149986825e" ~k:"8fef50b7121a04a24755b6f3e1cdc93848a9081c" ~r:"54ed4efaecdfc78d026471b65cfefc6529945bbf" ~s:"6d6dac296ebde3f873b751c6b14843f0b7befdff" ;

case ~msg:"b80a47071d1376fe617e59fdc005a890369a4ca5e678ff46eb9b205d6ec09cbd49373bb341fe7813ee442a6ece17e720bf71a74557ac9a375c059e5535e773a45e79e1bff3465a3886c86e2a2bc882f0beceefffb2ae1a522f13c82def4cfd0cfca6fceeb4cece71869e90cd10d0aff27a84b5601daae061cbeb3aa62b37fd3a" ~x:"4247e7e4dc4270fc7680bc05746807c183e0dd98" ~y:"91f50270a754055e5da611c720a4262f3cb8bd4161f77d07401604d3d1165e45518f7e1901adef6628f23dc48271d35ff492af8d62aa538c0e77e042f23a522f2214e62114bfeea46ae8888bdadacdaa0a9a5b503d79c23e4c20c98bd4ebb36f95bf4451ccb0b5bb44dfd011341cfa29a9e156a3cd828e126e68cb911e8f9dc0" ~k:"3aeb3383a3c0f53217c0d7077c3cd66d2ef74a2e" ~r:"1fc2d1cb80bf6e0e78b25fac293b752cbff2b5ac" ~s:"75bcc772f773d5fd98dde1f907e7ec2cba201dfb" ;

case ~msg:"a9603054465887df15db07c0709a8c878d2f1abdcfc6195eabf3e9b3ad07e8558b99cc4a7aa076daf67e9b7d8480f11e8afb18e2ac56a9547b48453fedca32da9eb0c29271eb60f0a1d95c18f42d992394b3264ff3e21e606e0beac08a7ba71b8e5795a8da985118e432cf5b30b6cd3a603d8b0d580f06c626ee937c6cd05f40" ~x:"4d2a5462ebccc5d19bc6c1cabb609c08ad088e08" ~y:"a2c4569a301473ae4f164d68b9a3c6eb705ae81f75d6e5cc3070a559cccb8b1a2d8c21090ed70e61670c7e9dbf5f755a37d58d3abb34c2dfd40db9f26f6868d0dd91be98f395ac0ebdc37e1b5423802bea7a6cb196d7e0f93db92f663b6c9c726e80feb2e9227154ce1c15f8e8df93ec0d37fa47e5fa112bb0a48f4a239d6052" ~k:"36a3cd0101358a4d30c5b7117bc239fb4f6ce2e7" ~r:"48539523815bd8d73ce702367c7712b9b13867f2" ~s:"20ff4cfef8a668829feae73b520e8aa4d02c8168" ;

case ~msg:"19eb088c3229a44f9586f00421cfe7423a486d5f7e28ad2c9119dd2e1395df1acc06cb28e9069cee62f09f48e4ca29269dd89df9fec1ffdf64b1fe2717fe52b1421fcf6c705c0cf39930f90ecb339b51ef95b2ef38a6d96a575f7b36f5edf4f2cbd6d261e1fdd77d4459288c02e68c82a3910ff8ca1747c86bb187d5205f51a8" ~x:"0842ddd5a04161e4579797b5d8eda0002dd847ad" ~y:"49e606aad5d2e54e1bae2517915c660ba30ec4fd28d718613a7c84464b0f44bc6d546e5a9bc1dc60423b45dd01ec295564ec08f29d6887e69f689d6b3488f9da5d5a60f39cdd5a158d51a3d073b2225fea559e58bb222e29a87b5f0f5ab31dd7c0ceaad887070dac955d28973607a99e46ddd7737beab65199f250d7f03b6583" ~k:"712eed73c8d2567809b4d9ec2f59e77d39290b2b" ~r:"6bf4f5d3251201059ee85edb99a67a706f37197d" ~s:"3125c5af397759996b876cb5857be2632aaaf3b6" ;

case ~msg:"addb5a045c9f4f4fb9eb5e5db44d6515980c9e088015b68593d8bcbffc6ff57f18865ab824d3d1586425cb5081197e9e01cb7297b06b64103cea437eeeec9c50798679fb869ec306a72575057fd368aeb0f674a29c3ac248b6a08f91331d8456d062025347c12a0a61c61f76e5206fe6ca437735af430dea7cc8f39f1a5b7505" ~x:"1f1cfc682048375915fb483b77037c81c05ed728" ~y:"221ced57a91325b10f8dcd1220b1af68f8daf397f419a43bbd8fbe644311755b111aae5257c642fafd83b047a1f56f2a829fcdf4df3e5dccb23645b28c0a34c6e8a650efcdfadd48fea467cc943ca4e7378829300713838b6c710962ba72e790c10ab879a01fe1457ea3dd4b7c3c3a542e3522a75d0db261e576cd3f22c998e4" ~k:"703154f6c6e12f163ecad0494897dfcf5657fbe2" ~r:"7cc662e352e0eede85140107a7773ad8663e70bd" ~s:"15c17b9d245872844eaac3d46bb08c3e08597423" ;

case ~msg:"02709d2be0d9dc1dc0ebc55f630d91fa23609f61b513c2275766034d8f40e819aaf9326c8db37c35c5a17e96bc956df6d11b558d16d91871afc010b3119c5798c2e29411ff4f0d7196e7e476bf0ad03bf72e897fed873c10613dd255d15243870b81cd87d0abc16e140d032fe5bd1c8eeb2f66e04d13d49269fc7da6b65a7c1c" ~x:"1d9cf98dc0c1d7bf8dec98962ac6ef6e9406ce76" ~y:"9e93bc03e6e815308734e3b8f1d106961bebdff10a525303257a053dea4da6dcf504c7839b54d57522f2acb3aac959ff4ae8610022ca5a1e688232336ca1ee8fd7028bf7b6e9eedf8a4b0d098969f5e5fd3d9300c340e7f19fd471a451afb92ed4829fa4d90249144aa363dc18807b3e29d27e6ec3da736c33b185511bb3aaa0" ~k:"68ae16534c5f6225fc7ef980f0063de483a76903" ~r:"72b0bcc6defa66fa8bab029676a1c7703f9608f2" ~s:"69d911e05acd7be52f2834c0aa005128e7fa85b8" ;

case ~msg:"cc061edb31c34d3981517f4d89afbe980f74185260cf48b3043bc13a144944ad43e0e576d2a58bf589cc021dc1c1d332c4d76896ea77dda197f683e51eed71b4d6df46666a1b142e679b0283cf339e5bca90e2ff9c34dd5fd7cc4917d66704fee4364f7693101dc766707104efb2b933c4848b93e13f94855f75e4fd756cb6e3" ~x:"78ffb40fd89416388804e56444c9a642cb5e98e8" ~y:"5d7d2e342154983ebc2015bc6750f9876f5689ca0ada8529908ed4fdbc596b972c5cc6d53e80a8ad8a8cedf3ce64b62a75db441c96207fc7477e3f7b9f10df28e0cc2fb77383e7ca4c5150f712dd823c2309f0161be0bd5eedd60cf6ba230861a08b9d9a7468438b4d6ec673d28a54d83c7010d50631e55f0a02832abc5a0a46" ~k:"07c6857486160ef4003470411573399fc4e5f7af" ~r:"21f512425670943477534e9075ceb5b7d63f20df" ~s:"73c6f6f8de3aaea520a083b2264299e81cfc91c5" ;

case ~msg:"79d529e40c2ba4e5b9c7d77d72076f1fd9490928ff4419c824e64db8fb9a051e01e8e173c6f214e0e9e645ed250b6daaa6f8c1a5cc900d52cf3e1efbfea25748e89a1a548c73e2d110b25f5308bcf757b2135216c91dca2783332c0d7903eb21c226dbd33a69eef575aa8a41cbbdcd1b3d94928aa8f8ba58c5ce0d317786e87b" ~x:"784b9db2d19ef0ca8e696884c7711dc2f8ce150a" ~y:"282decc0e37994c2856e61f36b831b61bdc02b7ca675dbc3c2032800b7efd3b711acf14c8869968831e145361bf2182b060e4838f07dc61f76584cf102a913bb28a52c7317af5f9d2322927c9666e5e87c2f2bfd2f181dd32612d7b2b2a645bf1a47c0ebfd79a940f627a668a8f2eb729fd051aa2c659abc918e5571994e6593" ~k:"1bfcf3290fa84652a476655506b145743213e1b4" ~r:"929a4851be0ae4ba91da0e6c7376d71df7592dbb" ~s:"7e6b6504b748ef0024d9d2a2e6f3bcd7cf135ac7" ;

case ~msg:"f5516410706323549b20c52dafa2f2f90799786c0ddb85048892ccc18720dce5c129a10eb4388788a3d97a03b0001799cb65a79c880836bc9f3204ea75a577204dc1e2894c572a258f9e517ca37c5b791e48b27c8dc1c821b34ebb1f29858c4a72a0d5172c565e9dbe1bdddf6e024891cd6291faa81ed565746c61c2eda2011f" ~x:"673a384687ef29ebfa66e331866bd206ca2f7664" ~y:"74ccc6eb83adbcbad0fc37144d9bfb85fdcc85ab92c9f8877c9cda66251d1aff2fb224888dddb7d772a8b738c53e03ecad9903796fa3c9c6024d06367e0870ad797694f598708d08912c0fe09881763a0a722dda95d94eee8824927cbfa6761a79a038aa6d331da34d9bd5c5833c94c526a86af1cdfb2d4079d2db6d0b9a1238" ~k:"52ee4510675f0da529684fa60f6848ce63c4689f" ~r:"27b3f235e4afc18c6613b4fa7f27d7a8262ba4c0" ~s:"8b22634e4d45b71a84eabaa1e5a4bf1e37337a59" ;

case ~msg:"55bd1526e08f6443b255acd32c286807542d34c0f3d79892713f9d6d6d6b3be707e4af6e71f7dab4a2c5f6bd25f5ae1f514b2644a4cdafcece1e58f7576f82e2ab0af2326c71279e9bcef1e1c54a76fa77ec2b2d056717645764e7991b520b0e5a1b049109519b22aa5204e3ed53b1e0957dab5ec32479d06ac3e11a5d1cbd03" ~x:"495009f3a92548be4c9a562ff703187b0ec2cc86" ~y:"5d6edf6db6e6c27e80a7f02597237919170b4936489d6f15f598b820cd917e172509b7e287b88b0cc14e1a0186793886809ab4170209987095092234b4fdc44b3d1fc16eeb2efaf852ed3916698cf9eca4612b4961bb6e20c32e188469883f97f49e29a8197c30d0723babb06dea704f7704b2788e57d76d6d9a3cfa68f6c783" ~k:"423308bb414ef959025bf1a4b27db278f904241d" ~r:"621a290930ac436737a72fb4c62bf5c4b67481af" ~s:"62db20f82a5754f109f7a2ce581d4c8d71c68d29" ;

case ~msg:"62789a89f0d708e21a121fc34009af884133681b9d4a66cc36c0365c34be72a4982eb0961ce257f35e6e7183f0204a96a545193001023d3309a8997e7c4b762ab4f4c40e03e13f4edb328b23cf00c09119deb40addf6567b3b74acef5ceff045304d618421e873c41a72d31e451d213b060829b286f64013d4d9342ae7ab8064" ~x:"2b8dd3965992fcffd158a0816a5987f80908b84c" ~y:"ad590590a82e8929ca86f405516c32913bf5282f70309c6d4a88ccf165ce15fcf11e140c366bb273839a711cb6ae52bb717859570fdbf9fc26726728596e6fc71923deadb35a9d57cdb213c0f29c1375f8b1d3c6b52690c428f7481c14aad82fba7f24eea8cd8da7f0ef7ae781c1a926671a61d4e5ffc8ddf1bc10d688a904c6" ~k:"58ccff88958d5fc48d671ba22ed71f5f82370ac6" ~r:"89dcbca7c8cd6b90aa906a4c547153762fcfffd6" ~s:"23e8926b18cfd4b67c53fac4a2d5321e5c3d880c" ;

case ~msg:"4eafcc6874ae2a6d525738967afb3054357a39670d1e5555d7dc55be24dd5a32a0c7ca3f1b5c6d948c9ce391013abeb47f7e24cd2c54e1fc7c0e92c4ab77f5973a7054bd1c6c845b802b7937d6520508ae018ae14b27ff4b1e340a4b9f6f6b4814d07e90cb8f19b15e915d6ad1834c0f7a3c3e1e45206772a0eec2d3f9160897" ~x:"6b3acee42276bba155156f23dfb7cdf64e4b1ae8" ~y:"b93d79472f049893779a3a0e83b3853f78b3cf69b75961a60e950f0c00f498f3eaa2384325f74ddd38292fbdbdb199212e90b14ec9e554727df81e06eb7783adda38691c63a7cb00cd76d8e18e3d29c793e9f1fe8337f1598b89651f634cb703f218e1906319f82ac6d58e6786da7aecfbca5939f03a13e7b4d5a8ac812d7829" ~k:"1c48f62bd097d7686879d33ee5771558e453bb3d" ~r:"633e9812a0657cec3326aa5415340c46362fcd4b" ~s:"6b201f0c3fd44247f6c28c01d1217eb99146c040" ;

case ~msg:"86d9892b48f5954101482742c0dafb68dc97122483b9e459f97495cc970e056d2162c7c71db167229fb7f45209e0c01eb06ff924b823eda51a7e990f3c986eb9af2a7a073f754cb84db453a9e8c0ae7fa5c05a2655d261ad7ec5612876fa7df09522e0b69ae92477f63def1992c96ce95ee7bd630ec1614621da6a512ab53dd7" ~x:"836d90fd90d21b84bb012da7b2168ea8f05202cf" ~y:"ae264ea96bf093ef2de27381738219e3bfdb08616967cd13e9415f475c4a794c19f12a607b898db1e3e6bc5402327585d32841ae15e3462880850e9e4136a4751b64a729ea27b72ce36128a44fa53752a08d73584faa44fb14120f47a04c47e989eadabc7e5cdb15d27c2b0ea4257cec229a2c7bf7c93c571e8d22aeaa2e38be" ~k:"7956ea15111ff392d6a9359067bfd8c21f0bfc0b" ~r:"77b480885c70c1fee2056237d1b79cfd9fb54a1f" ~s:"2283f4c0640ff6daacbdfbbef7224afa59ca3959" ;

case ~msg:"8b60b9b6ba375448de4f00de51d18706ef8c4f97ba34c9cce2b0abb0698436009d1d2bafcbef73a8b5dff6a3cd5db5258ac84ef724b28d8a62d715da6e111939735366a7c66470364557f546377d5c0e7ea9064731cb7149e1051d66a7bed14aa205bdc5d4b9ca029a1e68a6fa2c1db22d27fb79d83877cfaa6742119229a493" ~x:"5a4ae9f8fc82c9198d9400c51f282493b194a07b" ~y:"87032f263de2bf2f268a093f33c366d6bcda772ca959fa17cfe948f1dca3e75ec94276de91d9bc60fc6ab9224861c55dc9ccc5f715c251dd508bd438681cab205059050f8e11e8a5468da42d20aefac53d7a9fb71f6424d7bdc65db873ee4f9dcd918091aa724b261b6056f320ca7724518e14cb8dba0b713f54a0fe44ff1597" ~k:"662351e9b8c3a607afdf3ee599b46681e27b83c0" ~r:"5d159f894d250db90d7fccd49329e44d1112db47" ~s:"37231bc15195ecb6badb7c3fe80380ff912baeda" ]



let sha384_cases =
  (* [mod = L=1024, N=160, SHA-384] *)
  let domain = params ~p:"f24a4afc72c7e373a3c30962332fe5405c45930963909418c30792aaf135ddea561e94f24726716b75a18828982e4ce44c1fddcb746487b6b77a9a5a17f868ab50cd621b5bc9da470880b287d7398190a42a5ee22ed8d1ff147e2019810c8298ed68e1ca69d41d555f249e649fb1725ddb075c17b37beff467fdd1609243373f"
~q:"da065a078ddb56ee5d2ad06cafab20820d2c4755"
~g:"47b5591b79043e4e03ca78a0e277c9a21e2a6b543bf4f044104cd9ac93eff8e101bb6031efc8c596d5d2f92e3a3d0f1f74702dd54f77d3cd46c04dee7a5de9f00ad317691fddcefe4a220a2651acae7fcedda92bfcca855db6705e8d864f8192bf6bf860c00f08ad6493ecc1872e0028d5c86d44505db57422515c3825a6f78a"
  in
  let case = case_of ~domain ~hash:Digestif.sha384 in

[

case ~msg:"b0dbbf4a421ba5c5b0e52f09629801c113258c252f29898c3354706e39ec5824be523d0e2f8cfe022cd61165301274d5d621a59755f50404d8b802371ce616defa962e3636ae934ec34e4bcf77a16c7eff8cf4cc08a0f4849d6ad4307e9f8df83f24ad16ab46d1a61d2d7d4e21681eb2ae281a1a5f9bca8573a3f5281d308a5a
" ~x:"649820168eb594f59cd9b28b9aefe8cc106a6c4f
" ~y:"43a27b740f422cb2dc3eaa232315883a2f6a22927f997d024f5a638b507b17d3b1cbd3ec691cc674470960a0146efdecb95bb5fe249749e3c806cd5cc3e7f7bab845dadbe1f50b3366fb827a942ce6246dda7bd2c13e1b4a926c0c82c884639552d9d46036f9a4bc2a9e51c2d76e3074d1f53a63224c4279e0fa460474d4ffde
" ~k:"33c7ba88ff69707971b25ac344ae4a566e195f99
" ~r:"77c4d99f62b3ad7dd1fe6498db45a5da73ce7bde
" ~s:"23871a002ae503fdabaa6a84dcc8f38769737f01

" ; case ~msg:"ec84bed09ecb4a6feeec3a7071b65a4c1267a03cac8b5a0500c237b20dc058514da798335a21b23d7e8cbb15efcf92e6060a13fb77f4998147dec1d0fa0edd418b0aae8eb0056fc7d4008b198bd40b969dc10d79e15b2300820323bd5e1b7d894ce8e7bc8f7ceca129b5e511ee1c8caec25514f537353a912a971b8070e3f141
" ~x:"952b61ea90df3f788eab61d95be16ca28001800c
" ~y:"d7a0950d0e6362b0c942ad8af67161df07debca59a4cfa728f93d49b6e296a23969a65a92b2e05398a114d73d5a52b73b71ebb28571cf6b6002f853a8f594b5c93b9a84233f3c552823619e0aa847d60203db15d2a916ad02228325e15783988f4159e05c8ca088360e6ea7ace51b055102153c00adf335ff6affd1754f2a8aa
" ~k:"7982f6b8956c2bc0a2c2d02ecdb9e47d23a7ba81
" ~r:"b2570e0e19935438d32686c478473a0e45dad023
" ~s:"39a02e9803624f7e90feab8714cddc41e01f8fce

" ; case ~msg:"80f757fc06409b70d733efdb68b520f3f9078ab936c4479fb98d0beb1631d8303324470824862224b439bc85decfccb8de8fbf36a2bc4ce3a092688249ab4eb9febfad268245fbd7e72e0f240500af71292ea23c8ad4b71e032106f587f4611663137690cb25241912763c5e1879b3ab67e2187f92d821fc81f552e2c355bd73
" ~x:"2a5f637f4b886a3d37c4369cab04d78c79f1a3a8
" ~y:"1f03013e66fd1e633ff743894c37f6964839a52cfbb6e849cfb4eac9a3c9cdb55c28e14788865c212be62047cb39c6365780bb2e627957d34e99232f69170a8efb894d8029d1b8bea8b911cebcd43b86bd536693f18bfe50c84b99911181ace14c3fab9fb6acd98786f9d2ad129c5efeb8cd0941a3d89098d5721d435343cb76
" ~k:"b7c75c380bce0fffd59dfc3993e1d0724da877b6
" ~r:"c7db4a9f54d882ec5f561705396c94834dd53c5a
" ~s:"6752cb6be9b87265d76d69b382299678f96a5faf

" ; case ~msg:"36a25659a7f1de66b4721b48855cdebe98fe6113241b7beddc2691493ed0add0b6a9fbbf9fb870a1bc68a901b932f47ded532f93493b1c081408165807b38efce7acc7dbc216bef74ed59e20973326553cc83779f742e3f469a7278eeb1537dd71cd8f15114d84693c2e6bbf62814a08e82ba71539f4cb4bf08c869d7db9dea9
" ~x:"bb318987a043158b97fdbbc2707471a38316ce58
" ~y:"c9003995b014afad66de25fc0a2210b1f1b22d275da51a27faacda042fd7645686ec8b1b62d58d8af2e1063ab8e146d11e3a07710bc4521228f35f5173443bbfd089f642cd16641c57199c9ab6e0d9b0c01931c2d162f5e20dbe7365c93adc62fd5a461bea5956d7c11ac67647bedcead5bb311224a496aa155992aee74e45ad
" ~k:"2ff654b680e722ce65a560e785e8ce0b4773c86d
" ~r:"17cc53b5b9558cc41df946055b8d7e1971be86d7
" ~s:"003c21503971c03b5ef4edc804d2f7d33f9ea9cc

" ; case ~msg:"65a3c92453f961de7f576d5a1e3106c38b7f20813994b5dd201546dc455065dde59edcd84d0fa17a85c0f9f99171d67a34475cef4f311951f2eef7f6b64a5bbc6da6d1b622480cde56a07a77aa6040ebc1fcb265b3b624881fd27203dcfe8a12492198474a990cb9f34a1943356fde5bce3fd83516da8bf780f8cb1851b3b954
" ~x:"59d92aba23f50ad08b1d7c2ad560ded36b94ebc8
" ~y:"0fc514ca160f34f2f6ede1ba5914d5844c9de514208c72569a0b36ec92c8b2c8fdfb7d68127486e58a04a32d0d150e51bb05e66624cb622edae19a6b4b1d8317689baafa30168ef3759ee82e614e4761900182df90e9cd2d931153771b8be30d89c2fbb95be7e05a4b29da968ffebbda5c0c9839354bb59dc697a269063f2f50
" ~k:"3d0ea569b4dc69342955f5b240af66d228791e50
" ~r:"77ffaf4290c41eb089c1d7be5c8d3833027702ef
" ~s:"cb753a2d4ce0e59851f814779f343beb615f2770

" ; case ~msg:"1526b64ce41cc8e2cef26f3706be530a36ac9cd16ff69f05773e9447ed9452064b7751f3a64919bfa3a7e1020dfc175a10acfdf096fd41c03372e4d2abd7ba887e0076716ce9552f2c7c8eddb1b3fca1bdcd23300ce2b1677d4a2debeaa7053466e59b098771bfb9218e0fb4ab6b7418abebcc34d681e14c4a8975000d83bb44
" ~x:"716290d0ff2ad2329be2cccf825f2075be659743
" ~y:"d30eed739f46479364d4c2bec18cf4c75c324f8db8184d9c3c175556a00acfb0a6813887b68706e70c167f4063bc0046396ba1bb3226c29221bd64ec4cebc990a7b404e26e2cf042304a7c57ab7de418ba671e17f7f502b9e1bb5984469b304ebc0c3c3a5a69cff7abff4110130316651e0f93ebd2834dd044eae1fd6f045102
" ~k:"919d698fe37c027e3e40cdf6e77f81e96d8bfffb
" ~r:"31abe8e7458ce363a5f3985111b239bc8df8dcb9
" ~s:"1d967be0116128699d167fc16e5e920a41311669

" ; case ~msg:"d7852ee90b3f1120bb11249808c7e7be14fe577bff1886be3c42589a6eeb06a1834110862b65d26cc5a2e5d903ed24328d684c96e3babb37ae31f96d32f57657a3bd7798aafae86f44ad8981e7cd47d7f31bb4564a757c925c64da9820963c1c5148f589d6393004a6a58aa2c8a578f4db7595f886170e79e9d57bf7ff8fd0a7
" ~x:"0531cb42f45bb813f401bd239044df2d3d1968b2
" ~y:"0dd37985163f93618fdea8e3975419fcf7446ff980851e18932d7494f809c0ae9c03cc39779ff0422cb2248ae1986f9aad2a43d6fa6878d244b429aac5ea80157980577e5ba0d11b1fa340a283fa0a2d651e024331e6bbe7d01ac034db37b008b91f9f88d135fad23af8c22765d833a9c9eff7accf668e17f9a8bdf59317c202
" ~k:"739dbd1f84b6be2efdb921a0dfb76dbc6136915c
" ~r:"44c2d6509874ace71acd1dcc32335b394c4e41e0
" ~s:"37e78f13aec052eb7b07a8b9f6d54dbc77829006

" ; case ~msg:"9ab91448a0dc9694be173ce6d9b522ce0e2f75fcb57720fc5eb8f92d8fb0e195030063968925a568636f4aea1edf6c5fcb86dcedd204539d8c291757fb8a51620abda59aa8f8502e6904bce0667d92c8cb3fcf1a61b1fb0bb4e9383b37eb469bd5c2f5a77680da62f907c2e263cb48402b4b12985eaab90451885e819b3e8c3a
" ~x:"6d764cf62a268b0070bf80308622bb31941d4763
" ~y:"49d7f08fde0a83cfb8116c9b7cdcab29751fca5ffe310760fea713c30e95e7755e65ce60928893c65020ee9b61f6c9c89c07e0fc503b7b031368f069578a9e6b451fef369ef90c26dd660ee1a6b8b714d1cc28245e9f13f187122de26ac2fbf5bccb7caff59f1de910551104d3a0e8fa9fe6b7eacc9a5fd556b7bf7139d6edf9
" ~k:"c66ea7177cd6edf6b9079fbcf6737d3890469b19
" ~r:"95da25d06ff9c02bc893fb032508304c17ebcf08
" ~s:"617adb8de10da1a87413d64466b482409d27bce7

" ; case ~msg:"c9c0e69f840cb6deb984c2575d7f6816fa35af03b4429c703a5aec90e7cb26e52413587f3bc5a0772be7b5e589c9a76071c1739833f4611fa951d375820b48d740626c665534d60487bf3e0a84eb6389e099fe621f269491c3b8942e03bbad2a5220caf51e7b4a2650e4b300024a0a96f0861b3206fffca83d0850f2a3e2a06c
" ~x:"38a62d234e1aea0e847621e79dd17ee9d08bd9c6
" ~y:"26f73219d0e7dd3a80e7fbc079d9baad4512891aadfd2416b1859f41adac31171ec624d8a4d6a10d5de1b93959bc49953f23492f18ab765f963a98584807d66629e5a1e057d77d42e3363458641a0469166a0d853b27798bd848aa0d3ccdbb40fa21b9fe62824cb2c7cc62425978e672aff0bbd8c8cd08e46385b0d6219dc56e
" ~k:"5f5755dce464174adfe00affb55a71222d83da85
" ~r:"b6b25a9da110b5d57675889eae75ab58a4d8e281
" ~s:"5a60c2b0adbea4c5be065bbd0fd0e3ce4bf29200

" ; case ~msg:"4002de825bb87ac346bd8487cf6be053cb30ee67c66434217107a8b0b52e5726900615edd2fd0acdf88a7e65e7dd3ba6abbbb371a1c840250d9ce809e7b1111f16daf5194211715ff5fe631e378408749848a0c81a289b4338bccd8d1053f863197ad02920fcbca514e2dfd94a8b00f90cf034adfd776f4dcaef2c8dce3b0539
" ~x:"c8f0d697bbcbcc0e31864f8319984125f52ff5aa
" ~y:"149bcbb4f5983db56fbe998fcd02d736e6d2f18fcf96468cd7e99bc647436fbd74fd7a2cc2f0d8866952b97b44ff644b5665cd1065b07a2c33d9151deb335e3522c1b77da1443a1373c93bfa040da5a1353b88a78e3a5a084e6c442db03f7fbb4bdbd30b1af3963f8c5d3e83453294e3a07ddacfd43dc8f9e83032fef78420c4
" ~k:"b4281920a775fbeefb89615b236217fd1046f2cf
" ~r:"d3cde170d82154ec1bbd9077c486971120600376
" ~s:"b008fcd01b5e49a85a921bee1ddd706212799086

" ; case ~msg:"f7018ff0af6776ed4234c1fb9cca1f8cff31295cb9f76d8b73898430097c49a40028441771ea1de08ffd5cec7eaa59e32b3a170329139227ba86e0c5efcaee382bfff962249da853dee418413f201a28fe45b8293c262089d2ceeb9af67529ab011f04f5eeaf82ba32dce9a9821762c3351b00206591a3f87c5260a4263659f0
" ~x:"9dbd262da7a529f80aa667b27a29d6a52671fb89
" ~y:"6c206e71fed8b363fcf571786ce1b4e52a404de7eda7031e5d93a47ea668de43dc7073e31d3b6b125ae3e8ee45aed273bc878c73423b225a1526bbb149a0ce5e9a2d2962bd6d332375860f84ce0e787a0af93f44e64edaa2dce6ca22bcc6d48b84b0affba342753b1842941067d5b8414c356138e625bb506566a27b335094b0
" ~k:"0f0e02596ce6674684e7ec448d2938de12842fe2
" ~r:"079b08bc016c543d09d6b276c023347a3aace9ae
" ~s:"164c3c380f209feaf8ffcf53691ee3031c3b3fff

" ; case ~msg:"4a18bdcccd46bb89567ceb9c1e2e500a3baed24ff2c5fc7f83cb3cf6a6f38859a1a927fab5e2fd7ea1e1a4154739301cb1957709103af886c929cf88d25ced5cd6f8cf3ffee7b088edc2f6abd1114398a3ab00fc21bec02e8e539ba12df70a587fbfba63195c6449b2b849547c42277834e1ec086b5e53d949846769e89715bf
" ~x:"5b3e9cc0e0be3d714d1bb2d95e5146d27a58f2ee
" ~y:"8e668dd1527b5d1e56aae4a6ca225e677368412324a79d98bfdad9a84d9f87c1357518c9a5056ea6a0882e94d4ffadd97d89bcf2f32ff442b25dd2af2a78ddade46b75aa8a5b1a1471764ab699d700cb2a28b959a3848edbbd6c9514ee849f833c43008531365a01541f9c0b65d5e7d3c21dc8bef1369a41c0405f3723f67910
" ~k:"3d9b8166860e18a9306026ba669a3620c2954ad4
" ~r:"b22c00fe0bc2fae7a4ab74edcd496c64a999c7d3
" ~s:"85ba8dbbc93ab94a76133d479e3f79576944e6ca

" ; case ~msg:"75474711821766b065e2448601e82b88153a41bfb5c6b6a9ddcf73170ee374a6625de19c560bcbd2020bfeab5cbfad8fc60ccfc95a1b94fbefdf815d9bfc43fa59315e7093d5685274b8afc3139b925ebf697fe2699b0feb1e42bca65e5d4eb0b4514af92dfab85e7f2666c87e9789395f354ce33938e9623061113465a4e2b9
" ~x:"9b4a2536a108892240fc40c8c69a4b9b903ac760
" ~y:"74a93c73d75500ca4305ca3184475b53d96c6fdd417ef23d9dc61b80bbc1108228d2543c1c3a9f2e7783ca69b019c0cd9a6d2b62b0ed93d4229da87bfc21f9e4bd0dea2c4e6d4d2f88201ab0504b31f4ef1558adf493e470adfc572ca68debc46123589ae913b967983dbcac6bf3bd8611137e39d5870057ae18cb84a76aae30
" ~k:"a993b059a49855e359014151700d02e8292ae708
" ~r:"1ed131c96a2c310e1f7976d3082a69a5af45bdd0
" ~s:"70663e9ad7113ae57d4af6907712e0aaf88bc07a

" ; case ~msg:"340df708d457df9413ef2bda225c5f558b90966cdd531a0b5aa745d5c3ea790debea224861ef12fb1638bff0121ff26dbdcffc299bf9f3a9c1fe6027400ff14c34fb06f67db9c30a1dcbfd996903523d85046382ff280418d974a3ece6b5fafe305e2e79b1d07a7c1eeb7a1277a82282be62831df7fee38841462602986a8e9b
" ~x:"067e6e55be4744723b6f056b76629e93c297a585
" ~y:"6150a68d64adda3d3fb5a973c62b992ad3fe538af7161bae41ea2f1799304fe5b8c864e061d133d94c16a4c6b0ed8dffdf2cefa7394015e75c57b181419dcfefe3409d3b53d86911c749f9f28f7c1de99f7e4b2ea22a48817ace4fd9974fa53b8d4f05f5731488813803d7f3aaf1cfa138bc73c4d27ca1621e9226661883e9dd
" ~k:"77857e6de8f37eeb6925a87c027a3cd88b9d3584
" ~r:"4f182ad42cb5671d3162bb9d04a06cd20edbc558
" ~s:"a6c5417947447718ed1cb89a6efce2d3116e50d1

" ; case ~msg:"9f23c82563ab7c0ba86bbb989335000a493b291e5dc17ce729494958903623ed99df344230ffb626b1dbefcce059ae16c2ee7ee6fd2a7807336cb71b8853e2ed3b74b2faac82a831d53e03d7bbb96d38df98fd19bd4c1a6248cd507c89f7995f59579afe5319731b443d6871e558f5b77f2f9a4dd99efb305e27916594524e02
" ~x:"b457e1756ee9056fda7207616cf7c04a33afa66a
" ~y:"96d7451181fb253fbc3f441409be5e5e0144972610e37fa82bc2af246637a4c918023097875255a217ea895daddf46bfbb174749b04c59fefa6289684f2f9aeadf5ce7ca47f0032e384b7d50597901181501cb5915fb4686a6ad7bcd5b46862411a4df22b1ed2a56905e07c0a936c9944213194ebefd4ec68597cca036338b3c
" ~k:"b29f28659dffea28449435b5a044487e29d82d6a
" ~r:"b6599fbddb4856276df448cf09d62fd7657de6c3
" ~s:"4b49589099be5578322d829b87b43ac07f62e35d" ]

let sha512_cases =
  (* [mod = L=1024, N=160, SHA-512] *)
  let domain = params
~p:"88d968e9602ecbda6d86f7c970a3ffbeb1da962f28c0afb9270ef05bc330ca98c3adf83c072feb05fb2e293b5065bbb0cbcc930c24d8d07869deaecd92a2604c0f5dd35c5b431fda6a222c52c3562bf7571c710209be8b3b858818788725fe8112b7d6bc82e0ff1cbbf5d6fe94690af2b510e41ad8207dc2c02fb9fa5cefaab5"
~q:"a665689b9e5b9ce82fd1676006cf4cf67ecc56b7"
~g:"267e282857417752113fba3fca7155b5ce89e7c8a33c1a29122e2b720965fc04245267ff87fc67a5730fe5b308013aa3266990fbb398185a87e055b443a868ce0ce13ae6aee330b9d25d3bbb362665c5881daf0c5aa75e9d4a82e8f04c91a9ad294822e33978ab0c13fadc45831f9d37da4efa0fc2c5eb01371fa85b7ddb1f82"
  in
  let case = case_of ~domain ~hash:Digestif.sha512 in

[
case ~msg:"3a84a5314e90fd33bb7cd6ca68720c69058da1da1b359046ae8922cac8afc5e025771635fb4735491521a728441b5cb087d60776ee0ecc2174a41985a82cf46d8f8d8b274a0cc439b00971077c745f8cf701cf56bf9914cc57209b555dc87ca8c13da063270c60fc2c988e692b75a7f2a669903b93d2e14e8efb6fb9f8694a78
" ~x:"07ce8862e64b7f6c7482046dbfc93907123e5214
" ~y:"60f5341e48ca7a3bc5decee61211dd2727cd8e2fc7635f3aabea262366e458f5c51c311afda916cb0dcdc5d5a5729f573a532b594743199bcfa7454903e74b33ddfe65896306cec20ebd8427682fa501ee06bc4c5d1425cbe31828ba008b19c9da68136cf71840b205919e783a628a5a57cf91cf569b2854ffef7a096eda96c9
" ~k:"2f170907ac69726b14f22056dcb37b4df85f7424
" ~r:"a53f1f8f20b8d3d4720f14a8bab5226b079d9953
" ~s:"11f53f6a4e56b51f60e20d4957ae89e162aea616

" ; case ~msg:"6f39973fd225167a7673cd71ab3534d2686687c332f93fd66db5f1ca99678efd2825a84cd7a7107adf96501dd1d05e7bbc8d113e087bba77b2346b4364132125245e9aace3a146b576f654c86e07fc1914cafa209dd6d04845575dbb279cd1b23296d01ef505b5e1ce7f2194f18988f355c9b34f920ab35152e03bcf792ac529
" ~x:"8d75294b56262e42a82db41a4039615396574dbf
" ~y:"110e398e36c9d2726e77deb465dd23303f9e387778b549700a52b9a5468512ee377ce3d7dcbfc6b64ee353eac6a43898b26484058ba1b24b229cd69c994d976d43344c181ea6c47df0062c09a16b23ab6075c04a0899bac3e4f034a983bf90438f6ac26855d8a5fded90172e7e8c196a2ce7e1fc0dac94278aff1653c3ae09f5
" ~k:"66a1322607ab98aaa57c12a5cc3f59dce8d7d0cc
" ~r:"1b9ed39bcc4b46ed0007679ce9c3f6dc7c4157b9
" ~s:"258d4136ad95b704a7959d04096dcd781eb54bde

" ; case ~msg:"7f59744c790c0f985a9ae101d9fa00da3b95d2473d792805ec1d6d1e95222a6f30ee6ab8fc5a632057153f237ad3aa2fae8f1e51eae75906d07e576dd0021ac1711b1c8853e62d27fe6b098766b8ce3e76d347c8e49be0ab05d0d12fd777a85cffc7ad1207a9aa75643d7b415ba4b1b97dc0ee19d05a607ba063a0341f176104
" ~x:"25b7fedcba71eda85fe189bf0d0c43214ab6388a
" ~y:"3ead9cf211f3859d5baa5155fb62331bca3fff9ecbe182ebf8b04db0ebb19eda548c86db4cbb5eca98ce449cfd51f1c460d7848326eee22fcac7247fb889ee415c4933a909c78ce9bc50ee190116da9ae2547ae6242a340ddbb9a15ac818c4677f2919c64509d03c49d1307bb2cd78e01ce5b25a9f47d828fc7584ebce366c2f
" ~k:"8fd754defb1274bb7ddea0fc13fdc76722442d86
" ~r:"38f52df78b0e454d3583208a0fce03b904eec816
" ~s:"5cdc57a943ab1f269ca11c63bcb1059ee76f9c2e

" ; case ~msg:"16250c74ccb40443625a37c4b7e2b3615255768241f254a506fa819efbb8698ade38fc75946b3af09055578f28a181827dda311bd4038fd47f6d86cceb1bbbef2df20bf595a0ad77afd39c84877434ade3812f05ec541e0403abadc778d116fd077c95c6ec0f47241f4db813f31986b7504c1cd9ddb496ac6ed22b45e7df72cc
" ~x:"3fee04cc08624f3a7f34c538d87692209dd74797
" ~y:"6e8c85150c5c9ca6dcb04806671db1b672fc1087c995311d7087ad12ab18f2c14b612cea13bf79518d2b570b8b696b3e4efcd0fda522a253bbcb7dbb711d984c598fa201c21a8a9e2774bc15020920cd8c27c2875c779b08ef95093caac2c9cea37ec498c23dd24b684abcb467ec952a202cbd2df7960c1ef929cc2b611ca6c8
" ~k:"934552738360670c98b9c5384b639c46cdecfa83
" ~r:"00018f0fdc16d914971c8f310f1af7796c6f662a
" ~s:"62b7aecc75cbc6db00dd0c24339f7bdb5ae966a5

" ; case ~msg:"a2ce90b51a480c0668b55936bbeebe3679f8d406a0b694b90345749e3b9c67776cae9a62c25cc011cdb3180263ddd73aa2090ec7a749092f6c7816c26744c5393acb08c6b7b359bb3a3c7543684f8050ecc6422234ff24978ae06b91d6a24c086d71eb1761caf14176d8bacdcad53b7895bdb0e053c616b147ff73d2d86ba3bc
" ~x:"2d667bebf445cd3ee45d5815e07ca5735b858ada
" ~y:"0e6b419da8b4db802d938873e3b105ab3eff432d8a1376602059cf2e510f696a2a4e42025670db0011e9be31e8b1403615b9a339ce654a89a2d462ee20c080c4479648c5c00e172ecd537c934e7534af7002bd6fdafab56506680c019ced38779d954091645fedf5d0057a23ff634919fc56a96771ce21fa99ecd9aa7f7985f1
" ~k:"4aeb4911d38f1f634ddf5fe6c970d943ea51b266
" ~r:"5b13f1337ac72e419867c92f9387f9df62883aa5
" ~s:"90ab5b68fd8253b6bb64c61759164a97834c39e1

" ; case ~msg:"3b6eeaedc5fb38ce8691686c89993caf17c9e24fa565a9e8d48436b87db62fab839c42d81fb1f8b8968c826e78d333b1d99d5c36e08a9a0ec7554c2bde07fd8ec422af128246ba3beae18ef2be755db22a869202951cd95796fc2ff7ba2a6967d19e5ca2304655bfdf879b7747f80a59b1dac0461cf6e490378e56ab378584f2
" ~x:"71dbbac59768e1e3093f0c60404731a2ead482c3
" ~y:"4a7ff667f7ab2891a8a69ab5d15d93d1fd833906c9b629fcb9b49e84d8ecb35b397d93839d7985590326cffb55a70e4a51a2829e387290f6fafb7d226151c282470224fd717f3d526589c6eed9611c5bdf4bde63fcc9204c8007b0b143c49d1981835658bcf800a2157c5c143d76369fd6e78d0a3f800b8a3a8f0e11c9059dcd
" ~k:"8f78910d1e8a9daad9523626ee7ab1d0a5b4d977
" ~r:"61380ca86798fc5fb61c35675af08d5cc13c25aa
" ~s:"54ddf68f476884af3e0e6536f3a80925ee63a402

" ; case ~msg:"01197ae960de90a93d9736896fe136bc561f0550c6b1cc3631b31df683017c2ab8c6f41d2745f1a797e0e89dc3d5878866c3694a080366757e6fd892d26668fd2d860ea2a2b67fdaca96e32297758787ecc0a7e1d304cc719803272e72e339b3f34c347e47b91a1ed69ca8062cd350dccc9c2264732b9fdd8462d9f6fc76850c
" ~x:"45963a0771456d6ae897edf7579091f5f8c76747
" ~y:"373081770a9f4dae8df5dfa70503e149d759ca37408549aa34d1b49b3176a39d7c4661e421a1f5d61e3f77b3c4b39bb2e63cd24919a2910a8b155e1758f5a375da60f19d2bf4020e828f4237eb3e2a36124a6a3914469d6833695b83b9377fb285b27bd2689933c189c7fde42e1e0e20308331fd56ed0db2efbc77ea3ac7121f
" ~k:"1f68d020331b81fc1aea51907e94f7d62ace9135
" ~r:"41ed170c8bf6f20fd1ce18faac97565fdb4fe6f4
" ~s:"7c8c6feace68c97ca43780741fae58f2f61bf765

" ; case ~msg:"0d5ab27b2b7e18cfce4ccda13aa1a5a8c18baaf39b14e642b8f81b30cd5418a1dd05df22599fbbb3bae4fee1e4b2c150a23e216c133fe2d8235485e34f80685c66bc0c190af67a0a49930b476b2803e12274cd43090921bf668fdfef155072a3cdf17901427afa51318afdda937e283e2c60d85e3bfe07f3da5f992c1fca4b98
" ~x:"95bc588bb848751ba57d7a9ab340cb00e79e06d8
" ~y:"1ca36e3505ee70a56afd5dc40a48e97979e984dd2d896abc7a491d3461c6931668a0cef11e45bb66c611137999907ad7e1f7cfea7f7ed49aae935bfc41443293e71dd2fec29f37a9544672ab9250caa28188f390b5d4af13bb05e9692c1c6a4d6aafebddaf7eef1834fffe0f5391bce243789a2d55d29e2b90ce120429f2a075
" ~k:"07a4f8000f0ecddb72302cf4d7975c7efc41c143
" ~r:"66015e5fb3abe9d78523770f7ba0990031065ad7
" ~s:"4b8b153d5b01ddfa91f2dec6f0faff02e6e87218

" ; case ~msg:"906a933bc823a307e2ab29a4a8f7f1510d5d303504fde38169ded168913e3bf81d53a4389a3e73a3efebd5e42cf402bf9fdc5da5ef46878165ada6d2e07299035a3987ed6c2c6c8becc44ea131a9493e72aee28242cf7cfac38ee870e54eb95a6efa9fad74354b281cb63ea71652eba1ad73f841dba7778f3a03d3e00190ed68
" ~x:"8295ed7e125a65ea1762aaaada34602a7bc76845
" ~y:"4f3ade3ea4f7066107321e8bfb12eeaf9b3c7bdcc6147908754231156b46e0639c9db9d5447abd2d0a9223c85d63d8b1dfa69742ebf7e0419e608c4b18c3ad9f55f5d2848edbec4e7180e34bfbb1f6b6ebbb68649714b5fbfa6cfab4a01f655008a95a7eedcdc2f7171094563a3c1831e81f5ca075c6e853beeae87a67ff9022
" ~k:"8148e40362a255d240a3e6af746a8880162d78ad
" ~r:"99c91e0794723bcde34594dd2268418dfb353443
" ~s:"42e9c49d60ad8f9b41f290ae6b772f44be62cea9

" ; case ~msg:"1d6ba43a0ff677cf8cf68d6a1d3304d99490a7cae56fe35318f38ed0f5879fe254703fa77458c45e8a698469b899a215c25e869fd28741101d27dc111ffad6980f8ebd748f6977d5d60438e6edec37a49d3011f8f0f08525156ae60bc91abe661638f4b9c6c365c3af1713bf7f7225d4afad7a1b531a331133d8b8fd238598a4
" ~x:"8d3c302da7b77ece9ce6e280e603bd260d2dc144
" ~y:"08ad77f233334c04ca2282f6dac0b0d8a00d596e02e836a767a146ef80624b33fdba8b35204b20bee8ff2be9a82bd80131c0aa898b17eeab5af24c20551d5d636a11548fdd2e6c616b09df86b057e5702146ecc4fa2d44d85bb1427e7e3576f698b4f21645a1e00479d08982b0573dd1981bbd405c2a45d7de9242afae8f95c9
" ~k:"79989e8eb43520091706039415794d2306329861
" ~r:"a2b42cca55bc1ba33f8252d1a89c8d89b00b3950
" ~s:"2ec5166e35e63f0fa116b3db1bd18681a4399c04

" ; case ~msg:"3bd0c5b759cb710c52b81fba48b6771cab17bf1b67eafd08f4ee1777dd473064dd0bec98d3582ee1e991ab9a91a6fe558a41db9ae6b21a057932811440d64c786b22d150e3d38c71900ad5b61e0530744e765b5c2ef30bcb96e726e3079e440086ef300bae9000df3403c33a79849f8f83d6c03f77eae98052578d82d628e65c
" ~x:"4d3e42ef42a60630edcc842f25a1b33c8851c742
" ~y:"3a1ed976b7934bee3e80d69fbcdd35f82051ccc214bde6fa756be67017ff60ac6847cf8d1f823f890d26af8cd351716ad2d4eefd7f06c1951ea4a7db5caf250f407b78f21fff425d0cba1b5fb35a5b5dcf062a1cdf2507af74789326710e334faf3c501bd8c8347225f94f8973adb7a8b5def9896109d1efe550325dd89f31d6
" ~k:"6c59f3cec7e34db174dcbd6bfe224d52226c56cd
" ~r:"77d62ec2a95beba6c672d8422ee663d1d18049d0
" ~s:"2a339cc8f567c12149a8917375ec6ca4b47254a1

" ; case ~msg:"8dc582a2b5af65e66ebdf5b533d8e22b38b5c1977e578d3213a110e9d4837a74d4b7bf7d71875690b5b71c8e8afab89434c46a8641cced1a130e21cd0c805ee45c134c7c0df75d5cd30c41818f7ae475dd6022877c743d09d54f0c94581ae7bd4b423f02e19397be7bd4a904b88cbd2f814b1dff1e796d9f2d1c8470b796c69a
" ~x:"6a6a9874f0f89f04cbeaebde3833ae179ade3f5a
" ~y:"5d6dc1749f28cb8f7c014d5c5516cf5bc222c6d9337ac0089b19b90b321956cf6192f3255d0eec45840810c21fe91cf5308948852a57cd0189f15bd96af8380d19cb821b1c56afdc38a94b2c32feb18213939693b69f2bcbae7e70ab09ead3b6a8b7dad3c4f521ad0455dd4e872b3627d4fed20d5efc78f6ae467fb9267ab1d4
" ~k:"27b2661922214411aec66e58cb36142ab3e5a256
" ~r:"05363bcca193d726cd20e03489e1b13b7df3bc98
" ~s:"31bdaccb29e4a60023929f182199c070b71ac575

" ; case ~msg:"477af8c025181b557732b9568634b1324e6669b4c28a0bcd4c653d4c81ed68b2a2043a800a314ba95e50deeacc5ee9c2ba6f6f62fdba0e86aca227d727377552a3abdbab601c2601846ec27a192a3f33e7ffdbe4a4aa7beb2b3ff6c91bd5cd5c890bcb6f4c908ff5b9b555e2a0a7df8c3ef6770136bbf009755bf6c3e6307310
" ~x:"17949e838d7c93e1d837be65b3c4482433a1b208
" ~y:"2cceddc9e2cebbc1e99b83b03053bb14a9cfaf072b45e4746d18cb3901f6a2c3cf72da66b0b9b3e105bd4cd0e5427d7e9b657ed71884cd49f51fe8fa18a366018a3eafac3381e07a5b19f6d3862ed2916094906e75286eaf1d13c485744b270404ff9adc8e177833043bdc34c307e6fc9c55c53d8ff84a6e251038dbeb5ef774
" ~k:"620f07d7e7ced030e669685ab8c39174d88c79eb
" ~r:"3591c521b2a56cf46051c0cb3d444b9a22fff63f
" ~s:"7ac78ee252440cf9e8510494d1fad8b518f1e128

" ; case ~msg:"bb6593ff219c9f20aa47e1e157e88ed59ae29c8940a527c82e0e0f2e855fa98e94e07be1f6bce3832b7ea1e60a5c9ef583f2ec7b179227e4afdcf829d673e1377f832ae38e7cadede415964f12baf775d38ce38e945563e72861519197c2d08f28d8b6466562e059ec41741de349ed5de2c7d6cc7518a87720a248b301733a47
" ~x:"301c11a34edce1f7ab040754e0b2d4fd88572187
" ~y:"15d9e20c3f39cc9e3b8fb65feb64fb1568f6efdef6457d231c491ed51731d58f06e45ea5d665d04969823da4e6750a2c3d16c5ff6080ecd09aa39c006eedceb4dbc16ebd64bc5b1e44b831a6bb25afbcb3000fa6b6c2000860014011189c22542c145e407e7b59f6d3fb1e136295ec850b14ff2f4994ea37481e80199910be8e
" ~k:"117e12f88d6e44fc7f4d51d5384fc31b2e3419a2
" ~r:"61e727716cc96914509740a7cba6e74a9dec6406
" ~s:"2e77c14f01f22180bcda5725cf0eaac9ad13a7d1

" ; case ~msg:"565f19244468515e8463d07b425b4d5f81ff2efab5156ba19a63734219c226ccca5903bf9c35dbca0961db7c2e3f6944d057edfa6c2394c39a00f1c42596e7ee72ed644c6a182115bdc44b9010c86e7b0ec2e3bdf7016c5e04f455b4cb693e32490b8f494bb4103b3b5ea6808222452841b733faf735f10a95fb283dd86ce593
" ~x:"07dfca41446b2f4e1af2a67bc8468db9a9c2dfe0
" ~y:"664245aaebcf5c055c32109b2159a17473043087915f14e959dddc0c9b20c726f0124f1ecbaf202fe2676afdabd346a7b5bef769a25c6f733612d7378df1b2d4c518a2da5b3a4cd0252bb8180838a46389a84693be8cc24fbdc639b62cb21d8abe1272b5aa06222fe2133fc5556d24e75496a53e1934d3b5848e510b69da04a4
" ~k:"4522d27cd17a6ee739873d69f107d872ed7e2db5
" ~r:"5ca07bc7cd9f7a60cf79391d873b6fddf5a48cca
" ~s:"9799c74a806fc196e0223fb1a613fd178cafbd99"
]


let sha1_n224_cases =
  (* [mod = L=2048, N=224, SHA-1] *)
  let domain = params
~p:"f2d39ed3062b13c916273600a0f2a029e86d7a4b9217b4f1815bf2b24d9710a57ab33f997294b014585b8d0198dfdccbcd75314da5ff85aa344b45adaeaa979b51a312a7bfa94472fb633f1a6f156bb4458867dfd38403f06b851f00fe2d3484077bded71ab7513d04a140220575fb693395480e4c8402b7a46cec2d37a778c305accd1f13e9f62e865315f4b22cc467c8986ec8e4961ddf810566b0c4ee369ac6aa15e43f4744005826f5bde8071a19e30b6909aac4b3d174237270dad02799d09b8a2cc5f22e66894b5422228b2c234f11f5a771c5b89cf465a2acecbbeeaa1725fe8f9b59422be8991052cb556ddf2c8ce8fa9206dbf39feadc194e00f8e5
" ~q:"8000000000000000c118f49835e4ef733c4d15800fcf059e884d31b1
" ~g:"e3a93c09da6f560e4d483a382a4c546f2335c36a4c35ac1463c08a3e6dd415df56fdc537f25fd5372be63e4f5300780b782f1acd01c8b4eb33414615fd0ea82573acba7ef83f5a943854151afc2d7dfe121fb8cd03335b065b549c5dcc606be9052483bc284e12ac3c8dba09b426e08402030e70bc1cc2bf8957c4ba0630f3f32ad689389ac47443176063f247d9e2296b3ea5b5bc2335828ea1a080ed35918dee212fd031279d1b894f01afec523833669eac031a420e540ba1320a59c424a3e5849a460a56bcb001647885b1433c4f992971746bfe2977ce7259c550b551a6c35761e4a41af764e8d92132fcc0a59d1684eab90d863f29f41cf7578faa908c"
  in
  let case = case_of ~domain ~hash:Digestif.sha1 in

[
case ~msg:"edc6fd9b6c6e8a59f283016f7f29ee16deeaa609b5737927162aef34fed985d0bcb550275637ba67831a2d4efccb35296dfe730f4a0b4f4728d1d7d1bb8f4a36238a5c94311fa1134a93a6b4de39c085e9f60ae4e237c0416d58042bb36baa38cba8c896295b745d5376fd8ce42eb6ee5a1b38f87716b265b76e58cfb24a9170
" ~x:"6132e551cdac88409183bd37ee1452cd247d4834b08814b275be3ff5
" ~y:"289ff18c32a56bb0b8839370647683a38a5a7e291410b93207212adc8088d30f93e9e4abc523f3d46936e7d5c90d88742b36afd37563408f15c8c1a4f7ac24bf05f01008ffee70c8825d57c3a9308bad8a095af2b53b2dda3cbed846d95e301eb9b84766415d11f6c33209a0d28571096ab04a79aa0dc465997529686b68e887cd8a205c2dc8195aef0422eba9979f549ac85548e419413643b7244361153ada1480d238cd00dc16527938955548dd5d027ded1029eeeb8ed6c61b4cd59341d8b15466e9da890a989996f4d7691e6072de136af28b5874bf08bd1f8a60cfb1c00888132909f515e04bce81b02951aa41baac68ffdb8c5dc77a1d32d8f2c10dd7
" ~k:"7197392d32d0af6a7183cc3398556f8f687d86a8ff742be6ad38562f
" ~r:"45df2f423e94bf155dd4e1d9e63f315ea606dd38527d4cf6328738c8
" ~s:"59b3e8efa5bc0ccbf4a3cbb6515c4b9bf784cfacdcc101dc9f81d31f

" ; case ~msg:"3bd2ab08217878e6774ec7797deb75d5c94c40e24ddf1fac8dde3a29c86b26f57157d329aac31a6622e1d6dac97e22695d7d1f8e20aa26b06795c2f878ba5d2b9cc4b16d5fa60a5fa5c24c09031de2f970a9b57ea24af17192ece21a4d120fdb52e62b8238f778ff8552fa453c0a8891243fc8757188e9c4e0e749f7e9cdf1c1
" ~x:"32d53ad2620c156e4617a8680c543839c9be93103e80cc0fefa44ce5
" ~y:"b9b0e1cd37bafbedeed173fd709983f53c2c427f9f61c895fac9eb549bd6201d05efd551aecb98b2df80142dea7a35491d474a3adc83f0da8dc4eacd7f6d7201c6fc0ab798abe89dcd7d0310d5f00fa10d211f18ea853579e2fe31ee55371d1c9fc4cfb050786586659bdc0f1aac4c109b9e4f9416d22c42b39a471311e28a8ed62f1f41bcfe06e074bb2f1acd29597953c3b69d3a78831fb2f83665d04a1395775ea3a2a6ea142ec00507badd4de0d9c102eac7bb894f7453e6a8e0dd3f14978377d1ddb1fdf1c55835b9924f42ad45c847c79b3f83fbf924f80b78f50329731016763e01ba8ef69e81523e181584f45c21e3c8edfed4e2ec56fb7b02aa4ee9
" ~k:"2e8e4625de74e31bea9e480a5de92890095b6ce36897a2337ff97d53
" ~r:"6d19fe3c415d6b07d6a1039a1fe34b106daa2eea4cbca971cb669eac
" ~s:"14d7decc2cc05a1700fa256e4d2994bc4bd957bed0baf9a18bda7090

" ; case ~msg:"c67fa77cd7351d100c7624e25418481f8fa499d75f5949a5cae60f96a0f7bfcdda7dba373f9f7512a5f1460a95213077cebd912e2662c43ac6bbe38c4479b04151a5e2d2880902d031aa0dff3f41126dd09fba5c0507634ed16c3938fbd3a96473a8b1ebdc37d32c767fd5932efa235555f3825a1595369238675453604d278e
" ~x:"062bd01487e413074126d9d47258b5c7c77790a9db0af952ce799eb0
" ~y:"31939ccdd393f747541a5c69f8e509761dd67eddb42e0bdfc412d4cc30d368d878d26d856c5290ec746b59c5e5af65ef3fd62c9a2dccfc1558dfbfb62cfecd7838e6d59495b20db5ad9d78e82f046f9f1598113aae0a79601d6b94a32e05f6ecfdf2b9c4cfa720debfc212221b14b0dd6f7078205a4f218cd4b8f10bea8fa481eca5254f365d01f3c86520bf254323d5634b96920a13b8f29d734e07fde8064eb0c9f8ebb6ae0b40b4aa7d26bb8d80868231d4558a278045cb5f2951cbfe0dc97bbdcee7af8c9b1e3b63cb49dc29f300775cdbe4d2d27894e27e0e7c9eada13a359f0b92b449e9d069b95bdc2aa7c85e56811c07207a150e598735996a6e5349
" ~k:"7439c7aa4446ed540ba50b9c817792b08fc0278fa0af2daded03756b
" ~r:"7924b76ee76ad7ff2ab327dabbbd31336750fc7663df4b5b94eeb62d
" ~s:"5914cf965490b0bf8192fc6e169754bdfd31c48d716361dd15f45bf7

" ; case ~msg:"fde7434c41666022d6d7dabc7a67315b1ff49a2a85a6168f2b6063e3036a4f35e66d2872af3d97e5beba239698d88e13bd036ef08cf0e83a41664c3d0d21863c24129a6a9b27b8e96c8029ec673e07af7246ab77a56c21ca208df4b1818deda906b553b2b23a37b5a05e29825ebeb47f53986c2bf26d731a5b731fffc353258c
" ~x:"6e6bae97e3b37a402eca050d666b6483cf7d700419c5ab1eed1bed05
" ~y:"59a14e36c9ededdce8000f04c6f7401ad987f1c7a5a070b80e0aaed7751d1d50d99a580cf205dbcc3797a0a0406b04776d80f2f2df418cee249b98672de7e61cda85cfbe903690e54642dc2a12a90ecf88c59256a4d77c4c0cb54e13fa3647b11431e1734f3ceeea04fbf3459665e999fc0f7a754683e48cefeb4a95fe473911ffe0de0f738960753dac33666c53ed2893bc63dd4162d7a6328739a252cdaea7a948c97d024153b55d14fd5304e3575048418808a952675fafb95fad84b1156b24e98e048aa777a745324ec13ba378e83b2384bc2e96c6095aa786bd28fc3be6bfa4db0c3c44fed4c351bd88a19e179a6a7bc12fc014f17de46fd12ef1287f72
" ~k:"08544a6237ac967e5d11f2eccc6618399818b891df7a04d08cbc5e74
" ~r:"49ea82713aaad799e263809e161b0655f1e74323a06041836f676980
" ~s:"76b3f6c1647f8d17718ffb92d6e1424606ba9724e5290daa4ee95efb

" ; case ~msg:"6676a3a131cef7e5647ea7590da3c704a0f5dc3f37f26913a70d430609cc2497c45e68b7bd6f5893dba26287ff0d240bab8a0761936aa709a2162ebf1c20a6136a748352dc39ba4403cbe4b0a5a54a729286dd193eac1a2e6bdc150fb06369be4443a60e75e5330083ff009eabb05232c52368a26fd237c7c3185c1c7e7d5955
" ~x:"18faf583215bc4fa71791f6f34e682ab3529aa9a1a71c1fc7bd456a8
" ~y:"ddcdf4c616fd6e4016099fb34ebc4ec507290762c5ee6876f10c6a2dedec97ba86a6063aa8ff069f3f3db40c9464afb1ba7ed691773afd6083586b14e35694a9ddc376ddc39dac57132a05bf88a0a6085c72a80a21c13e590c68c4e98eedb67f1e16c8cc7e9e25ff37c87ee3be9adf1ad0b838651b0fddf8d026969d4a16bbb828fcaf00efa306fcedd5ae19ca1a1abf44a2bdf6f994123ce941fd3523bc1335f51fa8dc5d525358bddf0c55fe2ce07ce974408d9090488837976f16845eb7a82d04c43a704be2dee1be2c8683b2d9e5c44f1833f5c46c65b6e62c2a720421bb35843fead7b9e0b3fc04c646be39e890e370b982bde91f2fc18442b650ae602f
" ~k:"11b25b09408bb5dd784ad70264e585c978dc02cc1df8bb95a28aedfe
" ~r:"1658a7ef2f444b014a1885b1eda8dad3605b96c3948e544e4c8825eb
" ~s:"602150f67b19a5e3e39fc53abea02dd8f3b30d25c0b4ea0bcddcbdb0

" ; case ~msg:"071f06a11588584da5576013029b5a14712581a48408bbfdbe34e17568c0a0e4d12c1e9c3fb227101440dd8dcdc415e3b49f68a26a0ec7612a10bbc64ddb8f7ec9e9750d1efc9c0574700875fcf52d00d37b9dd744ca841ecf7566977c1b5799dc4105d0b7a92551c5b33a50133fa300a5908b18f4c01936347c6049447abf29
" ~x:"58882f1a41e08bf6c8dad091a299af0fbbd14515c1550906ff77f6ae
" ~y:"b1f4dfc9c83455f279a3e7522734d6b120ab8ed36ccba7854e26504c707983d2a9c075045723f441abfc7b36fbb5d4bf0447678f5b709ca5129b74888a0723e905769836b9dac1303f1b9ace26554342b6e6d032ac4b477e011a4ddd3e2978fc0c449c64a66efe7d4e22e5a5fa2b01bb17fcdbec7185dd4115a19d972fb01a06b33bb18b9349ff95fb10dbbf2dcf899b1817d30ad48a99a614d57770ba764a11a84a8db3af3041ec362614f807196ea3b90d05b014054ff4e18524c795e6722c0fa1f6d1205d532d94347633eb132e6cbb596d8b341e65f2b2f955872ebd4d3006c45ac33da11167fa46869c7ee70e9cf147b23368b3aacd9c1880b09ac86a8d
" ~k:"5ff04e754fe3246f35b3400b87a450192a7bfd9b3c03f3ece93449f4
" ~r:"07bd3f6718e39839304ef54ac48bda8d9ac8ee051a49bb9131dcc918
" ~s:"6496b2469bfb5845485004702b0c79941bc3c3007007ba169d8307ce

" ; case ~msg:"71279b848c00208fb4e4d87979cf973b321b20d098dea912a3b4b5789cdd3b7ccd8f3993a9c92c34b70e9b0bd57520db56f2ded3a612a6169d2a1cc6350905ed0202a25c113b7bf8faec4edd2ea3b8f447ca75d15a712b4b4394c22de0c2554b9aa07ec8466727e7ef6f1f04ac4568d7726d9d77f50a2fd551ac29e42f8dda23
" ~x:"292b1666d0b1fb361da268de725b11310000705964705ee975d4ebae
" ~y:"7c8d63b9d55f59290b02a0fcea6d98c6c545e2c0d4b109a069694d80cb034dbdbd9edb6b4d9b152849cabd655fc77071644bbf4a0c7ea4edfe864a43c44fded163dd899c21cc2f9c33cbf56e8caf84394b24f8c14e84f22c3b0f747129d9aef41b6f1b1fa8ff5a8f680b496595dbc7b7b63a7790e3628747011b3277b06e80de0b67942f602eada60b518f282cde69cd717a5f6a19c8e169449e0d32a9d8ce8f09a5ada23c12a02dccfcdc0290a8bd46e8b7eb397494f32a0ecb49fa5a8edd41845eb417fbb8cdb89a9f18b9ad1b41dd4122ab349bb3c44951e4f9604360fcb1b795311545a61cfd67c287a7c9d4d3530214988e7616979e2ce907d5c7f3e9ad
" ~k:"6b1b752bb180d8787c71505be758c0ce41fef428ac10591502c9a04b
" ~r:"4cf5c26c4c2cd48c05508e52d743ef48685f6324141adef23d79a396
" ~s:"59f64755a04c90a14b187ae142ec483c4600b6fbbe19f04a49e9ff88

" ; case ~msg:"3ea03e9b005ec1954fee0c73326d8aca1a4f63648eb4cc59265528ee8e969ecefecf2797a0144c8336500e26a1c7cb1a642b1ec65201416e5deb355201de2bda695d1beba8dee62772f4d5914a245be9ffecf39408ae7bf1bff7c2451029c4ba0c522516e89955ad3bd699cce94c744081a9f2d60f5c5127ec722fa57316cede
" ~x:"087e432b1c29c00508d768fda7c4b279fc088c48439f09980bfa159c
" ~y:"1239c347be4ce6f1daa721fbbb141ee6e2f7c73098effe8e71beb9f1ab72d1b5bd3e78df770f7fbd4b3a9505702dacf102eeb8a16f11b4f809ca002ae3774ac0407e2572ae3ee1716458e5f45c493f4b921144e858d87d63773d023745512b0cc02b31ebfe5c24ad37efe539cd393cfc2b951fe1b6ffad2a2824c0f54bd776aa0afcf9c1ef427afc6cf4c4b17f66355d68574132e1d88ade3722513e395fc62d65e9485157c82064c90803a1a91f9e6b10af2f80699d917daa6b81415e508193152b4ccded593dde35f645e54b7cba445775eb16c5e19073f0a9eb5369bf2513b92158165b94dea511e938fb6a8798e040a05da94fdb5a4d44bee943b95b39d9
" ~k:"0a8a45ce2412cb84e4e0174d7ecd2eb5b37ad0a53b474fa9bcf56d9a
" ~r:"5ca2e971f21b70127a70c655eb87e20b2517976228a2c4e648d549b2
" ~s:"44036b34667136a5140dd1948ddc2fb2bf679efee21f29b7ad87af7c

" ; case ~msg:"a3f7033958c5b779072b0548baedf4f88d14f11a6dd6eec0181b399943d7246a45d50c4f7b5295dae4cd3ba7c4c181fa201581ad5c4b38793bcf454f176868e9cbe0997aa41987b1aa3d5ddc046be7b022fb5130594c8a9df03cfaa7acef817e3ba5e192c69a120299492baa52a9be83b8e871abe318b4a1f588f9edcddafc17
" ~x:"5831abf9843eee928944e3dbb759dc7224910e1adab827a04f596e3c
" ~y:"62de2465edbc1ef8458beaa205f45f9dc0fc3db77bae0f2b13bef6d803db689b8f5c747e3a041c08d326cd7e1891675b022a9da3bbaef8007784c56c86c4176c0ac876351d1062d9c270d548c8f4ec39a4556c66e76e507fc5f2540abfa77c178a9bae153435a7caaa008f36b9cab213ecf5e19a0f7b1e62fb9a9c8223bb689e8547b5ec915b04a85b2f53ccc792dc0a7a41d172e6f59f5b5e7c440350ac6a72ca9c06562d4cf8c60e70870a978312e19bf54c2481c582296b64554bd871accc8b251a7617ca5e5d2aadc19d484d76bc3826841f88fad1491d80679243e1527197d02a406348b247ae786108e5400975a38f3961758adc07ce740d8d442f152f
" ~k:"36b3d1d36d1a8c41442b6fffd46bcd7977a306b53dcf7fa590538194
" ~r:"1823f0a807fb9e71ad69b8e9fc674cf76f67c42cadbea6d34cf1f1cc
" ~s:"667fc57a44b289fc34a198556117afd696dcbd96bf1baacb40d3f8b2

" ; case ~msg:"680d878ca6eeb87e4ae158dddc3732784013ebb1da89401acdd6109089e5601d695f9e4e6ebf16026aa746dee80a01235033f242079af1b7fa6965c87eae8b3291a009e4f19d5b8f1394d866e7c9b72073a95652c0eed98e9484a15c9244764d8cbaabd49d24c207c705703cc35ebfc7683f4a0e6abf23fa07678350a6a00cde
" ~x:"738a8bfc478e462c4bef8d5633e0793475206551bbddd08507f005f5
" ~y:"511a3608c4bda7c82d7d68a5d30bd4c71e457b0f323027d601d6324a94893ab3d62878b12d98c44dcf30adab4352b70f4daa772df6aed3c07587e96c68f8a847a335051481d53903d1d1ae0cf99a54387b00169a1c9704bb62f1d98047dba8a0fdda734cd41611584d50554ad77890720c8ac29932097cf2bb1a8d0daf8663241e23640cc396f9e6877348f014073f4fdc5bebd115a0d74c2ce857e100ae3dc0707b95effa4a2cd8629fdd9bce72091c0e2612d2b30320420f42ecb0986ac3289251b4ae54e51ed83d0195deda9d4e5b398b037213d2f8f0ffdbf727214085534a324d4fefc1653642035ebdbe8167b150bd92b7cdf276fcf5e0bffce956a47e
" ~k:"58d8b64bc8c2da02a294e9db46bfefb273e74870651e19d6cd017c55
" ~r:"7ceb71480b5a7133401b5227fa2253332e04f78ea5d0fe237c8525d1
" ~s:"484800e81f7b5692b79eb21ac2fff83c49c9f0d409c756c73fbdd2f0

" ; case ~msg:"697f9efc8653fedb898c77f90f124bea5c3b893c49d7f1b116479e83d35cb6c3940797501e7f52887d18ae9f4055e1bdd124b572f7a6fad101f58b52b30ca30d9743a9016af891896d25356e44f982d406ea26a9b25fc4f903092d7e8e8713774a8be7aaac93a6942c1f2c48e9dea64984ae54f7ef99961bfd9b8d93226af776
" ~x:"550c8755237857a0c8fc8a63525d4025713b89bdb127d1c330c3324a
" ~y:"64b588499c9db3e5864192464d32fa3547f648fe676c150a8f9e153c5af57925c76dda4b419d60b22fa5cdea0fb6f0b8479c988a324d275bd42ef10d8998c36039eb4021fc0d2788b59a75cf25ed6ee4d44882b0c5c5cb8dcc1002c0baa4798107e0b57cd26debbcd0ba41d1b1b860b8eb90f6f30500b2e4be7a00b67d93c87d3ff7a6ce53b977a930999807fcbef57d8dc67a8f366124991389328ce7e70f9e5c22ffdedb28498282b4a9a9c68534a238322e0db6088ed0afa8bc77ce998c814471ab56767b35d07b94290ea106ff0c998b51f02222738ef9301f290c6b485dbc4f12b472a1192fd93f2d23527a02d95af0b422be7640a9702ecaac26c9e004
" ~k:"0b4329f9e5ac4a117689883db2ca8e968d30a3aced61e27ba27c6242
" ~r:"62054d11529b993a6f19a0d5481b99b4b4461a49866c29534a361a8b
" ~s:"7a7fd0982e4e2118d1a069787a80b902493465f6620a355c86a94867

" ; case ~msg:"d080a7dff1ef20e33832b99cf83c6c919c07620bf608e080aa301831ca6178e44ef7a4c115e93ab6d877e96652171610a51d927d2034f42f280fe87d7c1747c480ebccbf565a150f3240f6d4ce5d6eb0b2e964416791376ed22b3559cf93a019676e9e0be3c8d34f0e0d1152ec6c326d3dbf1d3303beadd188c3aa0d77e8a117
" ~x:"2171d5e7cdda9a691dd27f0524f24ca41d5d801eb2ab0dcdbe6014ad
" ~y:"41767ce26c780e3f2019f5a49a701570148e9ff3382203833d1b18e9d8d6a00c0b2258f2e567db31ad4e8cfb2621794bac87d9b3b53b79199a775058febc190d00adedae0fd3021291bc2d1ff0508bf019eca0c573fd863722f367d5d02bd9fa0d07f75406ac204fd3a5ca16325c661fe854fd00fb26654752febbe439096dd2284d5ab13de9eb004847d1d8599fee687cb2ecd0e5b761d91a7e9c58e6921f103024215e74f3de3cc12f5ed7703def041dd3267f1cde0d4fda8dd5ccc9c07b65de59482c4784b4f6fa85667186e2df6c5dc8b495be8ec61379f20923576f17680c4dab99312d0b6441306ae717c95d3f352ba4c096f01d14a7dc05b28ba9a3ca
" ~k:"0f6626008e50c19def9bd694c00522cc861eb7069d55892e08ddff58
" ~r:"44e70d2ead3c51dd0c5461dd4186825e23b4e751d8ab17d0b7edfaac
" ~s:"48ffade27531db478f22fa0ec92bcfd2ffeb6db67715dcdc79bcb028

" ; case ~msg:"f6a9afe241dd984e3bc265787dcc49491b3bca67feef32fc1e07fdaf0af6c5d06dccb47cdb6907511cb30c109f62e66718c5c4bb43d4b00b51235df43223d60ce1f9be3493a4dcb02e25ed3ddae10d131b481a61aef334b690c7a1ec74865954b39ccfa7a51a9a1e62e654bb89270c774f082adf09b579c8358dacb9db7ca1c2
" ~x:"77207cf0963f1e961c3539d7d0f678fce517f67b728bf15e0cab3ae6
" ~y:"b4138fa4e1dc6772b47e5a3ed130a13b822394c3ce8a0193d1dde4c90e7da1178e1126dd296252fa7d2f139a148ac44dc06a058b84ecb03ad827e66892e85529c362ceac2e7104b797b2e9826054de350596ab581765e9a5c9ff5143332c2f3bfd249a87fe1e30efd6fc057e234a1cd4c19e072bd71b32d55ef122ea930911081e26d998490376e3b721cc32fed92b82d545a7e6ba6e4eb434063c87db848df4ef02eda3fdf4f9d2905b78f7b16b5ea0b5998f1fbb0aaf62a1735591600f9801977b1b947f61a91ff2afb8727c55268972c87216aae900617a56f535ed18c4c5ddf8d7a54463256d09144d889c149e5b09bdd9d8509314b103b846f3e6fa1bb2
" ~k:"57585204d88d73c21f66a150991531973978dfeaedd8024e268f18d5
" ~r:"555a454880084f6cb2522daf3399fb4a501a943a9b6aacd58e2c7d37
" ~s:"730fedb3a5911844146098ac5603e2baaae76962b33a327b50420a50

" ; case ~msg:"2d1c573bf324028dc2fe00928f55f7fac79037d4d99eb185f3b997e042cdf808b5382d50a6aa8085c5d1958e67283df66986b93471c12e3045ba146ed5965c8ac5b44668f61984d21736cf1c276754b848e9fa636b6315b2272c19e65626bf8b1214d70989a623b5fff7803d28a663bbbbebb84c839b42720fd0e62246b3b034
" ~x:"789375055f94b9ade40b0af8f70640336f5de213571ca1c645ca468f
" ~y:"5ccdca35551cf355ec85db8d68010ded63583255b1d5fd2a522e29513ad3ce6157be30ea1c305d87de6c27fbe3a3fa5007128275d6e6183a65cec5b694bc6c027335066e01273fd6981cc5f60c3e33751386ce792ccb6e6a6db5d7f073800329f9cc46d19f422923b9748dcca4971e43a9d1f59d1c749788a8527ad524df74150b39eafa7f4d5608d1c97255654456eadd4d382ac54fdd12538b2f2ef75a50980171a04d4054b4cd79c71e1c4deb3bc6af4c874f5cf0273896d4fdc5847fefdcc97f5402c7e76484d3d2d70ac16bda41996cadcd83ad92cb37c0c1e9d64fa1abd9a2cf005c2c29a1737cdd6d63aa2fdaa560799b9f07d448760678477629f22f
" ~k:"325b1562d5c9c61f95e6944fb12a4bb08d227c4dc0c8e9a79e391b08
" ~r:"7bf3c0c547e21846212bf4cf3e38362dd4d359b7af6420f90da57907
" ~s:"5ebd5d2d88cae40b37a9a5a84e6218d2453afa146c79a5d5f5df44f4

" ; case ~msg:"bab4db55bf6d3abefd1bb4e0f7bcec65ee6c6d8eb04b7c480df4e9e39150f10c38f1abb63dfe1bb9755c41b38955ba38ba938b6ceedfec02001fa870070c59df1fd2d72a814104c5143376a3136b8118f7b47bd1ffab53359e53f95c66ee12705e31a462a8caae481556ceff607ccc8bf1450772cd68081d3f15a710e656ae56
" ~x:"6f4a94c9254a557787de9afa08215414db5a0dbc67c66cde1c1e6f04
" ~y:"53c0b0b0269fcf2948667e28b11ccda9cbb9275463f21ee30da33c4575be5e111a182a6f38b890f20b8f2d224f5981895310db7c4703c1cec2b257f452d964be50c014b752360ee24f2fe1bcc023477a2d7085f58214df866b13a8d8af913146dc0bee078aea1ce645999b579498eae9277ed7e8b2c75f494efaa73a973f32232f08ce7f0afcba316623b94158de39bd4c0d513234ee1a481d5b72f4eea37749b40fff12ab620f11aaa01e3558e7a4c550707b71c16cb8cda98f46bf71769a476c3385a8caf7c886ae47d228b1771a8bd4b7f19e6f53047f62f029c339fe7575be93080ac748289149a57a0ddced54d72f6d4d344fb874ccc85ea7f3dd2164df
" ~k:"14fe2a5a75756885240ff29abd19d346b2e7e5dfa76d2430f0d069d6
" ~r:"118d2227be4bd91e98a2efde15609b2b9124b2e83c274b632300432b
" ~s:"3a447461944b2a59278a8e1118b406bd3ff416775d65530e54f9e623"
]

let sha224_n224_cases =
  (* [mod = L=2048, N=224, SHA-224] *)
  let domain = params
~p:"aa815c9db1c4d3d2773c7d0d4d1da75ecfc4a39e97d5fa191ffec8b1490a290ce335e5ce87ea620a8a17de0bb64714e2ec840bf00e6ebdb4ffb4e324ca07c3c8717309af1410362a772c9add838b2b0cae1e90ab448adabdacd2e5df59c4187a32a23719d6c57e9400885383bf8f066f23b941920d54c35b4f7cc5044f3b40f17046956307b748e840732844d00a9ce6ec5714293b6265147f15c67f4be38b082b55fdeadb6124689fb76f9d25cc28b8eaa98b562d5c1011e0dcf9b39923240d332d89dc9603b7bddd0c70b83caa2905631b1c83cabbae6c0c0c2efe8f58131ed8351bf93e875f6a73a93cbad470141a2687fbacf2d71c8ddee971ad660729ad
" ~q:"ea347e90be7c2875d1fe1db622b4763837c5e27a6037310348c1aa11
" ~g:"2042094ccbc8b8723fc928c12fda671b83295e99c743576f44504be1186323319b5002d24f173df909ea241d6ea5289904ee4636204b2fbe94b068fe093f7962579549551d3af219ad8ed19939eff86bcec834de2f2f78596e89e7cb52c524e177098a56c232eb1f563aa84bc6b026deee6ff51cb441e080f2dafaea1ced86427d1c346be55c66803d4b76d133cd445b4c3482fa415023463c9bf30f2f784223e26057d3aa0d7fbb660630c52e49d4a0325c7389e072aa349f13c966e159752fbb71e9336890f93243fa6e72d299365ee5b3fe266ebf1110568fee4425c847b50210bd484b97431a42856adca3e7d1a9c9c675c7e266918320dd5a78a48c48a9"
  in
  let case = case_of ~domain ~hash:Digestif.sha224 in

[ case ~msg:"e920fc1610718f2b0213d301c0092a51f3c6b0107bbbd8243a9689c044e2d142f202d9d195a5faef4be5acadc9ff6f7d2261e58b517139bcb9489b110423c2e59eb181294ffdae8aad0e624fab974c97f9f5e7dc19d678a9cb3429cf05ec509072856f5adfec6e29bafe8e5ba95593e612843e343111d88a1eaff7dc0a2e277f
" ~x:"7b489021578e79e7bd3ee7ab456f659f3dc07c88f5c9a39e4f8cee81
" ~y:"1ae10c786ad0902c5c685dae5c7121418a377b888b5f2f2bc76623570fd62bcb190b471ad5359c5f062f8819289e956d8aa6f90d1f8cf1ee72d3a1bdfd56c478dc29a19c4569b5a60e3a8f34f60656eac5b25dde5514a5c67b675423204f6ccaf0990617cc7355b9d3ed868978a252020a769ed59a6edaa6efe3377eef45f3f6f3e64179cc7db8b143fb835c5d71bfcfa1e2a9049bccf7fe9ab57546220fe3f4b7521c861739d138507e81a46a6993605441dcb90d6ee4afbc42cabe90a254444968109d7edd9694a023239f1d56175dd1fac115915e24fab563f4fc3f269bed2f300832d112596485a711417aa73bb4ac72a651a1fa5baed3636c720d397008
" ~k:"37fadd419fcbd2b073a06ae96b9eceb63e29aee9ac5fa2bdb31ab85d
" ~r:"65102e8f64ecb11f06017b1a0c0def3c29897c277c4a948b1f4da6b9
" ~s:"21ad0abb27bd3c21166cb96aef70c0dbd5f3079cab0dd543d4125bd1

" ; case ~msg:"da5e7b051c1859d22f2a3163335d277951973c172e06697c0490ff15b592c1ebd0fa5efa2463119804a3fea224b96b463e30083e002949a24e922031764bb3daff8101fa088af5457af36654c668f234a00cd828cc740a898c0cd3df09315da9b346b325b2fbec475210b75482affa61a3eff50c83c3a039fae5cfa8d971fddd
" ~x:"9d8bba124417c126c1c011115906a7bdb7a493661d8a945e32cb283c
" ~y:"5e276987b847b852cc372e986e8aba0633dd46c461bab58acae056d4d1a9df03a19df114f648b28e038506fd09ad0d95449d9d8058aa1b241b2acd3badbf9882697331de45b452345c051c2cd830f7cdd7486b1166b93891a72a8b7dc6228bad708720ef33235801c4d4c3c4f28036df6029a195d0019124d16fe8f76c525b7e8f04bf4b8d8ba6ef608e623224fa8d988420f40526c25ae3e4c79d5ae7fee69793e02bad9651ea0fefd3eadc5ff1ca2d142930355b1f3aea102221fa17b735a18af3b83327c8f33efb9a49b70211014eba43fa65eeaf25ebf452bc4b7dc1f407d0cf1b834619b5f73c6cab7051c92070aa06f7f9406c507d1a15d12c11bc839a
" ~k:"1abaec5b4efaa83403fa970ff6027fdb596359df930a02baa12ed854
" ~r:"313615836f0d338d81b670f116a5414d2ce90ea5ca5308ba4f0c8a7d
" ~s:"dc1d4c3c06203fd598a476c891dfe5934162d0d35f37f1c09dd6395d

" ; case ~msg:"f49895b3290d9aaeb4af611c5e30afc0047dd42c07216211d54977d1497fa4ee6abe11000d6ac04d24b4c50f31e06ee8a74774d3d304137cc6b114d145250ee7e94a12a1ab592ae307ef5d930cf39170e9756adc5e7ba62a54abb6f047b4500b6121e1f4a95d3c6a96f7f8333cbb1ebeed8b4db1a7fe75f4071cebfbbdfdab90
" ~x:"b9174a6cb4d3b2e7e4d168078e920ecb651343223575dd37c0677371
" ~y:"6d622525ecf54dbecaa811939ee07ef2975d9da9f7a3c58bbb893ce3880677404f2c6e5963b8c0b4492601f15bc6fdfd747a00ab8334e9053201e1c9fba55fbfde36ec54237501b87416992771cb5ab8781d0a967b7f14f3d5de6b1665f662885878e50ad37827b95c8f0e21d6bbebc9dfd47b2957d2fcdd1a2b25a616e698129b45998b6b6aa2a99c1ebf4275493e28eef1ae34e9ca63cd8886da58572907aa9b714e89bd3644a7ea029fa3a4ae9c26e665c85296204fdf86b7b1dd7866bc8e9385e9518a270248292594c54a4a03dc1492664ddae53277c6fbb9dd0cdd99bf11eaf6ae31923e4f979a7f581799dc432b1940f613a7a7ea6855237f776e91d4
" ~k:"1c52eec9523245bd82707f2ebdb05fee6d34749f23023ba72a5a60ef
" ~r:"79d544cdecfd1ec1b7d1ba6322a5e0eb858aeb4b76d5b3202cea233a
" ~s:"0ea53dea4ccb25978a0af5529598911b47c25e0ba3b2a0505fd1d7fc

" ; case ~msg:"31d739566914549eb25726bf6d4b6c674f479ba7a406acd108a106f36c7f5214976dcf3adf2c83fd26b37d52c0b5ff51e6b3811a8dcb026a1fbb52f95027ea6034d91149b30ab4928ede26ddd692ddb8ddd929fbff83fc673788faa0ba5d967fd1339299e55be51cea80609d2b3c3433cf713a9686e229336cfa7e720fd5303d
" ~x:"4cb56c8acb9c107087837ef5e021f77cb015023c8ac1ec73575e5289
" ~y:"386cbb8f7e728751d4f6a75f890502989b51228d3039dd1af7f2dd0186bf97a9ff763b40323b30ab0dc81bf09ef48db72c0cfbe772b3d314927ed19badee7b88b49ee294923714adae30c955d37b99c1dadc4a29f0f8c2b9d1038d17059c586a212a9748720fdec95b428971df1923f08a01d35893d12ed17e0b142ed8e9ef77d440a01d77905b92c51dace1b345cd19f91623a6964288ddee6e9908197f91da9a26f806bb14e2371742f849cdc6ce7a045a704a792e5760d6644eadb7cffaba806b0545fae3b9fadae4e36bdf3b69c6dbbf0d8b053da38b904e9c4b949325b2a005b249276ac36927b31793f80115b5e2f2107f987710380708e2c322894fa8
" ~k:"d223b9e9c662ba6651cdbad84f2616fa223fa8742f783c87c2fb9e8e
" ~r:"c8b8a92e8c101505a1991bcb02fb6e382a3ecbaec8f4374501b657be
" ~s:"20d161cefd584979224379f28d827aa219c572f9600147f4048ba7cf

" ; case ~msg:"d0a8a1ca0ff2b44b37ff860007334b23be4934ff89051d787ce69d3d7fa734b9779e2f0b38c235391a897fb8514b857b991d10e34a00dc25b0c4382dfb6d53aa87ec1784f1cae2599259406d4756539867679d3088913a138871e2a434747222fcfab079d9e655ba254463cb0c5786b9858dc429ffdadf4c3b6a253f90eeba24
" ~x:"2286424f368e5e64bac0c977ff0d92a560b78e4f21b49f3aee7cdec6
" ~y:"7247d4e1253f0b52a1388b794815db61c1a354cb0f73fd19fede615c1c3025840fff204b0c6e610ebef1113df56f67406badeb99445891dcafe18d28f597126064ddf7aaf203b2fb0d35d2f458bb74341ad937211edc394ec1a3f7909a3f972db27aa135d31bbd7e36c2bbc360585e7bb6e83276406b9525f688ee5995e7aa8ef7a72c27e990d64016b99a0ae4d04b2f1b7d238af88ac4c2e4e0f3294cfee9be2457e48955948cf4bb3a445a1d778cedfa4b86f59f156118034b2b834a9aa121e9d482d6922292823be2991b3b5b42c23925da294d5ea37406eaf78b7dc72519d8f261482d6afff0e567bf6e673dd89960ce734f092d98956352429a91845694
" ~k:"c2795f65f0f077e32c022a703f7eb8e5dc068fa67cb087ef366b243a
" ~r:"9dabff22a43012dbf47d56b9ae5a09f4d739dd69fe907725afcd84f4
" ~s:"b60c44728e4b1390f30238fba1dc1003fdd39507ff5d6ba7e609f2ae

" ; case ~msg:"e4ffe72c77c3a43af8a61f58f9240e1a07b5c2894d5bdb654b2b994dc0c987bad9b704075d3d0a969cecfc98b1dc20e76cd8e012285819462226a84dcdd67895f6ea278266f1575ea785a2c359f8f4593bef31a58091b64afb84cdfd23e4aaff29d9626f0c823d934283a4faafc9c6cc18622328cad96f77d79b9ba35a43d825
" ~x:"86b0e564ef08e089c4c85675b6e5281daa4e82bc2fc0e27668052e4e
" ~y:"7146009d12b03b2f32305f495fafcc4d452efb85cc80d671ff4249492c6699fb26a89ca4b224d56f6b8e745df9fbc7352ca583222f4deab118f9fec0b34e334060bdc28db872e0090649149499e7a1c197878d3c7262439303b90201d0b7f5be94d0a7c4eb15182935296c3e3fa2d77d74d78f41cadaa40eafd40d017888caa02a474868e40f496b7bc1ce367f503435e0d9a6375aab03c231d9cdaa15de23c48ac0878ef649eb144ce6be4d2de11da202fae82090673c83b32840a32df6176e1d55027d7a1c1c56e642f51aaeccb3c990898061bfa16b3dc1461073c333337fd76a3103f3fde821bc994ebedd6ffd7974d0ca1b54961d7df5b9eebbfa26c3d6
" ~k:"5aba2fdf6b24bf24151943a4f32d2794e44d1f62e8c968ceb5b073c7
" ~r:"4a2abc689d2a63e8b23214a3212a5d20a7386882d5e11c5d5daa66bc
" ~s:"08e0c6547087b58bc94fae247e962da1a2897888d1bc9c8cbf3ad6af

" ; case ~msg:"f8fec19288f3a8bd1d0d573bbbc180106065697481bed912f8752750d331e3a097775a12276bc4293a78a80748b2b2c37d20b800335c1d1b430a71bbdfd8f7afeeec82ceff2fd33f2624e49d37457f262cf5dedef9025ce96e0b7d499fcc7a7ff06c02590ea821dd8ed060cabcf4feec9592aceddfd32b4c09e4d44938435b82
" ~x:"e5ada29e91ccae11fd060112540eac31d9651b34b2754ee51620624c
" ~y:"7e50011d422986eae01ae68943dca0c87af44f7b879bd1256d4caffa0eb1925029c0633a7ac67487a7b6f98ad77ee7e1442d129d06db475a4f7804fd8c6a038151911f81397e963594b9c91e3bfe94328f056e9bdbb9b11f54939d7e237aafb0c950e0581cabfe94bc26f0e0d5560997bfb0f6357bbf2cadb0108ec0095646e4caa22f71e1f17a9f34e8a8c4b71cf0b1265e001554fa91f18a17562bc0948c431f25945962ba7faf7dcb64ff0b8bdde701e1df620a11aad07196d67a956ebe498ae6f82324f75cafbe80edaabef0037b79c3ed658d9ba1b5422c4ac053ba69bbaf7fa9db990e8b5e7f9af57a79f3e31c07611f502b3015962b02b6b425706e0a
" ~k:"cf0544a08823ea2ad5f13716b43b154aa4bf80d6bbcafe6040ad91c3
" ~r:"2f38c5cf86aa0e53d1fea0e65dd03813640404b8d9a8cd6d264d9285
" ~s:"47603880f3d67ba1a6eabc20137dc4882e417304cb95d622177df511

" ; case ~msg:"7559465af5ca04c1e74deb9f8e46b0ef17de4d7a2ae0faf4e903a2998bcaa09b7f1730393320ebc57d052d2e98f5486e8e92bd1ee6bb0ffd02d69e5d4591e2fa12e4ebff8b6b9d3270fc75274f8f82e1c60edb2a21f8d5531a2380cbebb24f6457176e54769a136601a9b81da68ff196ff8cc78cf059c04ae22459cec7da89b6
" ~x:"6ba814fb6c1d9fe5d282008dcc9af2761d1b03eb1fd02e2499c1b509
" ~y:"5bcd42e586ca180f743395fc39e2bd393820f5b4c49c7cb76921ec38bb53e864fbe809a033775f16c7f5c64872fedde6abc560488e572955edd3f9569092071e56df211564f33185dbff180e7ab2297700c64db6e220701cb8a21ead2ea809f06a16554319b2739de2aca8057a62d4caa7957a2b9f039b3c7d4fb0761a73302a6fbb583100b239d727158b4cdc9765fe0485afb6a1b0ac0db504a947f3d87faa5542c6eef7a681c5fcd28f4636360f5593bff7e433b6a338d77e3d63f6ceff69536e2a3ff77ace745b65a5160d7fbf9105a90f46ce1c54fa353c8aeebe16fb238c8ed998617b63287511208d9db3f66d503374bbda48a552d04b2c304a15bac0
" ~k:"70af9c79fad2b3a0677fccadd95e6f72eb8a51464e443d1e5c007f98
" ~r:"c5d33f5a4fe2280a9b96d7a9b5530dc17cd1054bf1e8cf6f4aa3e2ac
" ~s:"c9bf1c062bd1e86f3bd3c1ff582c33270537fa7769b9592aef12e104

" ; case ~msg:"1674823896c5a764c61fd19b125a7d6cd58c883d86794391477349f03616d75b6925e9dcc553dea37047f0cd153168eb26e5ad4b8fe7cc65e4fa275514c842af63507f901fd110b98249133d3d1266d2f967c85b7f88dd76c7f76b786b5572dcae68cc646e458b8278db346b2e970c7870cffd8457fbec06bbb5141575f40fde
" ~x:"b5a607136e5dfa76645f4fee9db17bbcd260b1f6023f28474921714b
" ~y:"5c34135c90f97ebc9bf1ed986eba563e32ce8c25ae7141dfefca8600ad2f3cbe8e45b4a010ae4997820a38b4888187bf207bde438a1ec7befff81a64265a4ce9900b37a38e4fc23613887b638a113ef41665ad2b1f15764cb53607d0eec303ac48c055f5aadabcfbe2c5faa85e029c43e1607a3a29f65802959b686b468e8107c466a7317b5063e038021975b2f017f1f3bad07cd0ebb487964151e4f82bb5277c35a218ec570cb568ad0404a3713ab7fcc1297b1ea9743f85ac5d5a7ec818e5f90a4a58f2c2192bba6dffecbcd39f245cc932953190ee353a0ca99dc61eac4b4f834618140c9a32eca31d718c95ee03b2992c63a683b0628883a5c222fddef0
" ~k:"02e860266b3b7919a3d74f37f4fa9054f62f37959ee1ce66baea3b15
" ~r:"b1a946fa42a36d836daab56fe015c9f29c4544a4a47d482ea2d9cc5b
" ~s:"e2905ee70a5dc099b7e0baec5566b229e9ca8e7e00840966cf56c4d5

" ; case ~msg:"281fd14ae2e702dbd25f77d8ba8af09fdd77b1839648ab9c880bd119d4475378fcd0d12415abb9f26bfb8e26f108b1298859235ed12e7f9e915628e3ca36c5986d18811a5905aef7878c6300a95ea87182016ec595d32e4dfc274adb47c3ed0f6c38ec893b331f7092f19b724b9fe43f0ef8dec14fb7bf8b9041b5390beb4408
" ~x:"272b54a77c97fdfaaadf12ee05e1279f65e8748ef873c407372aaf80
" ~y:"48ed8fa89d07deb5f8ee6d38748a4e66002020f79ff22d66fa53ad913d596860d4dbcb7c3a6633cd4224a80e5e95908f87b18acc2e364c14b51de6bdda7ad8961dfda454ef4798d0f7a30ef10eae87de40867764b84bc55d7c0283f9c7cd2be08e1852487512ff43a8d1e68a951197c771f9e6c2ffdf2c00ed2163f86dff5241f9e2ff1cdb05a0b3e647e6fd23ccada83b9c5961e6e2fef3297493ddb0e990295d38405a24448e249627c0a7998cc4072dd29139c5336d9856016642992cd245c758a3031ec2807b171abaeef14c82a3dab201752351de2bffa5085c137656e469581f63f86379d62868ac3e3aa24df9826a833314bd41e0d9a0ae5680e6a4d2
" ~k:"bc06f559baf16de28e915dd27485338abf2bd0e62cdda5b3f1ad05f5
" ~r:"5a77639663664e3f0b19fd583bab6e680688cd89d5e012ddcb1e06bc
" ~s:"d41c784b583cbc525bce87c6caa44062eac847bca8b005c12ab5e554

" ; case ~msg:"503f2042358f7e414296ab2d41f3a1f3f11182eca6c82b2ae6ee833dd737bcb34691793e30110036ae54d403a5ea45cbf3e5515bbf80b1af139853f506792df7ff5235995e080f82b562326adaf321159adeef20388024509f225e8c5235368a7b045d69e472e6b2ad7d470a11f6aa8d4ca6c6cdb0f3ed4e06fb9a95e2cf200c
" ~x:"c3ff27ecdb6a7de642fb2d2f9d93ccb51dd09b543a77fb2e7a22a29f
" ~y:"7e514a04bb575ab93e71b3555cdbac634d475c58c1d9b4802e153a858d027804ea748c2907eb9987f78e41c6757ed5cbf102544a714699a02a9ef14768f96dbbdf48f3b2b3792efb973a7f91f260e0dea28034c915d9d5a87a8f986a15f5d6f98d7d6d35bee7e059aedb59fe595ba7da17ce0db895f3411b832a1e221a831f706587841d9323e0c7f4435703127084b20eda9c6a2497280190a2b5273b231b44482c9253501c66ef1122253be4ea3477ff6186af871869af1ba10f6a15d1c432940317d119dd761ca0342ab606d532c471783a4dcd6fac9b8a67a6bae187c7dc64c7611ded7273dc348cd7613a52d02670e877e18d0b60c8bbdd1adb04eff213
" ~k:"ac8009b8bc2503f5a68d667696c7fbf66ebba6f88ed3db3504c0c9b6
" ~r:"8486ab31c8278fad0691fdd6cac2f5fd790b2f3fed52b09986766042
" ~s:"b6967b9eacde5f4883710eba387b3c6fedfc91944ea51f6ffab72531

" ; case ~msg:"650c3c409a885fa6d1ac1ff41e15f9001f6cd6a152c376fd22e2851c9cbaa5350d8a92b7401030809395cf0b1a0cb03a24dc3b4347050e8553da0e61d81dee4402b1cec97d898dc6886601024f6bfbc48d2f2c40bf96de9bc0e078e440c771f74e7115ad22ba994ae2f857c7fb865ea750b18c79e7b048563becef8898ced3dd
" ~x:"d39e52c39ea46d6ce274670d3e8a22875cb9873daf4c2ed83bd3be37
" ~y:"55186de39e6a0131adb7d84170a8d36ac4bf313616e750220de356fbb1899dbaaa650d8de9a7afabf3c4dd6a3c8bac241922acbcc4bb7fa4ce5fcdb5f231cb17a8c0978c8e69fb82d44683ebb9fb17898e0ba4939196ed9980ebecabbaad7b5b34cd9ec0ea6df96243823b1d170efccb4d59bcba24ce5faad32d591ad6ece0440d2b62a212059e000fb5005abfec127c1e9fa7d3469c72b89a96976eb4702f09f9c0a0971b30dfc339072b5e3a6ce40bfea2d52f2c930a11dd655dd36ac9fad86fc3986b4871e7c90459a2eaa3b3d22dd04cb824173ccc087d429bb2a188e05d8af0ac2911c907fd957b2bb330a6f3987a595930b312053c4bdf856de7293858
" ~k:"78683cfccca3e13d49999e7bacccb43fa33e11547014baf66b987b83
" ~r:"a0c49d3c47240d30d26f0c20e4508b360a841285de3fc1986f1ef9f6
" ~s:"97caa2b76d15b1f9f177e209004a2b1fdd23a3945034584c2c15bfa2

" ; case ~msg:"64129153eb9ccc74cc3aae1d5999c6e90d986be6fa40c6c4bc00b1c3f8072d10a9d8e6c314d82a7641f8a3ae29d3e7dd1942dbf0dc52b4b4b35bb67a994942aff029ca6fa18709915ff720ab8f65f231155cb1d0dbcba04fc5193afc71a5eddb4a03867e5c4bb92d37b7ef771da954ec6754d5fbe2e372b92df6a3ea8c3a4aff
" ~x:"a7d5664e781c28f4859f5c126cbe8d87f9b2aa0027149f8b0a921d46
" ~y:"23f538d4ec345faa906eff12f6c5942ac166914baf8e737dafc71e47285512ebc57ebf3ec666342abc059b0ebddb021ceaff6ef75828c7be3766257f7247a67e1408239fa4dd1caac2b7229e8c1bcfd57aeea4c04e1586764e28669c3612d8a006582cf8f82910482691c10e4113216fc24feb299f84ba58700a3bb6fdefa17a7fac9aa9bb410fe411fb294d6294396f7f627dca0452ef595dc24170c147d3863fc16e23645019aca63fcc1152b0f766f5f651c9bb699e2f5047fa1e9603972d2c7551b18f3b16c106ddd6cc2e24d2d05e79687efe655102e6bc15bc3a57f60c1a6ad20bf1cbe62052ad0947437b92b2c932af5d72775d43183bbc6f359a4df6
" ~k:"85adc235c0060b510825ed2b436bdf003f4d63e299e973b5ddc81fc8
" ~r:"3d728962aec35822fff99e1b5217d8a6264a7c608d8066f4fcc9008a
" ~s:"ca5c8e178a14ba006e93cf4ad119f045bbf82b828767d3e583d0bd15

" ; case ~msg:"9fd2791c41a2ffa6df26109804eaf070122e20bbb62ecd9811551136aa956dc1c321327893a0dde6dd1d5b3a0d2a5aa97ed754e5bc066753338dddfc68eba217d2483505b0d7c0a437732f8046cf3bf5930a11efd3f6599c0f8d465fca7676ce1f39102cc0cdf13281b2c7b9cf7a7afcde681005e5a2e4e38cf82e421357a41f
" ~x:"ddffa0c5aafa1acf98290ce6aa7a48db2ddfec48d6ea881745f2373a
" ~y:"147aa8d9e4ccac906d6a5a0b65bfeb59d4d66037ad40d288d7534fc9ae33c5aa701ca18e60f0b68908280562110af7d1d1bfb538c59d9100980384ae93b77be0332a03cc567d4d634f7648a1b9fd25daf250b2869683e9426d75561a5e1787c2bab71132757dffc4b7665143e7d87d50f12d01075bef5f4b0f14cb3f109d1599e5bf94de0111a01af57e8c13f583be4dc90089619c72d22a495c45256ec787a5832d2e4c4a42f0001837a975ac8fbb8c565f77b253303b1a873306fa5cf6a5dab62d7b1ba3d70dc11b4e4f875e3edae50ee8e5178dd09a334cf9260c3e0a10911d381d7f5601c0b3f26946682018629922946dd73f81240816ae9606911cbfd6
" ~k:"3ee8b1f03687b9726de846f54618ac45f8e2d6e8957ce6996bf50c2d
" ~r:"a7cc7486f47fe62fe3254ed655e1c994902d797f0d7ca93fb97df9c1
" ~s:"914bf7d15ce2c9ecc5ae150d6308fc557d94e1ef18c0860aa68ad48e

" ; case ~msg:"6b78b4de5f7526dbed08ee0ff4e43335b60cd3bc371b70cd4fd9ce45bf06508391085d142cc3891b179167c76a1350ca8ef8ce754ab1d624572e437195660f004cb7bed2ff3b0f7c7e53f853305a3821dfbaec33e220df3c3ef7a79f34e82cc8fff8415f108c000f21c3bb21a4c33267a213cb4a558e3b370d17c639247bffeb
" ~x:"9da093f73c714e0b9994078b6cc748a675cf4f3bbc502a23895097b3
" ~y:"9147670f64aedfa246938ba77fb9c1ac271ca1091d863f32f00d5ccdebe7022d268ba9051d80fe55dfc5f64b0716c4bb8da4b11e9e283448ed8be4278e93b52d675649abb45956522f92634c92a09ac5a5d603aae2a6d04a435239538de303fc05b9ed5fcb843f0536a8ab942d9c3bdc90feed97449ce309be8ab119676a96c2a60a06692e8cd59e55e6ff8d91fa462966555526c987fc44ba420bbff768f7a7fd363638d5ce4d9ea1edd7fd399d6c65627bbc337f131c7345b3d79b4db7412562547ca2a7c8ea55ebdddd05a4b4200c72ab2b83311152b71c99306c1d3b3d446657be65e58d7cf8a062b225ce937802590546853f192a6a8c8b3ff7a62fcf80
" ~k:"bdd792b1ece3d0ce428cc1294b9d7497208de86929a2aad2ef481557
" ~r:"2f85ee5c32d546c68f0aa2698beae53e2848c375517a570e0f1b5546
" ~s:"547667e8b13f21635a0b106d324d06c85b74a64ce9225cc5e0843581"
]

let sha256_n224_cases =
  (* [mod = L=2048, N=224, SHA-256] *)
  let domain = params
~p:"a4c7eaab42c4c73b757770916489f17cd50725cd0a4bc4e1cf67f763b8c1de2d6dab9856baafb008f365b18a42e14dc51f350b88eca0209c5aa4fd71a7a96c765f5901c21e720570d7837bec7c76d2e49344731ca39405d0a879b9e0dcd1a8125fd130ec1e783e654b94e3002e6b629e904ab3877867720cbd54b4270a9e15cd028c7cc796f06c272a660951928fdbeb2dca061b41e932257305742ff16e2f429191d5e5f1a6ddf6e78c5d7722cff80a9c0bd5c8d7aeba8c04438992b075e307c1534c49ad380f477f5f7987dc172c161dca38dcaf3fb3846c72c9119a5299adc748951b3dce0d00d4a9013800b2008203b72465bc6a84ae059a30c4522dea57
" ~q:"ce89fe332b8e4eb3d1e8ddcea5d163a5bc13b63f16993755427aef43
" ~g:"8c465edf5a180730291e080dfc5385397a5006450dba2efe0129264fbd897bb5579ca0eab19aa278220424724b4f2a6f6ee6328432abf661380646097233505339c5519d357d7112b6eec938b85d5aa75cc2e38092f0a530acb54e50fe82c4d562fb0f3036b80b30334023ebbe6637a0010b00c7db86371168563671e1e0f028aedbd45d2d572621a609982a073e51aae27707afbeef29e2ecee84d7a6d5da382be3a35f42b6c66849202ab19d025b869d08776476d1ab981475ad2ad2f3e6fd07e30696d90a626816df60d6ca7afd7b482f942f83b45cc82933731f87faee320900f2aa3e70b1867e1430e40be67c07f9290299ef067b8b24a7515b3f992c07"
  in
  let case = case_of ~domain ~hash:Digestif.sha256 in

[ case ~msg:"cec8d2843dee7cb5f9119b75562585e05c5ce2f4e6457e9bcc3c1c781ccd2c0442b6282aea610f7161dcede176e774861f7d2691be6c894ac3ebf80c0fab21e52a3e63ae0b35025762ccd6c9e1fecc7f9fe00aa55c0c3ae33ae88f66187f9598eba9f863171f3f56484625bf39d883427349b8671d9bb7d396180694e5b546ae
" ~x:"551595eccbb003b0bf8ddda184a59da51e459a0d28205e5592ca4cb1
" ~y:"748a40237211a2d9852596e7a891f43d4eb0ee48826c9cfb336bbb68dbe5a5e16b2e1271d4d13de03644bb85ef6be523a4d4d88415bcd596ba8e0a3c4f6439e981ed013d7d9c70336febf7d420cfed02c267457bb3f3e7c82145d2af54830b942ec74a5d503e4226cd25dd75decd3f50f0a858155d7be799410836ddc559ce99e1ae513808fdaeac34843dd7258f16f67f19205f6f139251a4186da8496d5e90d3fecf8ed10be6c25ff5eb33d960c9a8f4c581c8c724ca43b761e9fdb5af66bffb9d2ebb11a6b504a1fbe4f834ecb6ac254cab513e943b9a953a7084b3305c661bfad434f6a835503c9ade7f4a57f5c965ec301ecde938ee31b4deb038af97b3
" ~k:"6f326546aa174b3d319ef7331ec8dfd363dd78ae583a920165ff7e54
" ~r:"9c5fa46879ddaf5c14f07dfb5320715f67a6fec179e3ad53342fb6d1
" ~s:"c3e17e7b3c4d0ac8d49f4dd0f04c16a094f42da0afcc6c90f5f1bbc8

" ; case ~msg:"f3bb27bf9d412f13229a56d2d1533eae63f40004c143c6b92f6e606d263dd2da7581e5eb20b6cd021e3ab63b498abafce01b4ad7ac8628f7a1849c4e454f1168ae97adfab1fadbd313fca7381726f5045752dabaad6ea3250d303a5496bba2fa4895ae49f06a9aa6451ae70cf33b5f06fa17cac0144f28bd19fb2ac041a578ed
" ~x:"027d0171598e7ecf23f2922d0257e604291cefa77b5cfaf1b3e31ac4
" ~y:"00c7aabe30fa4c3d1ba85e7ae0aae79360e5eab3041bcaaa5d321c92f3471e4194c10484cff152bade6b7d619cf286773475298f883efdf64c08b692583de31be0a4e2b8e8d508ec145c65a369ce6195446c52d02372eba562f9a9d7cb24d2ec3b0a1ab833e4d7623b0455a41eec759d07a3c8a20d88a926408c20f1675601be53cffd65617b66fd4eb353a1f2db31f66343b07faf60de0b6a680809c6166adbf5e504c5c61babb84be72c02d3ebeee066d9eab0d0ecdfe01b8ccd6728ee9123b9d21154b2bc9a134363566402291ac8a484ee32eb884046d40fde7cabbf51d1d1206df1c5ecf290ab7ea72abb5bd3be8d91c02bb63f809718ba1d380af88331
" ~k:"7494772f199ab7a7e9a248f6c2df918c9da62dc2d4176b7db9419b37
" ~r:"79a6aed73ce177ed3581f5d181a77f000d6358514ea95cb0388a6add
" ~s:"2b8597a694564e267b6f250a4c76361f8cdf49863a7902afa48fd6d8

" ; case ~msg:"e714c01631704e9447390f5c315c9615a7a52863b143706583f661595c505aec477eeb5ad6d640ca812ce11750b67bc8bede2e4f9618dbe7376cab6231b21248ec914ae182df8753362d2118a65e66f64018810804ad97fcc1a87b8c9f349d1001e4b09b046991e6abe6338fbef7be48f1c80c350d2962eb6b8fce25b69f8dc9
" ~x:"6911c21a3da88d54ff9ab58ae2075a2affa3f3eb656978ea26bfa702
" ~y:"04d301f001821b03c91394c520839ab6aaa95325c108a02dad9db48b3c8033d6443bcbf05045230ca88aaf98a8c4cb6b095b352d91b4c416f632fab49d45ac90699a5a419630a81d473bc89122eb5bacb91c40caa4e4bcc476f3ca77bf6a21037a06be24f11c645b0c21b857fdc5c04fbbf0a26efc569cdbb0ea989ba0e037c23f22b0c5f1643d77d98f2de248ccc36672d397d30c1c5e1319fc7e5842ae1a9fcd9e96fe890a74ddee91a39ce732e4c0eaf7094b53b7b409303860b0b4944cc81b4a42d40538cfe512b9680e0a281b1fbbf639139e8066ad638cf846c9ea51fb4c4ef84921f16a6ca3f2bd158157c551739c9d023e270b3de7c2f1d7683cf809
" ~k:"bfb79665f7d6df843d2c39357173e415724c83e1a10932efb9e22676
" ~r:"790b4dcae31fe45cd3a7bb6fa10dcf9ede1f067123f93baad7edb489
" ~s:"71e3e46dfe040496ce4c5e490f6944a23cd5e66ce9b4d9acbe4130ce

" ; case ~msg:"3f6e482fd484ed3d07f1d0761f2d60fc96d46eb0ecd10a59dd4f392e3d3b2cbe184010e132685578b1f6303239798a5303a81169d4f52fba0d20a42834de293e3a7b32848b65dd308eef5350d633297465425b7b1595ffc8ea7b125896f89e2844561635f52ec62fab2ecfea288d23f0a771cd6311806103135172cf9fef1455
" ~x:"20328083aa86511140324fd0357067a1d6abfc316e77fe3d260f0ef2
" ~y:"9fc1b292ebe15531579f35dda8d706bee0da857cd696a10af770dc356232736cf893f7411a9d2718b39f388118d177cd8d0fd7ca3b3c220f3aa743d8b167219d3c2c783e1f09d8b8df8ec7e17578c5329488c87a89678d2818a99366b785d53f6ca6995e193ba5ca26c00b849f9027ca5df5bb7ec87fe78735ae880f1a97dabc3ca7985d8cbc81be824c1ffb953f1096bf926226fb5e9d4ad43e9363da5e6b738c9a2f951ab3294e2b2822cf5282bb4134158aa90ab9c8f0f64d05a0d625a75bc2d6a4ae3dd11fc05ede7b6647ae7c0750ddb273fe5f8828318a91db3172ad59166aacf2da4f3704d169ebc860d9e1c6464abc2b653013774d29375b77bac1ec
" ~k:"8f4398bb9fe1b393c1d90a62e178899261fa0501c98bd9a8178b364c
" ~r:"3b5d8034c4b8ad9701bf29b10006db69d017fde8638079dd7bbface7
" ~s:"cde01df54a66cef3c0538648525b250cb1f08707f5ff114bdebff8f7

" ; case ~msg:"31a278f881fdd375565c0f28ff7575f216110486d6fe08dae8fd072950978bdff601ded1ef226b5d904c47f7142a8f4665e03efe5870da2dd1ab80e449f5c757b3b6996a9dc0b5b2750b97bbad2f553fbaff2aedecfc9ff6a970d156e4fe3852979dc913bdb296a321f766367239de45e47cbef4d79bfa3d576887c65f7f8a60
" ~x:"b75ee80c896b42148eeb7d185d45f5872a3758e983b4fdd8c2e71ca0
" ~y:"7ec0a5418828159e3ec829f793b96ea3460328dea21ba157d71ac306f9bcde617db67368d592bf46d46918b130fc7e3d1189eeb7996d5f660ace30be509a26b218d865d9e56ba7f61942e567d8cdab96a78ca303c6b01d989b1e78ae956423e35b5a466c16074e0bc9e8372340d2c12516c22d5e1ff65abd5d448215b6baf565de18201c1fd5ba3de87e5d5b437d2f48deee72a12e655f8c7fa313d24bd0c8c20e59c90edfbf5dfc057c6b679850ae41826178f2f304ca3b92a9bac31ab3cf74dfb8ee5b643b4a341ebbdb5dbd24d0b782c5b450596abfc3df9ee05f45d0ea2e8ff4357cd3605f3506ce58a5394f1f2444c26359299af153532bc90daaf954ae
" ~k:"ba98b478a9e12a1d03b6aca65c0acb265764357cca67d04d782fded9
" ~r:"2b47e257bf72adf34d618d3a6c46142881bdd0689a46f1cb3199ee6c
" ~s:"cc1ff2fa3755a0e81edfc753bcf14e637413eaee0f22d7886b058dcc

" ; case ~msg:"a6d76047bd18deefe70dc0a4bd082a10fa521dffda782a9364b9e2b11e147e1a36a11c4300672144d9b974132b4975f27ea6e8e46b55aedd6723e53e7bc9b40dce2449285a690885c3223b636cb5c4873c5ddaebb0b6dc5b69438d881a525905a51bdb97b051dbfec6dd4a7b580297b08f2ba60f2ead3a07531cf299977413af
" ~x:"1c0e4c78a4ad4f5046f929e7cd3db3f48b86e5eab4a5e2be61a08dfe
" ~y:"8b2662775bb7f19252204594a84b469f3dc8d66eb7993bed122d8a065f59ea81d4c484cee5bd766a5c137dd57e43e941339852150509acbde6f7957a1b04ece718565ce8b637ea031bfa3410a580744b3d4959a5e75e315dd33c02b52c7c56218b7cdfdc24f51ddb4e7849faf289cf806c4d3c6b877c63dbfab569920a2b219c39215c5e3e638a3ebeebfb52c8b38e8285a85d625fc1b42fbf0e518c58eb8f45fa54676ed8b009415d2696ee9b5153dddc5eebef49cc7659810a98d4b5e8b9695fb2d9e4bf192093747c878a9565b47cba053c48ba7c0b9b1ce77f8a3e1043e87fcc6132cbe8fad7c738e9bf79bccb414ef24907675ba7cb059a8389eee7ebbe
" ~k:"5135933094326e3953250a29d5f5c4c9a1033ccb844ab35a14c19d31
" ~r:"b8674d1ba6f13398f5e8944b82150d9e9bc9b210a81495b335947e64
" ~s:"75fcfe96926186efa12c007c0985205147cf65abd108363d8b891190

" ; case ~msg:"f0d5b33327695536e351b37cd3feea693f10377a5f8bdd913402c2ed67a0fc1e7bcaab002fa779935950c76e42a491a68fa6fe445cd35575cfce5f376c29c4e8c0fed5a5487ef418b96fa5752a033ad07959653d1b8af6702dcce40efef21b2d64cf06bd8b03dadb2fdaaa73fb2d3d75b0985e9aefa1f94442a5491ae46d7c51
" ~x:"269055de62d0742324803624522e678234c3600ae7bc3996c8d17bc9
" ~y:"a448b0d448249a0e54a94586882985a08e19972281d10d9e7fb57f95dfeebf971f6d9dfe88dbd0a4950f528200be7b605865eefd8ec274ac53e4ed5b288c6a00721e028881b9725fb0a9ce4153dcc1fe7b5ce7259f16ea8b32456cb03bae81be77f3f6e8b39f52587bc9dd47a264278d5d8ecbe1ba574269696a7bb1e167a3ae7110ec057f4291a1bae8257d69c10ae095f3271621c6d6b5607925c3498189d751b7c9bf30e65683cb39fb51bd592f1f98279f2e7b2b53ad546816a8508c93f03496de7c47165f5cf297687ad7d60f010ab9faad0153432ec1ccdf26d4f441df625394e2104208bb675e7f972b6c66ed7028a1e3f45a671ab2716c60feabcc22
" ~k:"0d9d0b3e1f24cbb18320f9ce896cfca2a5a6bb28ceec83e1ff3218d3
" ~r:"01a4f4bc633ebf842a28d045184d250529920df280545cba00501cad
" ~s:"09fceb2df200b7c0a56ae7969f5473b7a1f6b703f743f954a4fbdbe3

" ; case ~msg:"f58e039d666ef064cccc7ed015017c68393d1455300d0c4fd4f0d302c43a0022363a7cb01bf0673d325293bd50b27f8187d88ee2b553b159a97d15ac543421446c2aec39566315211b9b4108cacf9085dacdb4de94bce84097c0892b1cc65f2e10d74e5293a04a837b616d4181f3fe4caa4cc2e744916e770ff0ab1368c86cfc
" ~x:"3752b20033843d1ea4f48018bede79f39c15de33df64140259aebb82
" ~y:"4052534a7726cbe17e34555648e5f297b963f22d3aca249785ad932f6ea1fb5df31d379b68522f8eebedfc9b5c5277e91574fa79ecf03780cc44351f3e3bfa1a0587c88d0e04e0a02cd1ee9ae210b3c9aacc65c71cf1b86463367e2be25ccadd9d5a4d1fcbd58772f7a117f3673c76ee2a8d93446ffd7cda7f8430490502c16b1a5022e12a3a95a7a9f20e98d3b285abe30e8de42a11c517c14ef3b6e5b6c47114a961d858c6875561c7d5d21b7c93f373cb330800728ea188b2578a6df34772a7acddb829c09b3acf9bc5b06140b9b035267a40e86c1af5577b3d02a89b20a46573c87500a2ebed4b00b1fb13a86f143e356702d791379a90dfcc26b80719ad
" ~k:"1220ac99b9124f1dc2212ade5691fd330d6d868f3e90694236d44b70
" ~r:"31fde5f22ebb426f256b175057a76125c40136974ad58e681ec2c4a9
" ~s:"77b0614dd99acbbf4c43aa926b3f0be1cd52d52775f22a408c4e0304

" ; case ~msg:"1477aa0b9f1b199b6aa0931d4d3f766d80a3af10c9ff7315391f15edc4e92632f9d4d21a8033215d5e99cff170d9888f020b0db0e5b97e123a2889898c5b0ef7c832d028afd5e385004531ff9989797c3bd954b1ac729066577667567884cd4bc5d055a3f645583d29cf4758507c883c5bbfa74444b9c5b9b495072c3261b6ec
" ~x:"83770784916227ab2a73edaac5a95f7538fd94f89650841d79a37d7a
" ~y:"4675f19b0095faf8ec96888e483f3a0aa675f5b425910765069ab57c97a12b7c506437c8757fef54ecc6d310921d7159ff39f2f1cd9535b64f27f136913715775a238fbe01237e181adebe551ffe5d21e3c35774e7ade8c79df741c52dabd8be4782ee5a3b607a39d1b455dc848301847312980566f55eba080621e3c123142a1a2074e2e39f6c0630b36831f074869d46a68429f62573cd2c671726131fbfd566a6d07193db4f367802d7de8f4e830aa878ee2cdfb86d8537746b71c70fbcb6a1fad66213d6fbea68241eb9f617478adcc9faaab26cf81b912089da0c4b187b496a17d886cef571e393d6f1f857ebf517c801f9231e95db661e8cb2095456a3
" ~k:"6406035023c5e150e8758baeb00a9b858ebd0e4090334c69e2fd2377
" ~r:"a2380b5ece76672669e26187a17da45ad89de1726c826e57378af707
" ~s:"9cc26c3456c0a409f4cc98c83ea5176eb293ec7157e51370726429ce

" ; case ~msg:"fc82372566ef2c626b2145549a5db973118dff4c6d1d7c4a2e16ecc31b43c14ad3683173535b0b82331f15a183e6a50200fd1e88ff903ecfc50bdd4f5875e264a4499eadbdaf807f974f8d8104477a0e4d30463dfc61cdac5bf44eab96c770a7db912eee2db248cdd2b9b36211f93870beae6bdf8e0aed0097519ecde3470cdd
" ~x:"8d2855e4ea3e5085a5c145e324e5d5a5f8f23756284669279728ec9c
" ~y:"3884ab23ab93d9d1b716712c8daa080b26af01657f0dab715ebe6bd766deca7612bea6a4cf1ff7d08abb2d4442ac0eaab01e68570bdcc222f84bc3dd6d8c5490132d1c36e23913f00d11c803b703a69a51a1b475f56db00fca47d234aac307b9e798e9fd891dff9c1257bee556314b021fbf93f75ed8c43433afa715b82d5ec6af8ef9471e9b02f9554ed7957c1f46d8db35a5921f4a83727f754e82b6ffa6d1b82595220876d22e18fbafa5333b26c2cfd47d894aaa7164a2630294d0a385fc8a8cf57d10ed0fc53f21f1fd6b4c27e9c69e65a288444619a3c248bcc44ec25605028325243274d72100edf560cd382babee1ca532b7f06a4388f181dbbb5db5
" ~k:"5cc12f090fd965c719efa2ee907a43b3643ca8f9ef7c537adcb09189
" ~r:"5461b20704453b6c51837f7b9ef5836131b501f2539145ca3481e6af
" ~s:"b65f69d291ffae2d16e3108d69aeb01b4f9202afa01382e53dea4d54

" ; case ~msg:"e66aad54048bececa5682644d5274c18068363e968e37e6c11c1f8a0d7e320578514e1874e9d4eaf1bd02da6b722ed22acfca48c3acb670a6f9ee62e3aa71deb18097508f431b05214c199c166fa42cd6a0797bc7b4d1a2f330cb62c2c95182fef0d06862542845e430d778c82076387adad4355c258e6c543cd656fe3cd2332
" ~x:"2c984e8464cf5716053520b6a72c69798b9eec1e115b0a1e30e2e44e
" ~y:"06245bc509b4955440b0e401710ddb2c4ea2e559598361a3666c4ab12e766b439f21b953962f6ef5a11dbee5677ab7f8906d8b325180ef4e45d05c1294fce5dcaf6360f71b10b70556f306993d295b695ffe5729c5c5bbb6cb4834ad037bd8364a12c992c2598e8ee6beb1606ebc0ac0ff00c0ea2eb8aed75dca01a890085a400ebf993e5879382ff91abf1be2ceedd1fc4a874342b77b6c55ffe7f676a1c95ee4ecc32358a080c92361cfcd2e3426f78c217ae29556709ed029b287e71feae0608cf3938857040d7f06b0f91b3b4da8929df4b5698e734a37316879c308a81c096b723bf2089910d5ab30b8eff38858aff6ecf764e268ed698b70e8fb7f3c66
" ~k:"b20370d79e097e7c65e956d76aea1e288b668dacb8e7944aba5fbadd
" ~r:"86d5bac3aeee9b501f91f2fa71b1066760df2e0ee147383f145bb0d3
" ~s:"8d6a207802d6fd6e534e1b8a1edb997b7cc9a25a97a9e4b6eebd0e23

" ; case ~msg:"c85747cdd2ac9da0999b7e5d7f64d11dce7673df5bc605051316b4b94bc7fc776fb1d3da5a4395a674aa8a0798a341b31b11e63cdfac5f854346f6a4b74b49f2d089cbb86fae54ebfd95eb9f05a1b5e84306e930461ad7f827cfb910014a3af4dae0d46ece912bc26870a433f70f0a38bf23b15d98cc658848f4bad9c84e89f0
" ~x:"4076f4abf4d3c9a55b3f063535f6a69c221199581e72c5a8c31f1a71
" ~y:"2972787dcbd67e5bddaaf1bd3f05ebd66949601dda44237ec9361591ce9b809f8722fb399e6b9b8109a79ea7b83fe98359a07a27e232cdea8f6533e34e37db3ae53309f62f108b2ee7b489a933e4ef58dd4db8c0108a3670c675b98b75798ac0884cf5a461af281f6dd8e7ea3d41396f049601a9af2e39088ae0a1ec0d2d10fae1dc1de962d84d8cf04215fc6d6262ac432541af2c48c09cd4e15bd9460e9a7bae17e0035af0b13d8de707870c54bc851112f4ae1d69074712c212bc7e13f199ffc8f37723cd6dcf539f8df8cf0cf1ed4c10eeaf0f444804f1eb9d9c329d6f19973eec273222fa04b5f1f0e17971ce399869582027b1c454dc1addd484902cb0
" ~k:"7149f49e3d07c45c97db09632740560e5b0e843240255da43ae97ec1
" ~r:"28e3dd71098ff04d1ca885c2774f78ecb3ecea708fab2e16bd5cece1
" ~s:"ac8b6ee498ee383e28404ba4b53e64aca0fcd26790713264fe3cf6a1

" ; case ~msg:"a7a59da62a9391cfe628697548b05f8af39ea9821d76c314478e210fbcd27fbf6b0bf460a65dbcbadcddfc0178ece135264a7d7c5b7053208bfbde54e3338d901927e95e1dc8eeb73d299e6fa6584555cfeafd1925e95e0b3558ddec641175fc7293c0310266ace18bbb16f9084fd4ac22ad2dc8528c3f3f332684039e74b390
" ~x:"22fdd44afd372e15842413c0829c9a894ce61a3f0b135c1546f57fb0
" ~y:"0aa040bbb23c337d58874d95efe9277080862ea0888d9209ecc2f5d7e0d56b3e8444ca933800450f10b8124ff8812f87e1becf1a317ace0c3a1376d624938cab617bb546d0aad4f1d0aa23c6670cfae0da28660393a90911b3dbe3847eab4ebb7dd0504aeb0269126655d135d2e9149cd8ac5221151640914d480569b383e98364cc41cec56ea157ce8d7e73a949b348e5ffd3ceefea7f7625f599aa9afe2db4cf3b0d59f2700f6cecc54f8bf7853892f07337dbe76be781994ef4e14df2f0cf7cb342ee1c8b188a7dcc317a097c9f9e33ff89462c26465bb53eec05d1085fc6156cad0f7c9b80d2a68953501a97acb746ac3a2b9bdcf18dfceaa196716ec773
" ~k:"b93120b594e8994f533c1811d61495f2ebf32fde9e7ecec856033f20
" ~r:"84934f3f56d64815fc66b0dbf3b1fa56d1387be7611a1e571c405100
" ~s:"431f11346950e77c9e9ed0127c50bf620f6f69a699cd017c7d87368a

" ; case ~msg:"d4c5b439a1ccf5d98cf0b931f253f733037921d4efb02cf87b2509e732a56ccb49e0c83b1409cc009f1d2d1cb4c0c7ab00c402ee018ec5098031ac9e7197d4395d491721708a41ff5cda5a03be6a1169bf459470b1aaf53c8a9668acae1385b921f5a26c73365444515c3c126c6940b4bf57591a0bfd6c2c74c724426cb2ad3f
" ~x:"c9ed82462158cc9c99231fd48a81e4f8318a88735c35b9f2c08ad280
" ~y:"37c5f029816322da5161c4e20dc4f5abde9f04f5f9dff5d581b253109191b38424dde75febac32d6ce31b116063494a70c5c1d9d8b7351252ed377ea38fbe85b9f614eca1346bff65345d57e646bfb032e9befa9e6e5a89c16d715420e24129b6f70e4f681bc1d38ad1737db79655d244b4d67ad3d2bd80fd9d80c2e15240214859fdc0b6c43dd1e805dcdd2a5b9781397bd4a4e8bc4d6f9a1664036e90cac550e83d6641367613707d0de4f2dee55e9a5be6d3de893d61561f4ba90d387b7ab48801086016c842f3e0ce60e6b46aa980191cba147407aa4ccbe19b00b0ac71648d5296d13e48c75d52848bbd39f1ded988c3616faaf64f91a30742506316893
" ~k:"9f1fc151bcf8fe18bde1ac505737dc6868c34be605bf2ead6ae3294b
" ~r:"1b51b8d2d3eeb3d6218da3494714d0e88cd7366f387e6ede00f653e0
" ~s:"844203a81fb38f57505bf83bc8c1da002a39e81abbdd2f99ab6a4d65

" ; case ~msg:"40d4d9736b54993c1bcee7071c682390d34d47c35f177939ca5b70f457b3458fd5eca4cb03f0efe1aec10bf794b841216056a155dab58a3dbfc19ddf05d45861bae6eea2bd7ffb87a6fd0fd2394e847dc36c94c81561dee120779bbecbc32206327febaa17c96505ecb97d560c934c386f6f766a2f5154f545f22181c19fc698
" ~x:"5a050bfae63d347d64379ad01441b0ef9ab06ec5842c952f7a1c29ce
" ~y:"24aa1c7c6a041f6d2c533006cebcc2ad048b3dc08fa86282f5879a237231d230cd854aa10158cebb45f387923fada8c5f4b91a7bc2dc3e2c39463797e6eb1958abc9b9e748bbfe80e360233e96952279959a6b80619100f6f1876fadeb790491462f5917da36cea3793c44db90908cb9da18f696ced90f2acb826355104c4c8f06c737d48acf985d6b8c2abf31807282b6e651d2967a16907be3d8e4b7f32ed34eba8c262d6c0ecb131946d2546362c217ae195d05656a4fcfac73717ae85a571d811cbc99e0b3124bba767fead605266d99021cdd8cb4c081bef102431007ee12523b48bb838698a5971e517252d6d93e1c7fe9fbe07bf434164baaa1026da4
" ~k:"5de3d5e6b78c888ba4185c1547272fe562b44e507c871a0524765aea
" ~r:"325aa7b173cac96d5865aa50ea54e5df45a10e72fd5dd1fb265aae09
" ~s:"0a7203f6b8fbf668b8f6435e929fd52f52e23ad4b8a156ae5f3c9c47"
]

let sha384_n224_cases =
  (* [mod = L=2048, N=224, SHA-384] *)
  let domain = params
~p:"a6bb5333ce343c31c9b2c878ab91eef2fdea35c6db0e716762bfc0d436d87506e865a4d2c8cfbbd626ce8bfe64563ca5686cd8cf081490f02445b289087982495fb69976b10242d6d50fc23b4dbdb0bef78305d9a4d05d9eae65d87a893eaf397e04e39baa85a26c8ffbdef1233287b5f5b6ef6a90f27a69481a932ee47b18d5d27eb107ffb05025e646e8876b5cb567fec1dd35835d42082198531fafbe5ae280c575a1fb0e62e9b3ca37e197ad96d9dde1f33f2cec7d27deae261c83ee8e2002af7eb6e82f6a14796af037577a1032bbc709129caabd8addf870ae2d0595c8fdb37155748f0dea34b44d4f82ed58c2f5b1b8481662ac53473c693410082fbd
" ~q:"8c3ee5bd9a2aaf068bd5845bd55ecf27417055307577bbc3770ec68b
" ~g:"43b5a6b6d0bb962ec9766a377c32cc4124f1311188c2ecf95c0cd4a4fa097225b7618cb1276c474578d3bf564c145199c092a1b14baa929c2f3f0f36e0c2dae91eba08be30992a889f2952e0442c37af484a4ecdc3243ccfcb9e3413cf5cdd6630b09fe17efbfde14d8725493019b7b73d1f782b48ef30bec36e00e02ba336d2254fc202a69612cd9446f91d76b739ffa6d8b86052f8dc5f1145801c56241af5ba9037241bd89e6338b58e01310671c268eb5e33acb57d1f99f16440a675827d4017754d601a17ada2fbedf904554a90b01530da8c93cd14ce293cb2bd3e7937e934b79e310fe4d80c13f92f63381355bd80a1abee1a73fdfb6da24ef28002a3"
  in
  let case = case_of ~domain ~hash:Digestif.sha384 in

[ case ~msg:"df5d564db83592c1128be5d29b7036880d55e834a291a745ed8dcd438c4da6b1b9f39412b2c5110730db83c1ccdfe9059dd96ec7ea2bbcb34e3eba72ef0a1d4721c7c0221e29279f014d63facc5bc8f18c539b92ff2af89e568225d6b4cf599cb3dff5e3c6ddfac0a27f10f636ec220abb72630bae9a39c18fd3663e4651ccac
" ~x:"4efa5136eb6aa74e92bbfc913b0bfebb613db7a47221fb7b64f42e6f
" ~y:"647979b7960ce7b971ff0e5f6435f42a41b18c9de09a301114a013a7cd01183f176f88838379dcb4efb67daea79def3f042cbcf9cc503b4c2151a2364f7c9437b19643e67e24a36bac4a4cfa293deedf8ec6b154a32aa72985f7d8de235334b546c29def458c55d0c5c0ac5d74e2024ec7d4abc2fda516a2a0b1a4d886ad92c204707828a4fc7794f60ee8a4be1101c9e5518f7e19eebd475f2de6f6ba89c28bd129f13993befe5818440319a79549833196342a31dbaf7d79497dec65ee7dbef70e58f99d0595f6a711409ade3151d45563d53c1cd0a8ab1a18beff6502cbb0c069b114ea7be77898d0f4e549991ba0b368971b1072ece4afc380e9ae329a50
" ~k:"7e0f1ce21d185ae65c0a00395567ea9cf217462b58b9c89c4e5ff9cf
" ~r:"5ab43ede66a15688146d1f4cd7164702c0c4457bd4fddebac0482953
" ~s:"6c58e8ab27d28512c46063c96bf5bceb8fbad232d8f5b39c4755d0b1

" ; case ~msg:"ebeb9e2b692ec6c9afad2a0c2b908939943fdf4bb7438e3bd9288e7681984087ffdcf86502079c291236d7f1adb504e67e0f88bee61b61717014cf06b5fad5cb36f1b223b63912cdcd2b9416524d37f5d7b05c37d1789669e141aff6670db2e0de31673b2055f6799ac887937e5664a659ea0254a8d4ba6f204df2a38c2a77e4
" ~x:"1c84c5c065ff165a0e1d276c2ea9fdbf8423c12aa1c73844d6c64942
" ~y:"31d31a5bb82874bdc76cabae3ec85690aa5103cacbe5234e0d5ef645eef380d3ae2f6239144b82b101a7ef4744aadb8fc98e82b41372e99d6c905ca974b81c9fa521f920a1dffab4e2ee15f61e03b742f42470dc2fa9ab257f1136f9fe4b5aa2ece5207230c4906d67a156a3ffef470cbf3a65e3189b389ddc66c6040a7995c68ae1df2085941b5b1df7d957fbcf366824e0291df88eae55d8d3040d8d09f4f6ffee34ccbd1961852a5a62b26c8daaaa56a8ff7fa863b63c6d604fd3378262e815f55171dca35d04761fe3d9eddc6d32657a96d643d4608ef2143b19f1c9d8c00ed265471b245b60f31f8c7ed48dd6b18b5bec1a6ede145dea40283230724ec8
" ~k:"6f399d636570476f7a2013efdc74a1bb75f5b35ce835079c4e19cc4d
" ~r:"82c3747a0658df006a7a205a6ae2aedd5d2948488559fc3cfd643a64
" ~s:"8636796df622d13f070fbed4184c8138358c21db30c606b8f9be521a

" ; case ~msg:"dbd2516b03fdc58b32c0233080ffeea41c0d9c156b30332ec42be5e10584be3e3db85ffd5b5bae16fc876a0c9217627d84011223fab57d176def61e40d912e7eeb2bf868734ae8f276a96ab13de558ec42614167c5aa4c60357f71fac58980e579440f69968d2280bc970d0066b5bd6a6f5002481510256b3eb21bbb92ef2cdd
" ~x:"383585098edd867a8522dfad08997095aa23539b9c816a5e28359b51
" ~y:"6e6ee0319af8fafd7ae02013f4227e266244ae5d87fe156cefd4518bcd71aa73f9364bff35d4d23d45b0f47dfe93a607d9f8b399b424ba75072fdced6c3ed2110606fa48ed633faef2064fb336069eec7ebd8ae475978389e6e433d5a435d6529a66c489ce153940d2b1b8c886c8110d8b0aeb641a40e285d6751ce71027c30ec62f4b1fc14f4da20b1d505742cada201cea81930c381f8a6f13dd0a42aac1e0bd7fcd19c6bdd170fac6a423767b831c1e289e0a29ef85d817ad238d91ac3ace2f40a163b0a9bbddc6f05d0bdcd8cc274a74d0743c9fb56556ec1cb8e9cba982c15a9a66fa6b6999b8485db1a86ee18be16e068e12a8a165e3599df96669a1b7
" ~k:"0183d11f1597ec9db32db21c1e910fa2be2f276f35d0583ce8b8f6ab
" ~r:"040405136a1220adbb64ab751db3307fafad5447ab2d9bcc52f79be3
" ~s:"1d35f3269c77c577243f1db8dfdbc4cc4531574276f0da1f7a44acd4

" ; case ~msg:"34c45435d0cc29269272a93d43320698e454a7c287db9d062092acacd7ca086455e583baee1276caba068fdeeb52183396d5444c5a14ad52a5c2bc082cd87452aa8f9b23056b5f8af2638d965ef4fe6e4e68e88b0f50e01248fe6a6a1d9d6d93b03cd55d16fd83cd4e06763d926f7c50f20f0ed6730613f0f4db571e22d288e4
" ~x:"0f115fc7073262e2f93a9d46b407b0f1bc29292aa09cd1a98a34a219
" ~y:"5ebd8152935ff2a3f9a61b275e9808a041aad5650f593f612af33bc462b8c994169372e8f80f51b15f5ce966ea3e76a912c653978337e962219e323b6e922dea4bcc23c646a22eecde02433126fbace0e3a01fa6d0b9fdea9245d67899a7b745b8847c8087fa7f6c0f3edafab4c3b47220821fe46f1bcb00a323dff3dee47ee1de2ece44e1fdf3e64aa20c9e6b58e534482e7313dace1c617d8ea9a65dd51fd33024f735c3844c5c6b4a3f447e714ab0c17dc88e33f08b142b72e811e6da00299c82898aaf2bed5ae5170c1dd005678d2b576b9ce3e6bc6b2aeb04c9f04e444e2a9808405ff5926548b59304dddca8972631f7fb136808e213ecd93af98e2e54
" ~k:"835a744aa418a297b7e11febe7f3bba590752e58fa1ae12ffa3bfacc
" ~r:"66481f241f6b443148f0b1f2459be5ca16413d947d0981628717c108
" ~s:"2cdaa73500d0ad291252d07ceff9cfeab87a739752291eb5dcefea87

" ; case ~msg:"d7ac5cc8a4c3f38cfe5c0e1068ea28f0f95d3250d1aeae5f66bdc4d22e23e246ff30429cbcbad3b02a62a0a179d4d107130fa3a780c0092c329c2b026e12e6735a75c495b097aa69ebe98a96ff891234ff379511149e07c6e2411e58976ee93fba7d3d570c911f6f208375783ff5d947a3af0c839d210a8e4a8c8fa41efbc57e
" ~x:"5339ec1f86a0dfd81324fca6a0d3e102b12fba8fe8c1bca45d8ddf10
" ~y:"7b5fb022b55fb61f8ef8cdbfee46c0fc61e59fc62dee5c14d0c3134b4f2659112e3f4e7017f9574a2724188ba6a1ce777a8915bc1171d738754b5ac1df923103ad7b198511ed36272668ae0c2e3142ba011cb45f893ddbf7b38625818cba9a9b78aef8d06007ed505e6dd6e20c92d2500234f104c1283f7c00cf2a3a32458d97f7bd17090f76235c6c4f8ae194d52d67c74a854973fd124751f7f5804b67879b023bb6eeac76e96fe676daebbcb1bc94d5d851d7bc56bfb3d2a0a6d992313786d9fb38ad29b762349451d149d0e5fde6ad497183e352828e251bcc7c3a918be4d03b17af60f3f3ef6d9fb2455df7e8b6b169475e5f89db9908541b567d0f299b
" ~k:"7c62eb8fd725a453fdb2d1e75bbe22f0c5d27a5835135c788061ddfb
" ~r:"5b6be6bad725afa442f29ab7d343d2f8b4b4941cbd23d69164b3c5fd
" ~s:"3a1b94634e313fc4df8292e038c6e876336cef88d691b894c0eccd3f

" ; case ~msg:"7a96873f0777e8ada9867532ae5f51938bae2d56fb471e0fefa693b71a2aea2571c0108ba59e634401bbaf20a848ad8c305848420cee654a3040007f055d4e975807894b5618b9392363bc7f8c88d526bc491adbd892a93751a21d137ceede8a04423a4d0ca1557bcf334e4f855b04474544212929a81dc71fb3fc41f70d6b18
" ~x:"494b68624728aaae9898c3ca22c1bce810a052e25c881a185af43cd1
" ~y:"0531518177087dff8d04a0666c1301a9b38427c2ea1b162e6fca520181ef22a2d205ceffffb1549c9707805560c6c4b31943d52556bf301c5e0e75924fbe6b5c362fc9801753e630433a9a348f53e62c0746b26e348dfb85853d1ef6eca02cf3f343e77c1769ffc1c109b88ecea16ab6cf476e54312500983622df41e695ec27a41ca7a63121ba97bee7b0e9d547bf420f647d0f8671bf4107a712a7dbc1af3aa8d15b98548d3909f72b9a27f81c46e3defa95eaff7590c626b9ba10974ae8b9f58535d09ca30f9f523539cf584f9bc6c74185c2ff12504f5598ffde6f86021ae514562fed3881197fca22db5590fcf9522ef760ed0e3631a6bd79f29000b42b
" ~k:"065a3ebed489d78ad676afb5373c7028f843816fa97c30169149897f
" ~r:"76bd6ff4cdc4fe37f6705e77efdcac6fbb9d54fc0b220643c662acbf
" ~s:"8a124a3640ad73280f305afc2bc3e57f7a2e074081be7bc90b5b1faa

" ; case ~msg:"d69694bf9a93ac0cc3915973d40e351247c3bcaca98069cd9c1e7a3c5850636a592ea75fae7bfd38b1290e3f4d0aae8ee689ce4137ea868aaebb17dafb255c4a20e0fac1f4666612f90c46320a62002ede3167a34dff74a306a0842427cb9d2c61599b05c67b673144f6c08232d771f2e0af38253f36e122870e04ebc54a512f
" ~x:"044b1bcb76db64ab7500741f43989d3d878991788947b679bf22c088
" ~y:"9c588b76269b2f087f7e7af4ec4c0ef263e9636f45e73e604502d62fae90a25101bc2bad2a002127d4b60f5c4a1388880cade9463ab5f7997d54a02c24e7d51a4b8a7d91cdf6afca2b433768094533a0de08dec1f19eccb46df1800f53d3dfeefbfb769a80e1686e8d53c60e8c1511a6dd4f42a155bd85f75740bcbb7b1127591822926d1682982375ea5ec29fd1ef4f283b94e02423a830b35e973caf12377ee18d2c6ee7771184d7a94e7a0c4a01044afc4efb2ffecb695e233aeb80c516c77d1c730d30d1aa4f39da51bcc48f44d07abfbe75f228abec2e7273593c98f323a9b003562a168752e837a1232f462a23d3b185ea8a05361570455aadd1037063
" ~k:"4707e611f7d2dbb66f5ff083bab786a525884b49390213300b088fde
" ~r:"108a082d2bf6358a737465624320c4fa9d3719744c2db69d18963d75
" ~s:"420f3537fa6858657db7a21e72e11ec0ec8cc85a09a0d1a445944980

" ; case ~msg:"17455bfbb128df0f96544bbf83ca0ff374bc086b2de18f74f59049f73eff3c8ef32a48429a4038256304636f3032192795ba2807407ef52b8d59b40bfd517583f998810279c0211771d9e54f2b84e898f9892ef77beba33ff31a2868693f1f0978b89895e350d5ded259fb1397e9c6989986452a0d77df99048fff84b6eb150e
" ~x:"2bca3c613be53a6aab121de91db4fa06b468fc6550c82eeec4bce9b1
" ~y:"850c0fcac073c56318a92104654e6a8ae7678fc4014728304649bf1070277706fbd32ea4d41f77f80a80c88f2701e3665be73f59f914a915d66b411bb05ae5c18b00bc216251399732fdc2a68be6a21b3b088797416ae05ce876b6802e4f941a21b1c661e3f06d501ef2a17659f088d2195dd161f06404487a27b79df1ec574ac3abc30ece2a1428c5e0c1d4c49803398d0714cacd9853854b08746fa453561545e6f0d96cd2c7ce1b89bcace1c697ec4d616bf14d1889a79a806a3699f84f19efe690fa13a3b4383ebf77261400fcbe309c2e5eab0b24b197cb856aa27d7d71d92d32aab656faec5ff792ece53874c4069f540d948f8b2e5599082e21f02d72
" ~k:"4b528d2b2bdfa4f2fce09dc9806ed5302e41cc52f35962653d7f222c
" ~r:"423de9e112ec38e3a034f5d9675c76f9dc8536b30d05678a2963ec16
" ~s:"74051e79699fa44de18e36ab116873593a310e4e09dce18b833fc2f5

" ; case ~msg:"de1f9606261ff82218c8c145aa4d5847673b459eb55fe7e6454c0443266bbf800c1d09051f5e3141c4370d1b990cf5fea9d2683986c3bdd2823107829ace6ed7034caeb2f657a07b25b7d60240a0205026c2e3018141d479c07787a14e702622f8e6df709b636c6d3d0b5fd54f5516dbad97038e5c0eb31f54db1264d600efc6
" ~x:"366a49173a1783b99550d84c7fa02b6cccab12ee9a306bed7bb81ba7
" ~y:"4d6e89b022c278f3bf8932e706e418ecb20c1bbab13ea8c90b6bd84384f38b311e8fb2c4c0a94ba7d3afca1ba94252a4c1ac1187622cd9c16aa73bb1b4a5cf55b5aa34bd93526f187beeb11700e4afb88c816eda50a50e81860c87fa66a1b63f5ffec3c3ae39bdc009d38fa13da863ca5ec134a7ffcf5dc3ca85cc34d61c5df8f9d9bdbe6a541045b45cb512ef64d1ad3db7b37dba33c6e3c96180cfb26f48c63373a0f0003ae6582679da4850ad2a0b899e0e8a1847df07fef3a4330a72f8a802c06e8e95707e0c7dc1915f6e1731fe650f1ae352e782d2dd77f54e5dac52539a10a22bbc2eea31efb94438a030c4b2451bbff6901b5fb3016cd162af6bf0fb
" ~k:"13894dda6721bf3af8a40603a3d97af240976a8ecb3ead998eee0ff0
" ~r:"5f3839eb663f026f792912d1cb0b448f5e2e593139001e839f71c942
" ~s:"6b07edb6a034d084a61bf3c0a36e7ee6911948ad8f6e50ac6844b1f3

" ; case ~msg:"c1edd86151af66c6223e413f17e734b2bc024ff066578c55308f1388a91ab87270cd25ca2efbc2867eb715ebed6d10012b6f4808f2de1986ff7f4c369daf46c80a618707888ae3f86e38e7f25d6caa509104d4851cbeefbb75692aad499a33aa35b11409300e495fe007524b4af2c20d33f1c8c04516b6973ac1e07df3f160dd
" ~x:"841ba91e273f1c57847ad336cea47c643335e68f611482a30d6c0bb7
" ~y:"90dbbe4741a76a5ff222ddc833c0e2dd445ad01726bbea25cac247f9ef9da643932736db07cd9aeffeb45119351e00332d9dfc89f5903a541e74e2e9709d0f852ad65240d06159fe54436dd8201f8c56926e8d23c2ecadeb8cbc9aebf12d52be6489e0acb0e7526fba3754b7ec163dc7e2fa9193319124f0cbb61c2ab7ab1a28c14e7d581dfb8de23f53364d204190a58fcb9ea5b6f61a7979b86bb7a7a4263a1066f0516e5870de423a7e3b906d90313d1ff9322450f72ddda4733ac74fca5d4ad2be22c2667b92212069446b42a391233d85216a88c25b76c947d8d56591003df2532fcd7b18f923ed482d464fb76f2c85617840d370ab99e320e88cf9ef8d
" ~k:"5ed84fb90761dc03a5e60f3b396d6cc7f8c16c77f065a6ec0049fa51
" ~r:"836d84d86271e1648466d1955c2b60b2a04cc021405083626347aef9
" ~s:"63c7eeb5e06e81d8923356f799810a26af67c0faa18b392258e4a9a0

" ; case ~msg:"2b5fb613598c02916bf6b4b0fd7a6b5426ac5b56954392fba32de00bdf4b70953be196ad51ff2c097a81e6ce1d17cf837d2444752be92bd4a9d1a8b41327527ff6bdc0e5c3e0cf46f7e37966aae18a29ce1981f212d714dd6c0cbb410d3a5f3d006ba9b593da150ce422b5cc420f3b561bfdf11dcb9910005709eeb129e20665
" ~x:"220947396c2de85d480bae730298df67283d0d0694950f5efa4ea5d6
" ~y:"95947fbc50d5a80299c90dd27cf3910091420d8af849240ebb541a21b49e528b0f3317acc10493d50e6bce676c433c31147f81286789e6a41f4b2603bac0f6e5ee7affdb44cceb42864358607d45f4655a709d7d67f716d7367bb5eab334f61cef3720c080cab17512329e6d99925b47e4960c85031bfddb13f0c61af80ea46b7b8702f8ad348d57d481efe821054fc83b5266782756a42dd431881ea6cfeb7f7900d8f747aac9976be8945952afb8a274dad03428088310a2456ec254d1ccfb63eedea5d374ed8cc637a7baabf8f422e1a12d5ff316dff8a082068931490a4706503d19f93554f25243751dfe62cd87cb856f644fbb6fc46fb9cf89af5aea1a
" ~k:"2697349761cc4ccbdb4550bb9ca73654280ade31f577ef86100ff4cf
" ~r:"7b455fae1002fa87f36cf6f345716225d4aa1407802af4082bfbb14a
" ~s:"235d8be4ceb0176f5d0c47c1199afc7e3041c7d7508b9feddcaa0d74

" ; case ~msg:"bd7d69bcc2e4f8a42e627fa21c7fa9fdd3e574b6dc5ad20217e80bcc9997b4c5efb31c7b65dbe8a0a394f0af580387b9917888152dc4f63ce52d3ec4b723bfea8114825f9f1e259f67b5d13bcaa66c97de725fae4ad247bb922497ebed0f092bbac12f2cbd9b71b229087378e8be0626b8d5e8950b0a6e69e05129f0d3842d27
" ~x:"42777374114519bf323bd03b6e0ec238660dc863b1a3b85e0cf8f8a5
" ~y:"6fa6dedc84a1479be43906f2f68df0e93234ca2230c832db079d9cbd9342b2df13de4bff10bdd831313453b33b725cd616acf1fe2f7927ea32d46ff10ef1154e503f71165adeaffdd500a83bf1001ed36ca65bb6974d0372cb0f2118278466fe1286adff3c7ef719c2a02cff9ed9374fbbe6051814d26848b7d970fbecfbbffedf40a03083fe33d3067838ace22854a8e88bfcb02ecd76c378bb5c8babd22dfbe090753abf9e97cb6ba708ce00ffea5c550b09f24930698df115c020b1301d571a470e5a8a6ccfc74ad18949a57f614fcb0f7e8bf7530a731bb6091a7301af42899d9ee9e45aa62ca4903e66733e47d01e26b299746da75c7a57dc00bceb4d6c
" ~k:"3ad0d788fbfaf4caef4beec9c1566a8c7a1de26bf75dba82a8243270
" ~r:"16a2a48578a0b5b57553cd20005b7e8400e1061c4fef20d033f72f8a
" ~s:"6c34d176e95dd49271ee48a3802edf4238401084bc3930201405693a

" ; case ~msg:"7766e1ab7638bcda3e6fdbd4c85b3661acb2763d411376b2eedb4b2c6bff5d8fa20c0ae5b3cbed20796a6d8b81a1096dc36a39826a18ffb897d36bfb16363cca7632ecb71d2f996cf7cac66669bf4c83114bd53be3be3305efc99d22769188f84289cb1d11501f040b85d15890d29af2c8eae614f74beeeeb5fc915afa4322c2
" ~x:"364bdce93df0eaad45ee0ef5c18828bfe2e381db607e5b6a77ffc6e9
" ~y:"4c2b559024f1b3ff5c7167270cd1f33bbf0f40b9efa25e137441ab4698154e74da3cad34236da4bd1c57d7638e4277278b508e85e3a98d30388ab8638f553e2a700011923e5d154f8c1407452dc4f80770c9c31c368a21e499d5dfb6f05fd67791e761a494200710af8c2188892c2d1c3195be4a0a1d67551ad466fee80d7edc435379a72c3bffad271de31ad2ed107d784f40e24c5a6e8d5aae8f2405964fe3c28cc3652dc3c9523b39d4b083ee65e9a07ce897a17b02b354766f1b19c2b1229ab468b0148ca8fe89484b7b360024218086af56403707bec65c52281cb8aa5346cb6f6481430e8e057146f390607c572b5bd8426b90ef3a827cb0d58bd438d1
" ~k:"576f8454ff45df954d123bd1384cbe004413c8f85493ed7d6425bfaa
" ~r:"09c61878a9917177058e9dff27106bdca7d06c500e09099306668cbf
" ~s:"7b8b6c4c5615976d7a735ac3e184cde96154ffc87b458924d4602895

" ; case ~msg:"84095278f7f1d578e798399af0bc9f4695f9302ea5972479adf90c95fc25d59e576d97b89b73dec629cef05d6173b55d015a3fb1d8191ae540d552409b03a7a8db511bad0951896db949fcc28870f9d17314734ca6a3472683d02fdc8defa7b9d3762ae9357ca2a6ab623b046350fa211d5213787127d2711cbd91405abbe50d
" ~x:"161fff26a7b9d7ddc15237edbab3c1f99b7294c70feb96f962df8973
" ~y:"4b52c56fc64922ac04ee7a80fc5c224013e2ffdaa167381257e00c597b433641ceadbc9b16568bbc9c6d31d02c8e36db2e3987520ce8590856bd4a841b725ec95a4659a61a0086f66a6bfdbf1e4bf92b441928cf319f929a6428f5e3ba7c89123dbb0cacc16bb0e2b80854b0f60dfaa99f9c4caa412c443a073b7a51259125f012d98f0f6699d70ade66df9c5e18185672e0e2830e0585413da2956c89d2320faac03aaa83fe718a0d6cf7feb38a194e4362d7c89e4a13967e3a2d4493f4ec09ac2fc89d56a595472e60332448548d91cd6aac84a2f9b4d7a80462dc154779be5f9e1f709b9d9a156273033fe6e4842ec47521964d2e2fe262280fddec6403e8
" ~k:"7cbe0c1c29b955fa1fdafcab79c02177c15ec5789a4dd53a6ad29ce8
" ~r:"0c4d4527815a94bc2d77063ea69049be6a2b3b3a3a0badd5e62a8f9a
" ~s:"5787ced7081fad3fe19ab5b9028e9e8df18639e4991ab6e1e243416e

" ; case ~msg:"30eedc9d630b632082c196b969d24f6eb9cf1b1e2c53d244e8d8b50a40982ab53c4d57ff995fa8458908a743890382da6513cfe9c1991824873615a8a16374a5e5dc2fab3f5cd25652ec8aa3939f4884f74ac737989b6ac2e43f45b885206a31e797fd8576357e4b4baa566291815dac2f546f4abf8ba1de1120fd804284e959
" ~x:"0209c00edad10594f7cd7878472169d512a7e8dc3fc1cd69285e69d5
" ~y:"8920f6ab95b1dc6b93e08ead6b08141cc2a8f1ffbb71d5ec5964f6b2c3d72ff3adade52254370f130990b43487775c2fe017a8200d8119818a15ed7e5636bfbf3164042f27bb1ea418698b6756f75a8fdaebf0f6e5423e460287f4fdd2a0ef305e658741373d3baecce79063962f883398c314e36230ba8c570e667c30cac8fbaa4e70202a9157d22708ca605403066d0fc84845bce9b8c3b41ec32f40c845a532fdff4dd10cf62a714121ea8a6188500645afa9316fb3e11628b163d35d8cfcc55272b650e8072c237645600150bbb66d393c1c97345d5820f178dd405b5d46fc4ac8a5f3929e6b1627944093178a8d65101059fbbbb7081174f2308b2653ce
" ~k:"36454e085b6b3dcc7c755b65ff46697b099485abd6ceb00cbf5dceed
" ~r:"45212d1c8c128002fcb3ce35583ff8d08363711c1598307d9ec6a108
" ~s:"4858105649db5992764dd32b102d9b9d2bc6af64c6a81595611e3e20" ]

let sha512_n224_cases =
  (* [mod = L=2048, N=224, SHA-512] *)
  let domain = params
~p:"bfebd000b2d6cd4ab38efba35df334df721d6c2f2b3d956679cbad009f3dfbd002952cc899cc2356ec8769bd3d1ba5a73023729888da92ca48a5ee94c97f4f04a2e3acb4f33a2f0fb3783c31f2c70fa7c70f38214a27dadec8b12e67996a9e85ee3bb148803130147392dc5253c04d7063535e6cd646bfb186984e08b58b74a7be5b333bf32b0abfd5665360e9a923a0c528ff1c62c7253458f5678528719d436e50148741f45dc7dd2c6cac71c55231f12a83fefd2ed0a33ede1b8a51f566fcf7890682cdc1931dc207c92bf2ef4e28ab31661eeb77f1601eea941c9591f038d3f00d912857db05e64b2ad569320061c6f863ff3354d842e7e7ea715afef8d1
" ~q:"aa986df8a064278e9363316a9830bcfa490656faa6d5daa817d87949
" ~g:"8195ad9a478fd985216ee58368366d2edd13c12b3d62239169fa042d91156408b483122f44ed6236b8308a6cdb52f9af3de88ec89e039afad7da3aa66c1976049a8e0a7d18d567baf99fcefe315cada01548386b10b25e52f52ed78eb4d28082e5e1ffee9480c4fe2cc4aafd1efc9d4fd2cc6d155968931271ef15b3240e7fb043a80c8f628befe09d645077c1029d21e0ac8bf0ba9c27714d1b580ede594aa01b3b76f6e745fc1ec07db37e2fd7e98c6c8c6915228e422c309de9f5db168f50249d1be1ed3298090808e2ebb896bb79b8c4cbf94d4c2064e37e612ba4449d7ac210edde211416d64b051dd8046ab041732665411a7f154d31b3e11a51da7fc0"
  in
  let case = case_of ~domain ~hash:Digestif.sha512 in


[ case ~msg:"e9f59c6a5cbe8f5b0cf75008d06a076a6739bdddb39b82143cd03939aa4738a287c2a6f31829bbe15f02cc2ee7d7122dbd132825970daddd8a4d851da86e7edc8940cb1188319218b8e0248a103eae34bc68d85f5a32830d7e5dc7718f74db5e4224c0debe1e841e1eea1a88fee0f85d9fb087cbcee55f86037a646e38346d2b
" ~x:"6a5b4ffc44238d1852fb9b74e4c1661be85984043cfeee023f57cac6
" ~y:"af6721bf75dec6a1b76ad35ca3750def31117c5b441c15a306835a1db74c003b86ae9099ebfb745b0aa9cb000cf43fb021513b8f197bc865b22bf949b491809ad752ffc1ca8e54bea16dc7f539e4c55fb70a7743dd28f262f60ef0f2fcaac29e8021a7938c18ffe03075d0b7e0a2b4dcabe46ed1953d33e37f113af519ab0bf0b6186c12b5f6488437f5193096e2fd6a6a1835604794c66b42ae5265c1cf1cb53ae84997975e0318a93ce41e3902e4ef54de3c56555bd19491acd53f3e57464e1f460389dbc5fa80648fa5a5a0f2956e9ec3b8dc441b535c641c362eed770da828649bfd146472b0f46a4c064e459f88bff90dede7ec56177a9a71d167948712
" ~k:"9ced89ea5050982222830efef26e7394f5ab7d837d4549962d285fae
" ~r:"9da9966500de9d3b6b7f441ca550233fc450944bc507e01cd4acb030
" ~s:"2d72f1f6681e867f7d8beaebeba4bc5b23287604a64cfee1c164595a

" ; case ~msg:"971d16d111c96de0f7098b256af213f4475aef31007e12e2974c5f64b2f335e0183c196c33d50f6445c5f614649549770b1874dd0756a9a8e39971dfecc3f267ebcc1f5301703f88743b0f376482cfc06d5948bd7926d96ec4d731a44b0c0eee5e85da26687265de5a66cb1a73a7e4f3236f60647bee5c163340e19505577cf6
" ~x:"9053ec8ab1f9700c2ab59259bf2e07892904f03c844cd58a7ff59c79
" ~y:"290517297e4249fc3212bad67269e032818d760b0ee0525dc5a17c97116ee29eb3b450b41d15cea405d5e983a8558184067f424acc498676415e17506a351c124b5404f1d17153272619df713ce34d03f1f9ee28592f22f829a31993b106c785fa6dbe57d0049c815db5ee2dfe948ddedd1a5e2cd2346cf2f66f04fbad619cd983a1b069b471ef9adb4df6ceaea23d09f0a548c3c7209634c8a05e5897445906dea08a52e4074be22d8485f20eaaeadbaab397199b067aa860056991ee088480b4921267a698a8f7a03777f56bac84e50903e88d07261f24d0a4f317128e01fe8a9224f12293949cb6c3f095afd19aecb16b209a99487dcc2a1b83c49d75e351
" ~k:"901632e0b8ffea7efebe2fc9ea0d1a52442817fe1e1b5455bd39a687
" ~r:"1f44f6eac218236a1d99cf7625abcf5c964b0a0c5d88b8d05d74a3c0
" ~s:"71015cbe8622d2a34fbb5e7cca8c59e828adee71f50524482d9e7904

" ; case ~msg:"08ea09fa5efde215bd8b3c4d6a9c90ee9387ffb7bd65becdb88b40132c6384106aa619b7c66ca92034d284608593864ce6b92877112aa139240cb44b388fe68a8fe0501ca584f6a2de27c0fb658e72bb13fddb8d039a6bf85d63a6c073b2668013ce8fe589a0150e46d5b1d9b0cbb5a14c100ae4b20d6ce81a987a50a949f434
" ~x:"a2cdf2515cb098559fa13cb70b6a897e89df120a971064bb377988ee
" ~y:"b3e2b7e0641721d69616679596cc75091fade2da0558e310b8d14db0f4686f1fed48d0fb7f0b3b27bf6e1981eafa7737a3e651828d1fcbf88387d06f78404a7afaeaaf8fae1893bea3a09a118893937ae2a8fdef3320942a158463de4fddc11987f23fee9633e06ac239c06610bc45319abafe517ce4aeae6247ea789d7da60d3eeddfdc4b232b4d7a069bcc0eac7b99fc088fb7ec1946034a98d7e69cab0cb2b06b3d9deacd1b433ebe94f547a322895cca9b0ed319b1d458c3bfb260beb641a5345dbe3d01ce800ec2c6bd430ce3e3f5f78fcabf91a29658661c573b9f6fd3812e560d888b6cdf3d57673c1630e00ca841ee994958b250dafbc3e83bcb8be5
" ~k:"077b3adce42ba0622772eaaa8cabd16107c92f7a134c715a4dda5ebd
" ~r:"6c03637d253a8dcd0907d6de93926bdb3e1ea3135a709da2309a8da6
" ~s:"236e5163f2c2ebe0eccdbd3351e4285531a4f53e45284e41db37e266

" ; case ~msg:"957cef163b16d8073d5d3fe158fa0c7338bd107c6a653cb0f11ebe41402607b822abe30e36ca9ee4c9de00cf72db97f57d78f3db49a8e1093285563c68b0f4e124830b9febfa3e75ce2ea59cba2cc6d71e908b5e6d8f463954922b82bb55a69fb2ff143ffcae6b5656143c8b6cc24f57b17cfb020f6e15bdc5f25436d07b7f8a
" ~x:"15ea86b973ef146f03cc701b17b589b0ffdd318b64827d49ee3c0044
" ~y:"3fcb8e44d6880f9eebaedfb75994605c9ec001f0595aeb5f2bcaf6b3987bc28a7ca905e1fed7e3c715401b5c608d12076938a18013473d8a433277fd9ce5a5cae038281e768ff909aebe4d257dcb5d93488022d07d4c2862afb2bf8a2b1e974a8e7b6e176b1b0b7ad6f63bda1b7142e46f504dcccca7d1e2e7662758f760e624e59528c5a0c9563ed517c691fba2abf66899241178223ba20013ed0ab21f91f3e6bef755c8100c51ee947b7a9ba38570f880b5e42f24b72d5321132e031b985a0db825bf3bb00a7771a03007387e03ce020fc358e65ed3de8d847f5be60720917c0616a450aa341ae00abe0a809c38e97314f303fe9b0c6cde446d0217cc4eab
" ~k:"9af96c995f0b7b8283e2ea288e3c3a6f751a56b38041297e2bc34cd7
" ~r:"150362da792701694e23f0b0a9b7035437cc8f4faa45c6df8f7982fb
" ~s:"6df4321c61738743a9fe78ec76b4952692aaa372d1c8530fba0fcdec

" ; case ~msg:"204d9cde24a2f0de02aff020f6363fd68f70420dc1a9b5138216201363f832da0aa801865a75a243427d9d6c78dc5e6041b27d033660e1e405abe1be27c909994bd6fb57180c3d6b498ce8793bee8ecf51e06b96411d00996209f44a380926c7b195e84e78f01fe02e0bc7032ca462a5182683475222f9dd8f3ade1ab8fea318
" ~x:"524a63cc5acada8557609a5f0d88fd3e9c6e63719704cd8bab8fe301
" ~y:"99b8fc6e64cce262ed741c30cd586986aa2e8f6371b848a2617c619897de23726bb54536ece4b460cc7f1f39e0c184eb19291e930dc9140e4b7735541eeef8ca8ebc81790fed37a5f08e9da9abc66a3a2e909902a4212106927d08abec01f27c6056b6e0381150bd742d409f6810fa5818ffcb3f182adf894ba7f80678ce883c1089a6ae71db3a115c386dd9153f4191fc365461ac86838ecf2f3f81ccf283297a6fbc644f52aae664901ae30c96fe4df930cf1a41757241cc4d9adfccdd9a6bd5004b05757443598856400dd771dc089095c7dcde82f721f986af636638eea2c71770856c2ba80315e8696142a11e51ebd7559e9da6a00be3f9f38c614ef207
" ~k:"028091483753f5643b61e4093a7e0a5135d71c5fa318d6e8bb0efc66
" ~r:"9c023331751c79d5da355bb58e2bbe2e973e3e4b4f52743ce1f1eec2
" ~s:"96ad0e8ca90627fb7ac4540c9b58a016ee6c4e0a6f0aa1e7def81a51

" ; case ~msg:"1e4e58afb34c5d6f645a82645be358a2e228cc7b9c23dd7f3aa79595814d054b923b9cbc6c9e6c6f94848c1a4d215679023a96976a44e9b59136241fdf26f8f71fe5a9bf366e4912b5931e1c8f63c37fae2bf1d55ba3943a650bb463cded9a7b062ae55aa57d9c5ceed323fd9a7555e48b834d3ad4441c35d9e07c7c6e4d5d0f
" ~x:"33b25c6bbbf816addad05e48b72ca560c5191214d903a978b6708a30
" ~y:"b4dea0d5b671cc815382d0ec6dce661c30ff93719dc7f56e7e61df6eb6a3207a05617938c874bc3ab093bcdbbc983a4b0b587d60fdeb7b87f7b0be4a656883f5443ca7864541ccbfe0d0835636ef08a936b2321a51503be1eec5f7bccd0c73c9cd52397cc214318b30e8be1eab57200a4d4df78af991bde183e0164e694d8308b7d20d067bfcabdcb50f7a2c190c66ce3dd0e18960939cb57fc3a2e5a604f3d9bd6fa440d54e9cc0383958a0d6aa2ab670970f9b2caf866ee507067343f7513e0a981f3a344f2f753af44fda26d661796032bda0f6cc30a9a789db8d3d546f02f898116805180c6f0d2f5388ab5110a521077d88d214fbb32eed2664406cde9b
" ~k:"989d87703853c4133b6d273686bf672492e90ce2a91b3c72a4188a1c
" ~r:"0434ef1c127207d0c884701e75d801725c451ce67d2e71534638b231
" ~s:"0c625e4a334db07825a46b55da9c2e8a5f600a36b71606834097e777

" ; case ~msg:"5a470a38b2ebbead08e010efef7461f6f859257d91a61e2f0ba809e28c0ea3d410e4f41477a398d593df58039c4336260ea7d8e98c9d7daad0c31ecd1567c7db730179e2a9a62007bd56f9d9da48deaa657ac92293e5bfafbdebad1afe25c41e1aa09db61fcc191971c37549155b3e67956913aae3a5f6245cfcb9aad5dc1e15
" ~x:"13411c1a6fe0063e7f9b2467ccebf2be5cf30e742f9a35d715558ba7
" ~y:"06a20d5571296eeb87e79eb274036d819e8623b15de44c2697dadecab2996f51a75aa088490e683f34d5e0e71d9fb8734bcfb71e9d19cbda3caca5cec417fa37a06142bfc0682de56f0dce6e826ee9f30d01279859d3ffbd4433bf4a1057ba0ad75060d41f968f6da822c33cbda9f772c2b77bc1b29305cb697182c0d39b132868932c64016bc9071b30920eb385c5ae41c5d4f631bf5f54b1eb4b373bb3e0bf6e448ad8c988fea16e643790307b8b85f009fb67317217d9148c6cd7a46136eece1950a119e5a416a197e00d0e929b04a5bbf6c988d8595a0b2a5ca71926ba351a5f7674af4183b5a68979bedd6491295b0f172e7373eca7e62d78d744fdccec
" ~k:"7406254d3cfe3d55267236ff63b0f42b2e3b55d1cee7ed1ca3f06ce0
" ~r:"74dddfa35b25d0c0b285a5d21719ee39d6e3f443445ceb90556b0186
" ~s:"474865d3ef07f5df49e0a6ebfb5ab5c2ede47c4c6314be4ccf455e21

" ; case ~msg:"0849d67ead3e8c44ad3b2f949be1cd9f9a4bf8b5785bd00ca66038e9a8b93727a652a415c1d8a1ecfcad77782d87d912623c2fef45b2083ec0f79a264ef7c5bfb76fde5b22b9845392e759a1ec05fa6387ccd2943ef1277c2e060337f82aa562cee5bd7c158258f2e779d51e47e000a7b0706077490976a07763e2efb275b5bf
" ~x:"5631c7dfd3f5adc0b7b542a8d121a07bb8251b6a1bf3a8cba771c724
" ~y:"b1c61442d8aedae0a04daef7b6f8a49c6d07bd958e8ec561906ddf31f3b4ffd481da5443fe8788056c4ea7b5dfa2cee6474e3fdc83fc043a2bba333d503a2a938865ec3f118640e8457c7d974e2a65659cef5b7ae4f49a054d94ae5e2eb6345f5bdaf92148beecc109c55031fccd90cef88213b69ddb754b40ca8d8f0a4bfc81a287637a38c21807f727a67025ff67b7fcc54418adad408a5c7d1ce05a1de7e30988d560e779fdea1b78753314b0b80fdacb6246faa4b4c4ee8acc5ae24b82312040134cd8cc2fd4fcb191fe43f64d140624a8c6c2ac5fa4bfdba5d625d7d21e3c3f6acd8a153a04fb22f8d3b244ae8c6a1dd0e6e3b2f73c064ffabfad6cc461
" ~k:"9c353ace0ae52501bbb98a2d1c9e28f3a833c2b0eaca49cd12c57ec7
" ~r:"2aeb7fce1b7764d32cfb7d85254ceed9f3a6337ee8dab42c8ab7a415
" ~s:"17cce13bcb917cdbefe0c566318fc974204b700c5cddc5b2b499a78e

" ; case ~msg:"e74639f2bad42fd6393f9b350d6e19cd4c1ce0f41e8c902684ef6f86790ffc8311acd9b57d6521e80339b3243f6ec6b01a06ea899fd75da91e1080fdf06129dd851a895d74b1efb9837289c11d68e1308c47bb8c59d5eb895db53bba29102a5b48b1e75c73387ff22e6c0461196a7d48615ffdb9c8ff4ec6587b4f68d260ad86
" ~x:"7319bdf79a4c8dbc115e3780c818f6e2a3243ab47263e84ba259bd3d
" ~y:"9e1b77243aba0886f9baeca6c11bd2c5c55547cc502e731d9c4725da8777ab6050e3399e25577704cfc66163f6df8d749142a7e974e49b7315ab7c8b85ad5d5cb271cf207eb72e1c3476b0d863721c967be15ecbfbf06eadc27de338eaa3cac1dde642d52aa5359198d8909d23d87d827090a8ada7b7a5553642d586603ea2464dabd2ef5e18db3a623be65be7b5a469890f9dde54a27ca723b4e05d56b7181b28d5c1f65415688ee41d5337a9952d92ede4d192b9091639caaa6033e4749418dde15abe4bad62c37fab05e3bef4cd7398a4977e07e121fef2aac56be7e0546e40fca885696a3850c9a28709e699d52611c9b7926e7ad18149040582c997db71
" ~k:"6defbce0e72f014526c8ab02c6fc320a4bbd85365d99fc5d3423fa4c
" ~r:"515b9ce53eb10c3e47890556e0f0fd19adb207b9c01f12ef5c6caaad
" ~s:"0900e3acc4c378bdfe9cda4db8f8ab54436931c73d8d3171c6dc8bb8

" ; case ~msg:"4a145dd5cc4a12ea43617ec9790f1038190ed3d8af24bbec14da3ecf5f387ca9764a8b9cbc5f6292a53a9da9533c751140f8da5fb6f3d48eba1e7b98662734d9a8b120dd515408ba756f75a5755212764ad92c3f2263835211add5b4cc0eca8d4fc7a843f49c38ce80868faf8b498fb414d3080ed41e3674e285d3e40d62f305
" ~x:"7944fa1a2a938ffabb234ca385916e01a89220cd16f06a474b9d4ac4
" ~y:"b3f6d44da86a515d7185b70c5adaa3f6059c0bb7995a53910761fea362d9843f92f2271ddb0bca0d4519e33fdb13af49d855cd0b9ab0b970267243e468d3c41677ac588fdfcb1cb9aa4d233f7ae017e67094f4f4d904e1575e76bdc6bd8299b42a2f39adef63ce047862aaa0bb8ba32ec2733493648406f54f5d8e2eb19eea837f4d5963ad3192917f5fe3b6d027b22bc1bf0dce8401d622ca72b1d73a89e888de1e62bead2e4e1da6b5d04b2a3694c76fe07ad3c66426343d67be12b2a72c3f76225573fc054f3b7d735915238d7bdbcb03ba6dde3edc00f8c983b0b50129fab426004a27a038139f2d3295b5b032701face34a7523559485fa631c219237f6
" ~k:"8ab9322319a138489eb773f3220d712d05cd14eed9aae402a8aa767b
" ~r:"5cfd4b9f92ca727d513ac14143b125148655f1642c53b73cc25131c9
" ~s:"2adef94aae372d579c99629ca0786362cb0247aa6d99957074cd7d43

" ; case ~msg:"428a20790cad1c7ba82118ae5841bd5380ee50be5b64b8040935ef3d6da37a26e6f02035fb1937c7a6bcd88c894fad7d8aa48abb89e0c64287cdc637454db89eaf0a7e692734c8a243856dd75690bdcefe554e39a0df84e6e0c96b2c5774a3e4e2afed028fb43d7998d3cdc9a6409322cf3bfa4d1e36f5e707203b59c49a753e
" ~x:"43dea1b4e5c2f22548074d7dddbdcb94a235a3dbbfdb7b3bfc5923d3
" ~y:"47a9340ac513585c83bb20a2fba946971811184fd20065fb95cbb20625b47b216f75e1f3d89797f540a0485cfbf07b1716a3ece7027d86f4940ab90bbfdd8ebf15137bcf8805f93cea259c4bea5a2d3bb3dddf83aa290d3573e91aa300bbf1afb9b525542d67a8d86051aed8ff8a2cfc225a9e51eb374c31fe103ae8f4a0c8911421d225c019e1b5c07dc149babc26b708fc0fc0c13c3b35390317c409faae81aac9ab5d01ce85add24917d94cd1b2141b638de3a253bfca6b7f1a8104518d1572211ba52dd175632c8f3f6748265a4bf6c2b8363d9810ba1f1e584794f62319f0451da831d457b5269bbe67784c474ffff692bbe2baaca32d3f85f4fe39e03f
" ~k:"5615520867828ae7dbc8e9b116e7661e18f09e5cdae17518ead1484f
" ~r:"8a96c419e0f391daa29fb162a1b9570f48a00810aa480cde0f27cfb0
" ~s:"028ed9165522fc59aeeb79c491a95ed8427fd1b695f3dedf4228a328

" ; case ~msg:"2a07e28fc102dfe17c79b9368e0ba92414d2fcb407d34e903a0a53370f7d2d33aa13c02e527587718c3b39666125eca2e8fd4c94b9867fb6ef16d555549d8dd0f6e10417ebecf48f992ad84b5d9774540785ddcd264c55796bc2162898ecef4027c34187f8c0b1c20d4daa108b70d76c40ddbebc1e0f50f4dc904dbfbe6beb9d
" ~x:"5f4f3c4f95efb91c6b49f43afbde6d0f9b5a1324b4926f3276bc913e
" ~y:"05f27ec035627860c31aa597c96837084605f270d15a3fbbdda1c3853db2ea6f6c9de4e11a6fbd773c300ebad0f9dbc33608f9c4c5cedede0c26791cbea35af0322a607739e97c3242f0ae7d36afe269aae64b5fb2db265cd756ced45d888eaab0465e509ab7f83d623f69e73cdc0c7670675ce0c29f49a19d7038623bde36e29fb854e6fe6ffdb916abb7d61fab4b620dc739a5cbd9608a45e86c2bbfb41b8699166822e832bb6cac66e004e93d190b951424edaf34bf6bd343bf60154f739c43562b03aeb4d23de1f76c18f74b5f7a73c805b22af8cc6bdc9b55779ccf6d441cfd3154616cda18807a9f5e2d7659e9e21329755157dabc622bd1ae2d5097c6
" ~k:"97861b777e2a8cffc4c2d24e2df9eedf0b65ea2c9373c1085ba44efb
" ~r:"91a4576931ed621a0342f14ee2ba8fa8e1bbdf894c1251afdf72146f
" ~s:"56755ca163f7dc89458a7a75d4dd3ce3adec42b4aa7d04b2858c47f6

" ; case ~msg:"7e96385816c97bd9de81de30e67db72436fb42faa9b6ccfeab1fa528c69e6351b2012a1097fb86d8c5cc60256ef11be18f16137617f8cdd29e3bab9468c12ae34336ba0e0eb6c828177d1d55b06698ddf753756af830a10ce9c99f1d13682668e3eb336a80618e666280096417c1e2b005b9351f5ea306b8c63fd184a59132b5
" ~x:"914e5d6d95ec12443f73c127b797229544971177f645b8dac5f6911d
" ~y:"2b69bf21bf689a1f5ed7096b27e447c1d52fc2473e9e4353dbf185632022fc605cefe5489102f7cbe984f00c1ab32f2def1a84f1bedddbc15f87aed0a2b1e912e9edd74edbe2c15a4c37533014b9d32b05f5a44d323def1cebae0e216bc35a1ca8a4265c3db5574eb23e17f1838e225e467a9426e8798c5a2e896536c48c4e24cd2ee9da1b61aed2e25b98e4c1f4ee55e0b4705feb2bb1694cb18a6414bcdc1a7489b4bf8967985489316b3e57ea281204ced3ed88ad1b207be7d294127bca86a9b861ccca192c15c815e2328cbdaa5899c9dd271fcd6eea0d2ab009a8ba001e6725139be26c5151875cdca7f91434443b9e5e47a45cdc8b7399bc5e8bed9300
" ~k:"7d00160fa1ebb10b0465321748eba9ca6e1b3b5216c0c51dc34b98f7
" ~r:"1bcaa2caf483abc80b75f670252faa2a8e18c32301ba6fc06f37c08e
" ~s:"909a7852b8d5c8813e17c040779ad0dc5e9e0556612056835e68d2b8

" ; case ~msg:"24ed7a16782b5c34beb58bab6a7d2028719f9738e5d1ba6978efac4b53b37c88e7ea02e0cf0fd82a3e50046052a9049541d12993254a46fe401f402d38943e94918bf7a6fecb08ed1309b7b0f2185967ef289a2efa6c2e37a74d6592a2eb7401ca5e98bb8645a94e57499d362e0f3133ef336e119561cee1b558c15508781868
" ~x:"3a42f9927b4eb39ee3a910e4418987d1af1ffc1f3d5df0c4920e05d0
" ~y:"9dccc137197bb29824b1c10e9e8dedd714efc936cff83f42634d64391f9b7f4fc3a231954a8c3bfe4ae0f82225fd52b5dde6dcd14c0ce5085971c515da38183427c7e2a8d76e40efb671af797e0c576e3881d434ca809dd553ccb0f7cd9f73c7aea2268f36c84170ab0ae03b2b46a219547564fd21c540b1603ad7306d22a9eb8ef37ca08c2b28d16c5b9c54a328ebb3c0f9505095c612270d52637cb5584ed08bad7138d3388c634b6502fa6473a2f594040b9acc1480b343d2287fdc70d16ba14b1c2117612dcc5860dbef8387af9aa5e1621d37a38f6cbe5935673ea3cbcde4f32a249eb6a5eed41cfdcaa4c87e8bcabaa6bd1fe5a879d17e9ae35837ce0f
" ~k:"23dda49474ec6cd13e1b0249ab24f50e9d69e40c6b5c07430780c44f
" ~r:"5f4f5449b8d0dda3ac590ba1640df9772ff08cec08528bc2d70d7ac9
" ~s:"5bea04bfd33248f26aee98ca8596774e95ce685465174d1caed7d920

" ; case ~msg:"4906dbdd9da6ddffa152fa2e250eead3c6ef708387a3ad64d34a0e057459471f48752fde0786db28a4bbf58114d8dc91b69e56be3c49ec1b9880d9917c73abc895754a60779b18bc951550b957a77c8cefa159908126cc801c665d1b01109ba604bb9e797c7a37660bfc0593bab0924df5806ca803381b24b03de3d03b484d49
" ~x:"0c53e5311c104f11f6eba646e4840d1960a92118204a49e3ec8ddec4
" ~y:"072cb5612596aa716142f5f756c9542013f3f1628cfc5497eb1ba0aa51bd5adb8eb8adfe059c0e0882e3c09a17d1f51accb687b243fd3052bbcb81b063c1e7d5be066587ebca078006f6d6ee71a69ef59b6365cbcf64d4cf1b9299e7403009272026fc1665ed403ab8dee40eea4ee7d562af001951926dc8bf0c783984664ffef629cb59d709b3d9aa06805d62afd794541a2b4ce0c59043acf73e18e74453e86a082f17914ba6b2b0fa80da8353c7ed9162609575ed41f8eb78dbafaa7b518de0c85b1720e7f493b914d5a3d2d0748273d169d55c45556bcae670575c96a444fc1d789f5bacfc8b24132bfbd75b3061fbacf2935a219b0f2ac5dcad718516a9
" ~k:"3523465a8417b3a05ba1032bf6c42511591f2830b55144f9662bf6c9
" ~r:"77475900fc7f3e0b80f3884af8604eef60ffe484bc6cd3de123f7959
" ~s:"26ca927da0d10b43dc1521bfeb58ff347ee143fc38db451c11a03510" ]

let sha1_n256_cases =
  (* [mod = L=2048, N=256, SHA-1] *)
  let domain = params
~p:"c1a59d215573949e0b20a974c2edf2e3137ff2463062f75f1d13df12aba1076bb2d013402b60af6c187fb0fa362167c976c2617c726f9077f09e18c11b60f65008825bd6c02a1f57d3eb0ad41cd547de43d87f2525f971d42b306506e7ca03be63b35f4ada172d0a06924440a14250d7822ac2d5aeafed4619e79d4158a7d5eb2d9f023db181a8f094b2c6cb87cb8535416ac19813f07144660c557745f44a01c6b1029092c129b0d27183e82c5a21a80177ee7476eb95c466fb472bd3d2dc286ce25847e93cbfa9ad39cc57035d0c7b64b926a9c7f5a7b2bc5abcbfbdc0b0e3fede3c1e02c44afc8aefc7957da07a0e5fd12339db8667616f62286df80d58ab
" ~q:"8000000000000000000000001bd62c65e8b87c89797f8f0cbfa55e4a6810e2c7
" ~g:"aea5878740f1424d3c6ea9c6b4799615d2749298a17e26207f76cef340ddd390e1b1ad6b6c0010ad015a103342ddd452cac024b36e42d9b8ed52fafae7a1d3ce9e4b21f910d1356eb163a3e5a8184c781bf14492afa2e4b0a56d8884fd01a628b9662739c42e5c5795ade2f5f27e6de1d963917ce8806fc40d021cd87aa3aa3a9e4f0c2c4c45d2959b2578b2fb1a2229c37e181059b9d5e7b7862fa82e2377a49ed0f9dca820a5814079dd6610714efaf8b0cc683d8e72e4c884e6f9d4946b3e8d4cbb92adbbe7d4c47cc30be7f8c37ca81883a1aac6860059ff4640a29ccae73de20b12e63b00a88b2ee9ba94b75eb40a656e15d9ec83731c85d0effcb9ef9f"
  in
  let case = case_of ~domain ~hash:Digestif.sha1 in

[ case ~msg:"de3605dbefde353cbe05e0d6098647b6d041460dfd4c000312be1afe7551fd3b93fed76a9763c34e004564b8f7dcacbd99e85030632c94e9b0a032046523b7aacdf934a2dbbdcfceefe66b4e3d1cb29e994ff3a4648a8edd9d58ed71f12399d90624789c4e0eebb0fbd5080f7d730f875a1f290749334cb405e9fd2ae1b4ed65
" ~x:"5a42e77248358f06ae980a2c64f6a22bea2bf7b4fc0015745053c432b7132a67
" ~y:"880e17c4ae8141750609d8251c0bbd7acf6d0b460ed3688e9a5f990e6c4b5b00875da750e0228a04102a35f57e74b8d2f9b6950f0d1db8d302c5c90a5b8786a82c68ff5b17a57a758496c5f8053e4484a253d9942204d9a1109f4bd2a3ec311a60cf69c685b586d986f565d33dbf5aab7091e31aa4102c4f4b53fbf872d700156465b6c075e7f778471a23502dc0fee41b271c837a1c26691699f3550d060a331099f64837cddec69caebf51bf4ec9f36f2a220fe773cb4d3c02d0446ddd46133532ef1c3c69d432e303502bd05a75279a7809a742ac4a7872b07f1908654049419350e37a95f2ef33361d8d8736d4083dc14c0bb972e14d4c7b97f3ddfccaef
" ~k:"2cb9c1d617e127a4770d0a946fb947c5100ed0ca59454ea80479f6885ec10534
" ~r:"363e01c564f380a27d7d23b207af3f961d48fc0995487f60052775d724ab3d10
" ~s:"4916d91b2927294e429d537c06dd2463d1845018cca2873e90a6c837b445fdde

" ; case ~msg:"49707b655b6d168c70baede03866b0fba60239ad4cf82f53b46e11b26fa8f6276ff6687d09e8ed1e5d963c11e4763b2e59a0927f01e8fffd1894a6262327c84bbb4298d7d7fbca660673128bb7dea46178146485539f9a8f88dac761d0d5d45cb557cdac960be23dd9199acd99cb64d1fee2ca68e423461a02abb34c1dc45011
" ~x:"62177a5b2f0b44352f643a9e69c1adb4a0b292a5ea52fa8065e94ad043d46218
" ~y:"385349ecf99ce783d4e7a80a7dd2c533a3623c38260243ac392d4eab6deda5b79b8f9167922e8b60468623e4603fa7681f535e20de673531255e108f542a26d5c87f19e063372d142869c5eef1325281fee7f1c74d2a96255d420f2713864d55d36f8139194f643a6e98b5bf9732c8597445af5a71e23e2ac5cae3604323f7bf09449786974ed53a5717f9aec14dd01bd1cf276bf3c63dec43c3ec8ea6557de469916412f0456c90f01291bb7125e9f855f455b360c03d4a7b4a8d4090e47aaf1111f382dd2605734fb54f4b8ffe23c9ded2900b3121b497bd46d0458a09a5df4aa9cf1be906f5542313384f93d377ba9e0a762b4793403b914e52865afabb67
" ~k:"2bae4225836dcbbcad976ed47ecb5f3fc05439358791be244e74d2cf0617fc26
" ~r:"0fdc5a5a4a2c2f3df50c868383ba800396ae25265be1a14762d3110cbeb34819
" ~s:"4b41841cad45fedea5aad0a16b053e88353b6f0102df74c9fce09e38f5e6c277

" ; case ~msg:"763c1f15c5dd8a93aac4e048651c4ea84af18aee255b56959eaeb1876699be75271af0da6c3ca936e99be4ff4436410f69ae7018b6c843dce9d8b71a91efa53c39be55f285fb8ad8543952fd3ca89271ec23d342cfd557bfb72db43b434d0ed5b30763037754bb0f782ab08235a64abb7f0a828f892cde7e05e301da7c21c096
" ~x:"5d169761a3887a9eca0f7e59d77b75671ae02210006e754bf2f12091fc3275b0
" ~y:"0becd917eed0be9cb58ff9d259a8fa415b816da4a25d3f569d7b9f317b3f47e4244cdef35796fb455c05c156452f1c8660f5346fba169276221446f82bbb2027b056b537cfd59c57299166a6f20871c74e6c1d3f5a37b75e8dad6cadcf12c909586a32f150c68e332306abef8be1abd56c42d3c36936cf8f2acaceb707994a3d4c0555a015de892037aac68e33813bf3050f0f3a8df5e81465852f6a195ea688ac5d258eee2076a6b236362e3d792e7f358c6ba994da7a64b18263969655473aaa37cb3cfb00a27f8fb24a4b73b025c96335438484e958ad0848277df950847d46a9874f1039fbea7e08bc79675ef1df6ef21230a79a3b161308a0a4600b5347
" ~k:"66011bdefe8cc4a04fbd5d69252bb72da8f9a8d6e00bb7ca75719133ecd86f1d
" ~r:"76e9b6ef7e8d48fbfc43bf465281592223fa7e0d9978392d355868c8a20209bb
" ~s:"7f9c8deab51c60bb6f866c76450138e0d2946aca6c5f88dfe35a0c1ba493ee47

" ; case ~msg:"67851de982fc70f969d82f65d85b0332d667114f27b58bb9e565d2e40ad011983d936049cc97a216260fa2e410ad6d6c98a548759aa8e2d022c1fbc1b16b10d83fbbbd126ec43d5fedc407c831461c7f33ed94740031ecd0f701c7b1df88a249265b3f60c38f4285bbc9bae164bc38e162c235c9a9dfc1b150eaeb1482ebed48
" ~x:"4f3e2c5901b656118d88a47fe2bd52f85cbf828dbf9b67365e2013a937f0f2d9
" ~y:"ab9a99ff87899bd6657b3a9e9b7206996bbc7799dde57dcfff8098875dc4650d791e90bc4cee10989bf49eb5e6230857f96841ae8362e4ee5cc8602f6a1a2c6f8f2f680ad3a72b0e07511ead301f575278a074138aa4eaa53919e34f001cbe2dcbc345c77f5687d071981a4dca29d026bb53ec9cf03a88d63c52206d351f8fca10239e84f4915ce347f48d650aaaa6b02d3164973f82fc0e0f83a2d458af65736d7e0dbb264fd779ffd5a3f066584494598526cd67e12d6c67965a70ec3f09e2cc447f177ec87604b531486683025e3b520a26e69c958cf8435f7c6ce564f0a72d1fc47205a50b39d516b14a476f6c2dcace50339cae20cd3421a75f6d377b8b
" ~k:"72bd0808076af461353d98cb0191ec76a7c04fbe3a7f793e390cc773434c1d4f
" ~r:"763e89fc8b2a090b75812aefa55de7b7cd61ec3fdf8730ce16b05a7b9456fd2d
" ~s:"4a97086b6717a73a6be6d4a95b8343bd20b0d7b51c3da1d86c5852350871379b

" ; case ~msg:"616de9dd23ebede428e032db7838108a224f7aca57b1df87f031fe1d86083d688c5c3ef078e64d8d5a9e612d3983460ca1f816f787c03ca43a1fd8ce138655df677056364c0eab8e0493c07bd4b2b05022190932de794f195dbef297093e7da1c4304db40b63ca53e1b8bcdad913d7a902af025c367c48de387f1a9bcd7ca42e
" ~x:"4d0240a34dd45aacaab9e24e4838223ccb759f1d93fa8791f28fc7c2e8318820
" ~y:"584eaeed2dc785d8e2b8c85fd0e5ec251f134958bd9eeae4f79f862b62cf602ab10d22eca499042f2c875f2708ba0d697af39f23f5e0b7de4ff7964bab1279efa2aa797a2d21e788d249f42693cdbfd71fdcb1aa93b79bac0dbcb587bbff4ef15a3799a5fca8b1589838e30096069ca7931f7408815b585d140a747de43bd92cac3f9a9b1862fd704673e1e58710c16ddbe7e52d31a7df15974958b1288116ed98ff247f5028cec86d9eb97b126a48adc952e90dc52f2bd7810355aa9075051f26129c2d2fb0ba8066e414989d92e29e689960e33ee56ca62d714a42cb7487f70c0c0ba643fa9dd5f85259fdecd49fa970c8322682b114f2647837637abc0ed2
" ~k:"325e19d8b7ee8c8d9cb7e70bb5417035a8183bdf73149a45f0e83f3af68decc0
" ~r:"748f466b7fdcdfa77017c865a33b1dad4db99dbd63efa1c87345c4833b0632ac
" ~s:"0bf9938e7972ebb00fb0a3c0c2476d2509db23afcaecb17dc571905317eb8ca7

" ; case ~msg:"115f0a8be34e84d09bdcca69d19ce17dd67df739aa4fc6e8077076535f39af8302881471a5fb0e1839a3aa76dfda4bde2f9fa25fa582b756a4966d75320ac1995472271666156ea86c19a239895e5578a3c39b0ba3258827a01df1f30db22ddbc267c9e290d5d457d0a94d8aa73f8e79f3acd31bdeee7aa32c792c22acb807ba
" ~x:"0800394a2ccdd1f55800565374d46be9bbc1190b55eee26502bf5f2459ac5cc0
" ~y:"2e06073f59196d3e29ba718e84489b6f447fd6f67a9ee6357c5e8a58fa3c4fb6ac8314ebdc3b4d6127f2b4d2112c27799f0c1ac5f7946b5607212d796741cc3be127212a125edc3a7a91a525cd62152199b18b4f1dc332215d65d64ad06098ff2180ab47bb5728720c937e1207649ed19c883331ea415faa51c556d12649665f1ece880d055a2a793adc74b38f15f50aa9b46786d907017b1d6235c43b37c2036a1640f6bfe3bec2b95b4300a3bd78f471f6aa56e5e6347571996f778670ad94efaf20991c555924fd55cd518df0bd558faac3f9826a865a3ced0f59cbea45c65412bddf8f2a8aab3dfca1dff50374163fa899cc7f7f108b194fc955cabe9ca4
" ~k:"617d00444047d8e943e429947d28b4718a8b7603475e5453cacb80fa704f90f1
" ~r:"5c8d76440735055c1b36698da73903b332d64ca5603046144fb7668b1acac337
" ~s:"11c54efbd492a7147a1c50b287377b52d2193907d5bb636159c15318a480ca6a

" ; case ~msg:"3c1f2b92db1b4315837baa863043a9b4496a78143ca74f6e67181facf50a6e08d27945d00e7b06f9c57c0e2f1527c94bcecea6993175d0f09bab4f15af55ab7aa9b16b48c94a6a99c2d7e477b744cd27cdb9b0bbf810756bc6376fa15bfbea3c9376ca6979752fdb3a655affd6c0186d1a34355daea8cc75acf96b8847dbdb8d
" ~x:"59edd0348ca6a85c408816549e9c58338ef92f56edd8fa753226acc0e181751c
" ~y:"a4742d3c7e7681b01cd6aae17423cc780491d08df73b4a71edf7bd2ee29c698cd66dba0491688fc7eefb4d709147bfd4c8c4b797ab9197573b5d36599c4a592c466955e80ae5d2122bcaa5d0e1d94b4ed2a99b1af5d08eec86c37753a3c3656c0fef0d2c471e4ffa0fb163174a4df1707879fe083655291127a3bbb0597e23802e424efe4016360364506c8ab4081f0a95692c2629537f05306181db669bcfaf01c15395614238a2309429199555142639b3443ef85af74b5e88b7c70a8167334f27294a8ba1266695a369372badcba7623aa58cbcf25b4bbe663d4eced1a18e7753391d6c53854c4a8d0ee1a790a1a21071f1386c235ac26182d01a1e81ecf8
" ~k:"0a96189b8740005f215ae5c5a8aa8686dbb4c353d2c55deb3904bccc4f9a9b9b
" ~r:"31c1c6aee7ed541a281f37632b27ba88536f36bcd92fcc360da041f4197f7f95
" ~s:"45e1019b2a1702b5df1eef4fb7df6a53aaa66ecb8be5cd2e28b353c870e01f41

" ; case ~msg:"ad389f53235deb068f7097780330746493607fdb7e1170bd1fe0da012714b8f1b128c69a53d7dd2646b09720883e2387dd15d46564adff6642372c838287bafa5f4343a27ec8069770e5c367548833fddcc5f8617aaf41289d96dd40f1098ded9fbb110aeb14d69272dfb2dd7d75e7a88dc4147f27c64eb1bf0aa0569bbda320
" ~x:"19ff4eec2e47301d0b70a826dad822b609c997bc1b3a9d7cbd3d1d2252e8acda
" ~y:"bf4aa2d867b433f934d1d567010dbe067905f4e35d7ce568b55aba694d12dfba95c235078461aaab81f1e4df32319e5759c5263ebfbebf7960c57aed79bf2de38948f8ff79ef26d66a7f98384117dce1f386aecc4369afb2e0de77ccd2e7dec328614243effac607c8d5fc5c7c0b1143963573d9f106fcecf2e15c67a3bff6908b286d0e4131fb81622fff9e10f5771afede2276e8344d9ae2f493fb4856d1ba5760ddae38af7ddca409e7907268691baa33dfcbfd69e9aa9faa79cf303ac8b1fa07c1d40d1cea01e8ba0d65265f4c6aabb16ebe2f6ef5aaac25c0c2730cbeedc177667ee02bf4523418a986d5b87a9b75ec201af0f1961cd51b85879147e607
" ~k:"7ff51bb8946842c7e2f7245e73461e2b0820528548f7ecb53bcadc7a20e826b7
" ~r:"2f9484aaeda9dcb88d2d3644db2c58eefe2e7695a6c8be9abe97173efc9c0bc3
" ~s:"0166a7bf4e8bda6b86396943a74a8ebfc603a85ed287bf3f5a30dd0bbe49cd8b

" ; case ~msg:"12f9582e3a1a76f299d72d9b1502b99060802660226bc47b71e54ec9388eac325902acbe2bd7109e19f377c9d2b4d280cdfaa48888b9cf4ed06ccf5ad866d6932d402592f6be6e6876db5a62beeaf373b60238ab96829243759bdb586f45ec4ae2cb22248ab0b6aa7a7583a61dd3b8f119cd840479a4a9af8a439db904ac14ec
" ~x:"7142b195eb2417bc234cf32c6fd7cae470cb48c74dbdb469a264c1988eb3e52d
" ~y:"72d8100692e1a30a32e37c909eb6c7baea7258b0b78668e75915070037479b884fa9f18066df89b490f9a2696a8505036977604dad268e90552835fdca3339b32360c94358ffcd0b1ea11066122efd017cd6fe1ecd0dd6678081b84cb6e144471dae7636b4a0929ca71aa47b4086665d66d4034c188d64d38b69f0ca171c85925cad2840277d2887a7f7b81e6b12870cc3c69e18ca9c22c3d3a39ee286ca65d23f3e8111aa7c6ea9a0d14c84ddf76abd44db3b9833d69cb99b524c98fdb9d0ff20c9d268e8e7175f13c11c5795d0fe0b3899b74c0dca91476febcb509f7fd507023988145242dfc809ce95c6f1b31f67e01650dd45878efc7ea89cf6e3171e43
" ~k:"1043805a13045a36e1b6498db97d163571c61cc4a719e506173b5e6df33fc81d
" ~r:"38cf6b8cbae82e6295f83316a9c49d2dc7c92cb90b19a2c2d45649949354d930
" ~s:"356a5850d07aec6e9d4a4d7f79d9b0352b087d7ef48394128c5ae4993e8259b8

" ; case ~msg:"b6ac84c49f6bd601d5868ba06d49b8cba87a9d6e7905247541fd332c2b0374cf57d4a0dc0b5a6c3f8f7e24be3a1eedc4a8c575847c02e4edd4745040685670058996250f73e298a43b391a4ad567f0c9bc4b6abf6d1e5c56b22f4eab36aa1a812a1dae8d2873cb2c2a521d320019c7cab1efb11fa4595c534ce527d43ba605f7
" ~x:"1332c3c6e2d1b7b16f501b6d48c7b866628f0c82bf33354535df99a843dd68ce
" ~y:"06dab48a076e8cec27d4c4fb98e7c00f36bed73f11e491d913864cae0fdf883468d735deee5251dd38a1f8b1d2bc19d37f3187a4ef69c33dc9528801a23a98d96fd3f129b8ca2941421ba1828e0c4f8d88c53193930292a0df1147b07c20aa726c7177ef660ddd4ecdd73315d4b9356013e115f067e843c896c1a54c81ffab1bfe7c785edec32fba652babfdaaa039b0568c6beb7d13fb4e4588140ed626b18749b0f79f669f6e7045738cf50a6d0028ba11fe1845a2dcbd9c1b02336fb30eaaa397418fe17e149829cab13d2c2e6b90e5cc81834e32fca8a173634e01f9a973e029644f0165b3033dfb054dd21d65e0c0e137b48c34d42134c47b972433ccde
" ~k:"167b97578e52869f49730df464f7e8d786594bb830d72db9af2cc88324ded288
" ~r:"1d600a745a1dec933868dc535a19ee9f1af8bf09b5abee15dc4f7cbcb95ac8c5
" ~s:"23b81097d583342ebe4aed364a7af9882f74e64518aaedce346c91d6d7ac470b

" ; case ~msg:"a92e2ddbfd18cd307373fcb39dffc33e0b91a48c62071f2f7a8e50dbf2c290889307975b6acd642c8e3d3444acac98c22ed06551fec5dc7c9f2243b681cc9fa4fcc12c318237e9a5df0a77ac22402039cef31b1e623af58212a22e7e60419bb36b777cf6ce65dd1f56963eb28b7706f137c0f7363a002d827e45badc20233c16
" ~x:"119ab8a63a22a89baf4eb8f016dcce9423d5f40a677b258fab072a8cb622ebe5
" ~y:"5141223f4697de272269f3d99437c48dba5ab7f1373fc6bad8161018c5d6fce2bccc40ca78e4d73b6eeb096f175c4cd0c8e9f3e9311951d51ea244fd33d9e47de75f1000248fdc003bc07b501ce58f6ec1aed1754c36826cd91976b408eb7aa9bc42448058ffd3b4e513c6589f8e1bc145a47b2470e7241e2325e54302255c3d6d97abc5c6056266a9523d461fc744146da35c04a4fc0b095881cb94fc4c03bb8623953928490dbe7f84ef68667f23d4cb3ed887449f77aeb158a26d1b39b4e6297f23d49f5b41f170e72f7213ee40364c1c9a63985f69e44eacdfdcb58c35dace8b935d0789a8c0669a23d673929b2a582d6d3b2f9e67be891890da1236c6f0
" ~k:"77cefd7a6b0fcd0237ff8f51c458e5e8a79116eba6f11ea1af7f29aa608393e5
" ~r:"34a65e99bf01698b5a68f215b9c292115d17b3c202ea1fda17fcd8a0cd74b636
" ~s:"7e67d442b8f9ac2974e84ba65aeff0df5f83c271ece792a8dab9c4aee87bfea8

" ; case ~msg:"b5aa1cfe2348d57f0e5333fc70276d2418ddda49122f4a88e8010f6f78dc829ba5c7cc68db664080945c43eeb705c2ef13de6e4b8f4de1d04fb33d5bcd7893d8ca8bfde38c9feca6c4ec03b2ce7b35ed60a6a43f7fc9ed08061a099b3eeeae7f0f1516149d175a953f52c8c518f3ad247c9fba23f1f829d5cae62673ee201ada
" ~x:"4b75db034ed0b84dfcc60b493a00940e805feb78575fd256b24d146b05a9500c
" ~y:"0b66ef2c7a34205d70fc36404957043cf46b28ac4f083ebac3787f55e8dd1f75d9193a842759376f0508c94cc7528d6611b50a73261a4a5cff730d9985bb341dfd739a4e963d1c40f114d7a7ace89e81dd70861efef2ba9d1c6425d5f85809059e8ef31f453c97743fcc94d3b1bd62084e975790b37193eb4058454ab283fe2bafaae803de892879554a340b9a3e2532931eb95d3ac5eb3f290a3f56936951288e1c05bda1fa74dc78d631c2e7a56367ec5781019dfee71453ea6bbd90778e92fea8c26bd6a823fbca71577b6335f3bdf40a30836e948db032db5a4603dd31b851ecbbdf76b4a6c9951d2192b97ff01daa5cb030e15ad1d4cff367f700e79ffb
" ~k:"654aa8be3b7bfc32f9b560b57a88a8aec1cfda276661283b7f44dd3b0944c20f
" ~r:"517f7df4831fbd01908b9218b17ae1c40e00c53404b3bd72b64f67cee75215f2
" ~s:"1903434a727c8ef0e80a43dce2834b807839ef43c22afb502b35a381782bb639

" ; case ~msg:"27aa81d2bc49601c3f6bceb0870bb55dd10e7ba6d1f8acada70b5f902a0f4062eb93ae72cdfd3f943099cc2a10a3da7bdc9f24b00bf36a29d75136af10bb71ec9c1932058e22ec9c0600d173d37970d58ae1f66cefd27e2905afdde4223979b4041fd7d7166ea326befd5dd896ef47abc6d045c1ca23c1953a6e12cc3c54b4f6
" ~x:"7e6b77d4bc9220c3352e91abea67e33a335ace34ec4516646e8a4ff098166ff4
" ~y:"932b9c0f2d310b6bfee800c074a0969efa246244fb062a745a9a3cfe6f5336a313192e92a2027e1d2c3cfa93aac53dfe05cb8f8321ac882a63bd375af0f3d9ecc73aeebe1267f473a9f90b94f5b6de4357b74eb30cd41aeafc259e85cac7d365ee33382a584eec63719ea325a2414e116f84d2af9654268ec44d6ea2e981581d45d805b383d85c130d2dcd1c71fa68d9c76d79aa8196152c1d9440c33d99de451a359e0d2c51d6aaecb26795406e528f5de3e00947d3dacc695c08a960889a2e94ecf0a461c02afc58b51e00369c73c8140e8b92388caabd1f37a62d1b210e0f314127f46b576a4b8edeb34713aa4136b8a1875bba8a5937066544e34c206aa4
" ~k:"73c28bca3c8067da792f6312153b298a8f714cad70bb2349803b6dad024f6bc1
" ~r:"05057a982ab4a2e32238ef2e3edba07fd193d90c5f053c83a9f176e21a9d5208
" ~s:"03c2b26cf46b7f72691a72d7cbf33653df347f02b0683ebc6cb7ea7e72dc8a0a

" ; case ~msg:"7527533f2d10c18078f5a8dec350cdfad06d3157871e4ff7d7c2b7ab11dff232d34f07699278f075442e1d4ee00cd6e87c1931333841c399576f4e587a251684e731f7c8369f712656bc1e6c2d209f511179da09368d93290e058e0ce9b6530ac6c5e4cf0a1b22d588d98f32b34e85206e09aac04a0e1f2ae2a5cfdac4e6e2b3
" ~x:"40792e5ab46518c6ffcf5357f0c5de5d9e2de99c92aebea82a307ab0f5ad252b
" ~y:"72c46505e4b071f46ed6b6d530801664a4fd518e4c6be8468a38c22bf74ed966fdc7bfd7c572218998fc4c144b59462af7e294bdf5797ecea5cb2edf8c8d2dabba88d0b84cf28524369c5040b58f090772dac0fe453c32907e9b6c740fb24ed4dacb8fdd25e0661bc0d79d41f103fbc8f96b3e3a4708a5a7f5dbffc98f344bb7ccf0d5ed07af2c2f0d5f407bcfefb54d9b947604e7a78356874c01b8c1fdd749f6a3d619d1090c83725e725706846c16bf9dfdf39f2180623f4f585402cc7d6e2c10b57c8300543686a386056a931be6336bb6173d9fda8b102cf32989cf0978f956d9ae0d8f30752f156f9f92d2954ef13100a75d9f7ff96fe15df07e7993e3
" ~k:"0c9fe826a7618108684ba2d74f10ca39168feb85f74d2737fd12d18cf27a2f16
" ~r:"6aa6c4d7afda30ff2d7178b52a3e437ed5b0745a247c9c9e120bd3e833a1dfac
" ~s:"26e0887911bb5edb6a566a2a1276353391b1e4ab8ae0b259c1bbb3af3d85b439

" ; case ~msg:"994a49e5e8a5698fdac9a7faac01fb09b2c6113a186677676d11e6049dc98c93c51eb5144af181e1efbf44439a13d295653854813671f032aa62258c14195c4864afae0b5d154f97565cef075bbb6d97e34181410309ffe98b45c1f874326343c36c14f55fa058489dff3b49dc7888f45a099c3c919b25edac1706bb90f164ca
" ~x:"266cfbe6060134ece2c8b9e6aa25bd6cc935e49c23fdd4fb6adb2ecde63a4960
" ~y:"05e233ac49c1fda2a0c3c78b0bc72fa39674055d188a124a58ab3850d9a888861c2fe4d046c3e7c75ee254de70cdb1c3150201c2e04733ebcc25b88770fc2aa82f60526bc664047a026c2290fad8e9f81cedddde7fe3ba406535bf2710d79da01bd2d42bb5f4099c3f8bc2ac864be7892aeb6a1f3402c81474da23e0795cd6c21367509a541591ee1e6364f7e755b1419e90af869930152f34de51f0f06ca3076e68c3e3ea7f4f1bf1d3cde3a0dff0cffa1b5842752347082dda3475992f15a74d298524e636220bc9faed08af7aa5e481ba78d2d2fd8e51942cfd084efe0ebddd7500efc95a6cad37fc4923f9bf65297805840876c689ee079b7fa6169768fa
" ~k:"60f8416735fa49ab567c0bf1b6da434e1df41579699c1a92a3e70e1d90705379
" ~r:"3cc269bc7b895864a03231318cf39379ae33c7180a18c08b5aef7414fdac058f
" ~s:"6a6eb83c5fab10e34f0416628c821a6de0ad0c202443c6df032cc9d8e4948ac6" ]

let sha224_n256_cases =
  (* [mod = L=2048, N=256, SHA-224] *)
  let domain = params
~p:"d02276ebf3c22ffd666983183a47ae94c9bccbcbf95ddcb491d1f7ce643549199992d37c79e7b032d26ed031b6ba4489f3125826fafb2726a98333ebd9abdde592d8693d9859536d9cc3841a1d24e044d35aced6136256fc6d6b615cf4f4163aa381eb2b4c480825a8eccc56d8ddcf5fe637e38ad9b2974bd2cf68bf271e0d067d2465a8b6b660524f0082598945ada58ea649b9804eb4753408c2c59768c46abb82e3295f3d9ca469f84cc187f572dc4b5a3b39346ec839dfad6f07d6d1f0e215209bb0ecc05c767cf2e7943ac9cfb02eee1e9ef5946e8ce88316b5e15fdcf95a132ef2e4bb0817136528cfa5dd96532f9c3abe5c421620edb6bcbd52234ca9
" ~q:"8000000012997e8285e4089708f528070c6d7af8a0bd01409e7a079cdb6fc5bb
" ~g:"778453049ef262147fed7b59b0ee6764607c51e7b5b5fc6fea7a7a7b1dd6bb283f4a9ae98efd3964b1556758cb15b2a53af8619e74d85898bec77d3b3f382494ae5961a13ffc745da386182291519800f99dd710e00aeb15adee088e2798ee2e46f598526cf0f4667055d1ba009750041dc5cdd2725ff1d97dd340c8518af7671b87d39d67aeced84b66f84e0701efc82a5c9ef954ee576d24c385b14d63037f0d866fd424b4975bdd5485ed740cb932e843f906683f7c7b2c74775d901c361b847b519c0da699638da40bd736b783d2710b2c2cc26ef91271bf4e2c1929f876e902e2057164223bc78d6a2b9f6c0c7a7cb85922f7d6c4287ae23861f8128848"
  in
  let case = case_of ~domain ~hash:Digestif.sha224 in

[ case ~msg:"39f2d8d503aae8cd17854456ecfad49a18900d4375412bc689181ed9c2ccafea98dca689a72dc75e5367d3d3abfc2169700d5891cff70f69d9aca093b061b9f5057f94636bc2783115254344fb12e33b167272e198838a8728e7744ea9a2e8248e34d5906e298302472637b879de91c1a6f9f331a5cf98a5af29132990d27416
" ~x:"6ba81e6cd4367798aaab8b7af1135183a37c42a766dbd68cd2dce78f2670ef0f
" ~y:"7bb31e98c7a0437f978a73d5dcfbdfbb09cc2499dfaf1eb5256bccd6358cabb5f67d04a42823463b7e957f2b9213f1fa8e5a98d614484701abb8c7d67641fe6ed06fa4527b493ddab2e74640fde3de70da693f1db2b8e26417040af0eea6cab451a795a52e187d2ee241b93f65c86c6d66f45834cce165ac5eb670d4f0095c23ce9757e3bdc636f991ee0073d90a09202edb35cc3ea1cf9adca1617fa0bffd9c126229a604a1d3bf4931ddf0b9942dfc8a2f8c09fcc97032564a79ae1ebe1e2ce49ff57839e7c43fa60b1603d15a450898aa4e4a1ee8065794126d64f013367096a83686b9f158c33b10f5f3b36cf1f6358b3f34f84b101dc26d3db68bcc95c8
" ~k:"45030b79a395b1632700cbaffead97998d02bed8e0656876fc0174e4bdb96f79
" ~r:"059bee9e708b7f20c3f791a640edee964e0aa672893c484799715817b3a8f6d4
" ~s:"4bd41c84a724cc86e4f0194ec0fbf379e654d0d7f6a1f08bd468139422a5c353

" ; case ~msg:"0577ee4a9b8dbe3c6fb9725174e89940b27e8a989217b64417e66f396a35e5824f21e58236b27910a3be6b57d311aa778bef63dd025d9435301aefc92223c1aabb03d3d5d385b1a3d1f937f0f1f7f8baba91a011207480b5c23a78ebaea69ae8ad4373b2b052d60c5461111479591f8330123bf74370fba66bc7e2b400192c47
" ~x:"7bd811cf6056c1a821a85a3169113639d775247bc6578c9eeb28d4b09503ac0b
" ~y:"c54a57b08f255db1c776bb2126ea3c1e60229f1e1981e43f1d6b9110f950edd8245eeca7d55ba06468040855b736db502f01d6b3cb2d9d621c4db44cf8cb390ab2ae332bca219e09bbbbc225541d4a0ec0b4f11a591c077f2382f04bd93b364c94fb1c6147ff7784e82558e5fb68427459fa9a69d78a9f6051bd9431887ace46fa4970f0e22d75d2befa5a228e489e009af97ce9211408b4e5bfe37d3e0700b258b54174a5125eb6bbeca38805da53b1f5829dfdec8c4c9376bf235b7b0eb7119d3d69768b80ee02234589b8d95faf8062a8e1e9c3a686b6350e30fa535eaae71d753b7c3b048f8e9722254dedbc220ac9c9af0784532032ab65e48ccfcfd623
" ~k:"7ce602ece3f821390641dec7ae01b44df0fc822de1c013496bade2e3e44fff0b
" ~r:"33c198ea68bec4a7fedaf0309c317d336b97d1eb1f1dc44ebaf5c85c5a3afa98
" ~s:"5c9b23c13bb607be5473b32ae2b5e8f2a1e18f59df8ca7fd9303f76ed8e680e3

" ; case ~msg:"c643695d29b28210017aa5a7d16ebed81ba00a869d6681d1c0fe90a5e8be9d597329ea15d24ba12d77e4c3f2160bcbe808840c6e77b0528bf9ae588738e22f41910a80a7c6e3340c127b9de17945e7f9229953e2850217b6d486f7cc804e720de214cef02df4a892f7e42898f15caad26bb30bfaf4b0551aeea14035cb756b11
" ~x:"3ff2653cbc1f27253400a9b6b1f064247053c9816cfdcb704b14bdece2a8558b
" ~y:"17ff2a5eff3926ee1520d5a63a13b4f701dceed25a653966f525450b3a63b03229d615ec54cf4f6ddb868b54df363feecc95eb8a3ab2587fc4de9c93dc8f8d7f38f99082d2867b23d073584c831baa0961651e071b43f9d5da97b60e7b5b7a935f6c1dc88279608e2bec5cac6162488085d092a97c6b6f24536589b801b6b48d478796b52c05564e904bc58ac1505074db3734fcf3575f79952ba0a2a0697e55e579d508a400ebfb2d4694b720804a9d00f8845ef0a8e690e675b4c1ce07996d64e666b0d6a1d6fc6bbc3cd9b5cc3864e5e888e3c335e05e83c67c0033ba5efc3dcdec0446d3b40793236ca074c54d2a74dad296d7c639dec938e3bf1ca085dc
" ~k:"356b49268eb799dc4db7781a06be0f8b96d28f6a13b7523c0ecbe70cb3eea1aa
" ~r:"4ddd2a1f411b570fef6d9184409b4fd55d12c5e4bddc2ac7211235873322155d
" ~s:"4043952c108ef84a25a168ea5b64a4386f7a483366054c5dfbfc5fa98579432a

" ; case ~msg:"2f64d11e290275987b7d7430242289afd54f1be028cf36f8f55db54be70b8dd5ad74ae26e079d0ed31a361c116951bde94d686abf15ac5ed1470c3e902461cea8e5d58f407d2e0c072ee61567da7b353f6c47e694cd607f3ae894a9705e8ea2bf9ceec3acfa6d20b238bf0a7a7eac76c4462b7e4e4e868174a88a6a6c9476cdf
" ~x:"4800e9ecd9bef5a4d46aca60aca96955d8565e1b85d84dd8141d4f597e178bff
" ~y:"41cdb2c1bdfa3652ee49695d5e5eeec00f64b54b5676ee27f043b43f24133f61425b0cebaa1f88da072cc68865c12790c43285b7e19c3844fc7d81d064423ff1e19266f69f7dcb3d0203739f84d73bf00c52d60b2875171216678d59fb557553edc9eba6b84127169fe5dd2f81fc902c970d1d8d9c4779dfa1b14309f81006ee641776a6fa36339e963117447aceb823c9ca3367172eddaf6e361829dae43c4038cdb90ebb68b53c0a22d410b6f1bfa7c47496ea3aeddc36bf24f219b85917a24d30847c77d87d22a7f7486c6684755e045ddf72d41650e97b64a64becadfc47d53555127f8b7ab78d480529571996eede4618882d838bd695efc87e74d68ca5
" ~k:"460410eaeb111a18cf894468e10a88b8de8ef9dfd9a2ea1882a9fb696fd7823d
" ~r:"4fe6e2a75d9c72e81ac60dd33d31180df829b31a0dbd5fd20b7e28c4fee27d5b
" ~s:"3ce4a06bfaf70cb6cc93f33f95a43ad77ed7ad7c77a1674bf849e9ebbc5eda29

" ; case ~msg:"173c4a23621c32c3e4b157ef96b02fc1bb466a2537d3f6e51a58e510c4aef3aae4bce4c0b4d59bb1c00e7a35f98945ca9d7fdf1f0bac732d425043062bc6d32015233dfb295ae08a324ac7c1e02a117ce436d77d4e46d0b794af04b1db82a2709da1c4449c29ccba93db8ec48eb17921cb389f6e0ae32995d7fee1fa07177a7a
" ~x:"3e696f226f21916455f8ccc861b1845303867b75303ed92f9ac79088f56ea708
" ~y:"673e349cf6d05caa16751d97ba6e344e40e158e6a7fc53ea2db87891341e6499825b5b9edbce9190bd87c3eadf7c6d5bf0a793af2c3a1c8ded790bc319449394c64384305864723a8a7bfef26c082030ab360bf9abb11117e61b00549726d772221f6f67c4a6a110cd9a9658781ea8f7ef2f176c6e8816a865af396db95d8415b541cf0f83e45a417374cf3acf5c6b4a98390522e7140cc8aa3f9d2dd26341d4eb79e4d931a178e3d57dc52bfdf90115e01b76094ad0294979d35d92b574ce7b0c627f08be66f99effadc33aed0f634f6a89507455d7341ee64183aa610d8bb3237147bd90dcd9c1a03d89b26ee31dbef5ae7e764ba9f77b6a7434ad2a8f966c
" ~k:"2837f7fa85efafb433093231983ccef5d82080e6063f67c68ff93465b59d581e
" ~r:"393d681c3edba28f7cb0f30593b94fc15cca659a80cfbcb3b236453722d5b402
" ~s:"44f7421bce1e5273a30ec016bb9969b757197987548e434e395ab3de1b0e7ba2

" ; case ~msg:"7d6f2a97e1eb085cb9e83aa24047af9ba30a05d7bab564a149b9cd2366518e8f199134fc2ca403947f2a614c0363ed4bc1349dc496a8ec74d880578475e47427628bb023f0272208876a3a7333307a596c158eba64ce42a3c790e7167ba4a327ac71aabad2f36341edea12ce5b2b735807b34b714a49a0aa476893578f0645db
" ~x:"660898413f7a71804432ecfa11cc68f85a34fdf75012c965259ea6ca0bbcd976
" ~y:"777c251067c8ab16cce2c4a4d784c7e806fd296cbbbab0132e2ab91623acecd830e7cc7cde03e544b51fb1d8f0b2eec09f559539aa9d63ebc0c1e32579f095473d12717ce88f6671ec7e3d2581f61bfde66cf9be216d6a208086cd7bea770150a9bb0a5a7a0dace82b464180241202a30b26ad5fb933c8235ac2918e29bc53a5c01ebc1e30b1b46e37124aec596f8d1a73baeae588ce7d4aef1ae84e9a9766c24367321c047c3caba629f5d9185f0ffb3af7e50eebd1ba0eb77eb121b98073794cbc6622b678262ed3e229c6ceeb607274ce3496f370b482bf8f68c27366818486b72adfc810b2f579779adc9c25002e277641dd9ffbc5db5239f677ba1a9c1d
" ~k:"4abaf5c6f8e28356fd0dc6f096e9354baac1c2049170b2db05c81bacf02092f2
" ~r:"463b1fd6ef2986f75f9620779bb6f47e0beafa9340e3e5ee589d92428acd4f2c
" ~s:"27edd33917e49bf771f3fa1355cd3928d0bd401aa7bf0541f3af1643efd7b677

" ; case ~msg:"7f8785e1c4f82bc0bb75f78d8c4113e0887e761a86b48dfa43a3683b2bb886ba53f5603c8d94a052af3671c5c1e7c232908e10faa6cd54efc79ccfd64811131acd7d60a9309729455aa70443ae8f32a34580f9a1aa7d89e5fa8cd4e95809a573ec6dfe9fe35b1130571982a0dd46eeebb6a16f85ee6314931839e3a4c29dc700
" ~x:"4be0926fe24da1667d71d2abc2bc0bf87172c05d7c363a324ec61b4642777e57
" ~y:"28c06e5ab3c860be8c13f74f28b5792b39487b79547f4afaf6f77a5c3a43e88132edf944ee00150a78b58a78cf92ed941578ec679e106767014e5b279c0eae9c408e6ed60687ee1464988ea545f55be3673ecda10c63fb0b1908e796d6715abd5451843da6e63bf8802ccada32c7c5342374ab26ee701f9db3d34fc96de9d23021b98a93df6877f84fad6741164055696f3b72050343ea3e5cca01a3d57e29727ebcf8583118146c27f42adaf62365b9697cf03bddc69d0bd151f715b23bfaaa27a368114b3dfb54c084e06d4343ffde1cd22058e9623a70e9942e090edc73db06dd3180bb960f0d7fed005b149b69d6d45f40368fc25ae04321eda46d52a592
" ~k:"057cca710c8e4998e9fe154cc57847bf35a512e6caf3cd338372b5becc66e8e1
" ~r:"3165b1cf3ca9bb89154ad684e089364f91b6e5d594526072f7b9db3b2358e711
" ~s:"49e1c8c34724ac5532fff1c7d243b486a2cdc0872ab84fda6cf2ba96f958f46a

" ; case ~msg:"3e17ea8b9feb2f4e55c103e58c4ead96b5cb892d0982ab2b0cb1eeb9e1ddde9990233a22588473421aadf52767a8df524bc6e6ed857a9fd5942ef976b1fd8bcad31e403b1febb865d2872a7b34ecdbab8b245ada45243a49c7be67aa09788029779d619de30dead9f7d8c9c42153b865b1a9e81180380e27a305a6392f4b2a0b
" ~x:"75c9b6c63c80755f7a7bf38eabc58e1bc2e0cc5cb4f2274f2d63058157656608
" ~y:"b71d0ab2d405a5012d694e0a4a82769256cbdb49c18112efee8153c8e816310486a17bce19748b11f3d5d18cb44998eb329b951c23a57cac47ec9973839b130f3a980e62705c0702e4d68425845d54e152e2e83646b56a6757cde06f85ba3779eea585dfe8302f12ae77fa58cbc6dcca70b461024b7d176510a393ec027c769cfe49b698e575fcf99c60293af2ade3dc4df23ff3386f13777306c52de97ed1a886b824788863ff7263bbbb5b5fa0d4681c16942272f5e441bdf49eec7556c1fd409c78e3aaffeb95c1267dee12c24c045ef67aa70e9a3d9244f2cf1ac68cd918df5f62a3dd3de7bcdeaa3f61de51cc01af636bd665c0099d13938eb4fc289b42
" ~k:"568b8f5049c2c411f05d74e1781be5718ff921026728d285f2a77025208dbd41
" ~r:"11b7ecfeb339d6014948de5ad4c96f4ba517a2cddca611c8887fc44f14ac9a63
" ~s:"13287a22cffd825302b0fdc0955458d918727092c7bfb3ec4c3d7a838ea6c491

" ; case ~msg:"c3e1903ceccb2af5b0dc6b1fbaaf1b2e96477e001c43eee3046eed06128c4c81eb2bc917aa8ac30d07e66c9a9469518e3cabc264d6936e5d724a613bf9a44d60797b890cc5ce0d04629e5faa1dd53e7a125a14a26df3cdd9878d9c67e7e18a4655a188885363ddabd73a17659d191e51fafb6d4171ff6c4b651168ce167ada01
" ~x:"5feba370a58c16f34e931b65c042e6bfe794309cf30105d2fdac4d9fb3e14303
" ~y:"429e6ba20b02cd69a29b4a97a6ea564e5b8874ada195a49c3a5293c9bc8d19e0a3a3c4ac8547bfdc7a209bf3a6037e5b0bb7aa291d5940d235c787a2af79a9cd7f83084ba7df85c036ad8ea23c4fdbf91d285c7caa6497af388017bd581ff308d9b56799029e21400c0c99d103a2caec195e40c90d244dac897bd418ae016d25f71e989af516d5e2491e1e4bc25914ec3ad0a9f85968a6777fbebdc73b1ac6814496d9421d2b7cdf17d53f00624010ed6618f1258da194f77c28286225d1b16da3fab76c9b70db1f7dbcbacf4e60b6b91a1f475007ee4d2c5e37fc31e89a0fa808f89e8a4e546bc90e696f454721be71c0731f99ee368afc6998761af9dd9d6d
" ~k:"7ba86d55b8b5a465f661944832862baf5f565ff0d9195986c809956db2872da9
" ~r:"77470f0d3923ff407e71a86f0336811bdd63e179891fd30e3452dac1e5175081
" ~s:"4b969f77c70b5e6ff9350ca25e7d951acaaee907fa7b830a32dce4f91a89afa4

" ; case ~msg:"4b7c0828b715ec2da1e092204f55ddd65d13f1cdd64c109478d3847487bc48a8cb0299222a7495efffa63ea158253faedcb5314881ab41b5e773337662cc2f50dbccc736974e31b3d080467589951d511032e4cba6647f94c679aa269fca6db92715a4da28ff9803a1dc61675fa5ac114e376fa4dadb37c1b09ed5c31bc5aee8
" ~x:"1ba85c9c8f4f4ae97013bc9f7fab372e733f3445fd9a68f8e015c375df3b5515
" ~y:"09a16e0a6003f45aaaa3c6311aa9866217d4a7c8cb5093514976f6a341260e5aba7cb00ab2adb7462a47a8cfee4fdcae5accda6d42a3144792a14631bbe85534c111d2ffcdbc15b6db9dbfc4bc71d300324fd310c465443cb2a6f2ae33701f39668b118c38ef562e8554fea661a3ef80455699c23430d28ba6dcf042fc920a677c2971b2df8c6729c5b3b1be6c5a047ac1bcc8cd8dc519ada221bd92ca6893c1cc1dc158f9d472f89a8e02649440dded0f723485558effe8cf9df121c969a2d1b76a37dcbffb17edf3121d4338d4ab68b154226c0072d8bd51f23e5659a2afe520dd5e91005a6fc1157f07973610c5577824bf1666ccf851d69efde347f0b996
" ~k:"11d09ab8f3140f98dd4076d398a9aafb9c98656dd7185567a562cd108932eb77
" ~r:"1b8b8d67b640afda26fbe67cfd4bea521375526ad58a22d4d97d7af134384f4a
" ~s:"66d6c240992256eebe078265c3029a88c34095142134dfc31ff0a2d8bbd609b5

" ; case ~msg:"baea89dcc102cd649135d63a5f52df437af7840d699a9daf131eaac381348d45b4e60477fea88803fca31b54829c5806c703eb8fdf412306ff7a79b55aab9064bc37cb26bffaa671debb74c228ba2d2a06da362f613b78e5b1f0a0b5c5febf6bc326b021bd7fc70471b25e153ea51de1010b87110e01497a7f1ac39cf4d424c3
" ~x:"2c0a2b700ea43f5fd589e665817339b60f837ca0b7dbab50d2ca7e4c362a14e6
" ~y:"cbd465ce9c3d0a137ee3d582a5172183b8a63cfe414070b247da367456203f986e6786ffb83ad764aba309c2ef7442ce38735f492c0ce6d92eaf9ae6b1cc873ab6ff58317cd166a510c3ffd8d4e6008825b58cae217fa35c94c9bbd12a4d638c20116398b21b5929dca1d49a7b748970e45de0d432fc912f76199137f1bb0c0d2c95bdcba0d303ecdbf489849be8e630ffff0603948c87a7e58131655c9f407708e8a9d675e28e9b57729f0346c0287f43ed67f9c0c0ce1542984851cc3b521afa5b9b8fa53680bdb2d73c2b6b090ef085a7e7c6f76a2e501064c852591df60439a96dd8d663b564c9e5c253ee8d8ee58ab27d8332113bdd51d8b41ac73c143a
" ~k:"05c7a20e6e4ddb833c4e30a564436fd66716f349af551e9943bac61572e04107
" ~r:"7689b5249f1943e685095106d3f68359cdb76be5d9a50ebfdf36e731575f8bda
" ~s:"049da42de51e617cdcdef17cdf6059345b8e181bac64c47123d47b5efe105ebb

" ; case ~msg:"b1303768be174d83578407dde1ab91cf021124a34c4a35eafa4512707a3660d1f884fa6c3d7df299598018dca22f273f602bab371592b11f4574885741ab3fe2af5b71237d0057ae59f37b61dfd1ad5ea27cf8f05f5b69f2936ec79d104f4a46c902fb6790dfdc75b9768cc7dfbae011c795e646f9a234728707fb112c461007
" ~x:"247fccb44c2c0cb1f1e58d1033eabd203d8d874d0bf18ba70f04b75bd6495bad
" ~y:"5602dd579fbe37f187d49d76fd5936fcdef2369f7af29da43c6456a6ac8317b39e4cd679143a4d97751b80ce1cb45186da7bee991e25eb9a1aed1490fd74f6ab507940821a1adfbc30e19a933cc4d21769ccdfc57c96f0d21944f8a0f131626ed013b3e5c01313a1756b67b7d2a21edac486cbc3cd1d2b6fcf20c82dd70b4f72929c1499ad796de894db8af103d9b91c25737073d9df62e6b624b90fb352db781c7f2ff8d3a20a7063fb51272395cc7d35ef79c27b7634e39f74eb152975fdf3b903c23990eede8aa58df9a29954333a3f525d5baafd379dd57fe396a51876f25d9e8265cf6971edc6278ce996bdee206883448af184fae23af2a69572b20090
" ~k:"0b94ed40c05a4ef445309afb5583cba8d411ff4092452c0a064dbbe6e3ccd1a5
" ~r:"1800b6bd5c94a031d977b9d017541790a9fe7e414c90fa4d3803d56ef16a6479
" ~s:"07ece1b64711c9b3eca489e75f2e63438e097498e2890dd0273729a55df0d2df

" ; case ~msg:"25ca3dc8e6ea4ebb936fa01b1ccc08bb1de923be6292421ff9f773af9cc7393510df2fcb6ec188b27c2688c72fdc2ff6c90f0ab0ed59c9c3a6503f53e32778b954eae582c95803c511ff3918adda02e68e2c3e73f8a6ad607a89d8eba0059eb87f4d9b0081f296961ec6ea78853aa53d24a470a74acf16a2f86748a8da34fb90
" ~x:"32b6f7ce3ce99770b888c1ef23a86377f3e00adf5dab2e380ef8c4298d20a1ee
" ~y:"bf2e140f8b8d99d2df1052e981fa0ac533c0d4ea9f266f9267cde7ba03cf10015da1cc13612dcfc92030b7c7d1c057e28a6fb45748eeb9c4bd2e6e79b217f4b68ef03f9659c8e84a20ee920d29711381ce39fe0afc9a7fe2fbdfce63249651230f3e72eed579f0d3659c2bffc70fb5d8be889a34bb67f1a904c318568394b946fd40383782cb5e4809d0c6019d20afad09f29fbbc994d28f4e41daf4666298f351898d8def404712c409745a88962e4a618c234976645559c90c54fe764eea46fa03543e4c4f25c8d2c3c1979f952458177dc6963e3f346a7fddbe0cdf23ddc7d2fa8a3455cd5b546e47169912ce7f333ac6f01e64aec596080b5d3e0f25adb9
" ~k:"73418db52c6594dd0956d9e3616a205de8204220648addd4bfd3a9fee412462a
" ~r:"7b1dfcf39b624d64db08a3974c8e14173105010f2bd5135e926f2884e30b46fa
" ~s:"697eeab669677469f62cca46d3e68c849f447881e2c9f74294f4e8ada4426c7d

" ; case ~msg:"d58a8f5ab44f9df9ed936a1318657c324fb1399c251054986d19214c15ce951f87ccb3510aed9085411d9c5a6740df5160f3e57ea8c942d33547317c7a387c60c7ac2f0e14171f0b7719aba76ac418d157a4e3bec6b799b5da10bd3ecddae0857a29670c99d37810349b82b7bb37c0937b0dd2734da08b8b1cb7beecd43cb615
" ~x:"23a1290f8acbadc352a282015713d6cf5a88e8901cb9588a57151772619f5ae6
" ~y:"baa13652642d950d8bcec16c624a07999fb557fb40a266297c15659755fd615cc7e2125d4e8c8af8c43335539005e9e2f2d40428e7c8cc055ff3f6fe3b3df604ac128d995cfb9c867e2a9607aa3b77cf0f691738b784d4be2fea4739fda1f0674260f21f666acef5bd56a7800bbe950792ba05eee42e80a2578d2c50ec28d44afb6b687652bb9452408fcaf257c4b5cd564ddc4e63ce9ca13d4c7473f51b01ac8e4c3f799afc908eaeaccad062b0f97d958a3008cae22062bb166c7300df0b4386baecd599fa8b083fba6e7e4e5ba119860268517d79ebdcbe02437bf4eb5d91a843725db0eda66eedd46d66b781aced0dcc23154e4b8a8f0453b2f466033bd9
" ~k:"25f8923843d757ee4b7571b42de58925b0c2678ec89df07248b4cf34d83db926
" ~r:"1876b20926d8ede78d28174eeb4cb0c1af8ee206fc8db4a8cdebb5dbfb0c15cf
" ~s:"231af07aeba99ffd00659394ab6ed19a5e9f9e60e2197f65fc88c815beae7fe0

" ; case ~msg:"aa134e9db73982e7a37a1034aab82b50d5e58e034a5637081dc880a6e265ebc7b353df210304ba00771c5bab445dc6c24999fe8eafdefabcdd46f7a91f30721a6896333c3f301e197f961944f545e4fe0730cd967790504c49b0ab5b890809be5c7c1c3f8a2e52d92a2c199b981b648fdd528e768e6ab392579b54c72c41617d
" ~x:"02ef078e61df318237c9a217b5ddbda12ab9ffde68a201971782b61b73214cae
" ~y:"691dfea144e51b9e0ff7536557b58ace8716263a70554e2f4676d172332aedaa67736d72667d328170aca070e1bb89868bf4cc98962d87eb0599f10828c6ea24cffede8ed7b39abba666bd6d0d35024ade6aaa06fe6ae45dc4b3a91c219d472db0efed469d69cb5f11d40158ea81672b1ae116ff2c3016f245254e984a59945e4e3b3d37ad12058d84082955c768643e7d80c055c1703a883f2abb075a24c2e93056697340931c25894d1d2ffac4b1022012c15cb707fb359683ad0408b668779e9d9ba21989baa6a6b0b256a34efb4751bcaf4285b15635d409fda993c0438acddc9da006c390360304ab12da76b444d64e11ccf05d963ffb7f389bee831dc7
" ~k:"013e35ddd416e092335f3bb24a5e826e3e06cb90daad599a42cb5ae8da830b24
" ~r:"041d229349cec75fb2bd8c35c249f9196a18962ca75ebdb42dca61d21cb0e910
" ~s:"77bb7975a544c51bf249dee2359523072863934497d1a479d6e4b245d456eb2a" ]


let sha256_n256_cases =
  (* [mod = L=2048, N=256, SHA-256] *)
  let domain = params
~p:"a8adb6c0b4cf9588012e5deff1a871d383e0e2a85b5e8e03d814fe13a059705e663230a377bf7323a8fa117100200bfd5adf857393b0bbd67906c081e585410e38480ead51684dac3a38f7b64c9eb109f19739a4517cd7d5d6291e8af20a3fbf17336c7bf80ee718ee087e322ee41047dabefbcc34d10b66b644ddb3160a28c0639563d71993a26543eadb7718f317bf5d9577a6156561b082a10029cd44012b18de6844509fe058ba87980792285f2750969fe89c2cd6498db3545638d5379d125dccf64e06c1af33a6190841d223da1513333a7c9d78462abaab31b9f96d5f34445ceb6309f2f6d2c8dde06441e87980d303ef9a1ff007e8be2f0be06cc15f
" ~q:"e71f8567447f42e75f5ef85ca20fe557ab0343d37ed09edc3f6e68604d6b9dfb
" ~g:"5ba24de9607b8998e66ce6c4f812a314c6935842f7ab54cd82b19fa104abfb5d84579a623b2574b37d22ccae9b3e415e48f5c0f9bcbdff8071d63b9bb956e547af3a8df99e5d3061979652ff96b765cb3ee493643544c75dbe5bb39834531952a0fb4b0378b3fcbb4c8b5800a5330392a2a04e700bb6ed7e0b85795ea38b1b962741b3f33b9dde2f4ec1354f09e2eb78e95f037a5804b6171659f88715ce1a9b0cc90c27f35ef2f10ff0c7c7a2bb0154d9b8ebe76a3d764aa879af372f4240de8347937e5a90cec9f41ff2f26b8da9a94a225d1a913717d73f10397d2183f1ba3b7b45a68f1ff1893caf69a827802f7b6a48d51da6fbefb64fd9a6c5b75c4561"
  in
  let case = case_of ~domain ~hash:Digestif.sha256 in

[ case ~msg:"4e3a28bcf90d1d2e75f075d9fbe55b36c5529b17bc3a9ccaba6935c9e20548255b3dfae0f91db030c12f2c344b3a29c4151c5b209f5e319fdf1c23b190f64f1fe5b330cb7c8fa952f9d90f13aff1cb11d63181da9efc6f7e15bfed4862d1a62c7dcf3ba8bf1ff304b102b1ec3f1497dddf09712cf323f5610a9d10c3d9132659
" ~x:"446969025446247f84fdea74d02d7dd13672b2deb7c085be11111441955a377b
" ~y:"5a55dceddd1134ee5f11ed85deb4d634a3643f5f36dc3a70689256469a0b651ad22880f14ab85719434f9c0e407e60ea420e2a0cd29422c4899c416359dbb1e592456f2b3cce233259c117542fd05f31ea25b015d9121c890b90e0bad033be1368d229985aac7226d1c8c2eab325ef3b2cd59d3b9f7de7dbc94af1a9339eb430ca36c26c46ecfa6c5481711496f624e188ad7540ef5df26f8efacb820bd17a1f618acb50c9bc197d4cb7ccac45d824a3bf795c234b556b06aeb929173453252084003f69fe98045fe74002ba658f93475622f76791d9b2623d1b5fff2cc16844746efd2d30a6a8134bfc4c8cc80a46107901fb973c28fc553130f3286c1489da
" ~k:"117a529e3fdfc79843a5a4c07539036b865214e014b4928c2a31f47bf62a4fdb
" ~r:"633055e055f237c38999d81c397848c38cce80a55b649d9e7905c298e2a51447
" ~s:"2bbf68317660ec1e4b154915027b0bc00ee19cfc0bf75d01930504f2ce10a8b0

" ; case ~msg:"a733b3f588d5ac9b9d4fe2f804df8c256403a9f8eef6f191fc48e1267fb5b4d546ba11e77b667844e489bf0d5f72990aeb061d01ccd7949a23def74a803b7d92d51abfadeb4885ffd8ffd58ab87548a15c087a39b8993b2fa64c9d31a594eeb7512da16955834336a234435c5a9d0dd9b15a94e116154dea63fdc8dd7a512181
" ~x:"853f75ac81b3a842c999448562c584d1cd0277896ec2f93c05c337eed414367a
" ~y:"356ed47537fbf02cb30a8cee0537f300dff1d0c467399ce70b87a8758d5ec9dd256246fccaeb9dfe109f2a984f2ddaa87aad54ce0d31f907e504521baf4207d7073b0a4a9fc67d8ddda99f87aed6e0367cec27f9c608af743bf1ee6e11d55a182d43b024ace534029b866f6422828bb81a39aae9601ee81c7f81dd358e69f4e2edfa4654d8a65bc64311dc86aac4abc1fc7a3f65159661a0d8e288eb8d665cb0adf5ac3d6ba8e9453facf7542393ae24fd50451d3828086558f7ec528e284935a53f67a1aa8e25d8ad5c4ad55d83aef883a4d9eeb6297e6a53f65049ba9e2c6b7953a760bc1dc46f78ceaaa2c02f5375dd82e708744aa40b15799eb81d7e5b1a
" ~k:"d41b335753e1ff3f828f57b797ff5b2db5cd79f6a1abeaa137a2a830e24ed4b5
" ~r:"bcd490568c0a89ba311bef88ea4f4b03d273e793722722327095a378dd6f3522
" ~s:"74498fc43091fcdd2d1ef0775f8286945a01cd72b805256b0451f9cbd943cf82

" ; case ~msg:"ac30fb155104954b9d7139de9346d54ca05178954053fd361c9719cea530d2d2e1737fc46b0ee27357cecbd47e0fd47ada0d5236a9d77dd61a1b0db52e628b14588fdba8774882866b04b49cf5205db49445a8a202a5fc3fcc36efe0bd0c1e51eb08616c4a7afe120077ea08caf167e90446862298011ad9a1f11cefb5f74335
" ~x:"d692d2c653bfcab2e7492ec56e512724c912227d793a59882800d37ad260bfd9
" ~y:"84741bef3d9f9dab0e3fae7839d39c1a1966ab82798d71aa46b7def465e39ea5e7adaeed2dfc92c9bea72d65268b8df955f9b7e7b6923d2bf00e7e43f83a0e54ca944275dc39c0fb0c8a00ccd0b29b790d9d8f3496054390410b4ae5c6eaf2e21bdb52421179970fa13e0948280a06a576cdffae6fdb239ebd486bf4699270e2bc0879be25a6a0c2f7280ea33eeb32c5d2ea6093381fc4c83c8f9a591b0b0e72fcc149c685b01381a74af4ccb902c0050e05baf732bacd1606533e2acc6308c777201eecdcdcbe935149c4e572a15a205d2b80e75ef2473160f85e642d28370c0f19464125c687c969665b13b095aa87ba476802d72c354ebcbcd89f28ef001c
" ~k:"39335e9193222c7ae3caf8e5ad77b751e9847c37b9016d355ac7520407c91e87
" ~r:"28c6bfcadb5f52324e39903bf7a04faefb89383f473daa432cab9178f2470d3c
" ~s:"4e88f65ff776940bafbbfb35643bcdaeb43b25b45de2de3c011ff1449c8b8b32

" ; case ~msg:"2225031fd26a6bb4fd9990347bc2c8ea4ba45bd75df68476f983dffb5531899f1317d95f7cbb493de45cd2f11904cd5c5d5a748b4aa127ca730f89a928ddcd250a6551c2f7cce109e64d3ab74afb2d4f4f7e3494eb7d557060a1f29ecb5b75f64848370902bd6ae2fbf6802b2f9c37f34836ad71dd2e2abf6a0a47df4fd5573d
" ~x:"87bd74c5d70a292914d96b47dc5e9e97a6799c3b788014e7f106ce7ce7e17a95
" ~y:"04964c093fdb852c97b165e179f7ef3b39350c2588e60a0177bc2e890ab08ffd73d8a5a6692cfebd0c912de2d50bf02139bf017ec715c2dd7be1aad9d0b96c47d6465d4eb0ea0247ff655959d94a3409e9f9262d877075f6f0c7783a8df3cc115c5287c69bdbf0ffe0ed3719e418ff99b5dcd5f0cfc1065e404a216e095086a6e2197a69c47774377203d99a234e7be61cc4a95a809f9b9dd0a550b712bce5d1cfdafda232d7c831ec52884701155a3df2b086be870af8e875557518b035c84957c1742b8c02b0d46b64a773012809bfa4c5407c3fbfed3b960816604cf42b2defb4feeabc172afbfcbc82836b44b927e0cd4ca63a1daeb3eeb30d1de608127b
" ~k:"64f504110193cc4a3f400b6fcfd71d64a1e166c048829d23206da12a7dc1423a
" ~r:"5568d810ba664a08b301266d08c69eacccec5aae870a6d579eda51a31b184655
" ~s:"9e818868e06787fb9519b50546ee21d0546e16bb1b5920311ba44769dc69c7a6

" ; case ~msg:"4b1f9335fdfe88c0866bb648c05857b79c2fda92a987b359282bbf0822db747a3940fee05aeb3cc081231e29b9d460ef30a55f0f88702a4ecdcb842beeb36a976136c9241f2eb5c2d93fe38a1580cd58fb93ed137a7d05ea22d5e87345633a0e393feea616eaf83684c3baca4fc5bf80a87dbec3a9787daccec479661af0b968
" ~x:"afa080287898b0787f5d06d5826cc285ae5bee41768098750419a5c8863ae729
" ~y:"57767c348ab0c61eab4f2e0894bb6223645a331c5be2490d764839fa4dac814e05e70925d720d0e0ab5faa3db6dc58ba573b4e0b7bc13e4c044b96259385fcd1eade0d7c5174498c70ba8fb8661ed524fa8171570fd52faac9915d947b51f6cf5b74e3edfa064a5161c7623ec6e80d29960b573fb98de9e710c56ee45aabc4022357f6c3712962ad19e43a4148957cc6b9c8f691877a59f43162d8f98f2472699ea510109305f8f98aa3f3f31e4302eb05e5f1a462d0f3bfdcd0c84e76bfdd14b7c90b982b8c0ec7c78cf3e6c216ed1d20b52a132f53c9747c7fbe39092d5ccfcc01a119c92faa3f13d4643e5db22ca1681d6536bc7b704bb09bf6c621c2ff06
" ~k:"d23656910f6e8ea72cdb979cfd8c8f6676c47c6161c3aa14f2338392891d1afe
" ~r:"7ac95d3e0936cde441e4a290711cc044e6e98e8a8de68298bf7fb90eef589eb2
" ~s:"140e9de37ec5aeb3fb795b016f51ea3e92d6f198c5a0e5a5d236671c91042c94

" ; case ~msg:"3b87109bf21571fcfae92b859649bf37dd23d59f76d50cf26f4b2ebf7c5f4ae0b377bf3bf2c7e015a74efc808433047a71bf1ed4ba9025f4561dcb94bef2c2a2c94b3f55ed611c432f98a683abadc2c31d002eaca9b070f2b21319d072df75c62385d7d02897a00f863c2882b2897a331332bb9568b2fdfaccf50b3de4b42e8a
" ~x:"668606f4a82b50876abd7f3dc0ed580a10344c1dd092c5bc1b26c427028cc5f3
" ~y:"7c16a9644c18257911b826da10b5b10115ff77675bdc3c9f77097162fc059e86b04c1faeed3c66306c7e5fe2d5c63e8fa5fa2b82565ac6065445de5819a2e4a56925bdcce138654dfb490ac624a38ad65849be4ba74d14c829ef102248a18193933335eaf0c73b7bfe77d669f857ef3addb1f4ca424dbfdedb9e2de1fc0cc2d9777ee834a0ac7d0cac1b2a613890071490efe5cb2097ac830fbc27881f9fa51d3b0247c5e1b7f6be13c30dd31c2c59b7683ce60a0ebd6663de97870af2dd17d91431323a4686bf32e1e39732dae1300c57bd600be790593b2efa045bbfca956768157b4724ca0a1472fe6c8dcd82a38024766341d1f548ad8f36dc676676fbe3
" ~k:"a3d781e5385d66989b38034171da11594b20f15733fd4701a63cf24bb58ec341
" ~r:"1e219eefd616caac549a859d45186b5c528627573958fe55cf57fbbd1661f7b8
" ~s:"b09545843dc0f6299b48f14311503605502868859e8c43867f80df3c2391c762

" ; case ~msg:"042365b1256931a111facc6c40f618c428801b03e4f222a1e1b7763c3b02a6214e4c517beb587a4ea69fdbd4ea2d5d5f45afded96ddac87dc89955613aeff7644fc6a58bb859a85221318fbc5e175c6985b19a1d16ab6ad3ca8fa1903acca42bc6d9efbe88fd6f2a8650425be97bab9cb670b2e39f36d526278e0bcfcbffc3c6
" ~x:"1c08570d1e1ac0857f649e4ba20de0e9aca97374acba6bae350104f1fce20be0
" ~y:"815411ac6aa1b495c4bac802806a1a3592924fd9c0a3cca41e076db293d815c2f2b0a53e97cf657c8951b856cca1166ad433be5829b0b636ca9de49111ce5ceccededf36d795edefefee1d553250fbcd5bd05b4d99de55f147773ab3a0f754d090ca7b6ff75c160eefd1709a5df3cd8a0cae3e341f2275faaee3e3e31737e7e9c7e74845651f4f839c9d08da6bfd00f2c2b9c6ed9acb78d11175fa6ded7ab95dbb2bfef18feb149bc94f6de05a205221ba0406c96f63972aefec1beef030137011e6796af2e4ebaa100150d58caf408217acb1183a1a46e06368cff6fd744da7019e7ca109acf1244a763cc2b2186f49272ba3ae0425f2ebcd30e77e9f7c957a
" ~k:"0bc8f6e0b01bcb55a4d134c967f3a9411737103d400a33a968f9036292d6e3bd
" ~r:"e7145c70e0038ae7e7d901b48828b0b8bc960cc4fa29a52e11ffc9ab08eee726
" ~s:"b9c54ef6cb3e1b0498952299d1465ed2c5d4e670cdfd2506462466c3b0fcc538

" ; case ~msg:"98ffb2899f17c80a83e82ca6265e6f361733a6bbc63cdf8880dc756bc768b35b90db7390cfff745ec1b56f1655d8d9a29a6e8a63be0b1b2f9aa7436209a1fa061a7aec28622c472b3d0285a701655a496546e891a8ab29d9f40d2e748d0aa2babc06cfca641b300b7a219caa9e5bae3bf689f60567f922e7796fe47bb72ffb64
" ~x:"79885ff112bdb326577abf52db67784768742b36e575f06b8d1e4f0d2d49a3a3
" ~y:"14111dca30c0138761fd2f55972b9846041e5ca8b9bc6b2dc820f2a2f5100abaab337c7e0d1bc59de5ae586bbdcf4d4b14aa23be40095293123badbb11919b78cd6412548d9f9d15f614b6928713344148fd7d30985fd2c509b44d396c5672a082de4183fee03e45a90eef6a08b0d9d47132c82a2ccfef05e2ad0f340dcc06d9e2e979ecc43844c6054e4fa5fb8a73a1e3873f2145b0fd40f3ec7946f1f43de8b8057c1be5bf04630a12453d623c9b8d9f0e30c88c30434215d48f77348e6b047f16934ea09743dd3b009cebc49dbc3a3d3567c3321555ec96b2160caf7870970ac3cd8294477a0643ad52c23d9d987dbfff64aed1a883c30a49f14ff0620095
" ~k:"13ab2945ab2a40067a93ed8c1a4b305182cb070022b79a56740238e55b07e8a2
" ~r:"4551b096446db6761b708f35209edb91cc51ee4ef96a7495407ab4167a05c791
" ~s:"cfe4c58bdbf61caf09a42adb1aa5d98b4c459c0112c57823bc15b5b990d92ff1

" ; case ~msg:"5898cc0b422bb89f066dabbd30f59e9a35a392bdd7ad315ec8ad32b8f0f3d02864e70ea36e9076c395f0ba9de1ab6080df3cf4a1470e2b9990b8e7614bb8312b075c0b2a132d7e47ded9e4c0a1368455b9d1a67bc44af2f37428f48f7e089ab41d046378b6d48d9cb135eee4574072abea93bda7eb4f15a206cdaf3bbbebd318
" ~x:"416634f9b7722188c2a5266cfd9baf1bdd508c0c068010fb228c099fca7cec11
" ~y:"766d7e4f8bc3254d92cf6a64abd504d01cdcf6c239178b0aeb3f69c9bf202bff7566eca09f29cf5d6fa4736d57c08205500d648336409df06e7f2cf99178b20a7ec2b5124bcffc61adb66f6fafc51e32521dea2124e5781c383b116d06a6a6e89dec46b5e4ad69f5a1e8dd7ac5e160da336c11860b601e7e6d58895e6797db5aa92deb7b942f2edf58d43d3dac9209557a6aa07b228e73a80ff0e92e4ec4603d362e1cca7e928d9459c21405aa0f6548732c0fc501ce50f0896f0763f633c8c1a8531321e1a0f47134a0d2d8676f45f13ea576e64c7870028033a4261bdfcec948ebb1aa25b02134d0259d73024a01da0cad1ce67571e36963dc130496160ebf
" ~k:"b057bf5a5ae4204f941ff5a01560cbc29033dc6a2e06ad168403cbc6512646df
" ~r:"a237d2c3d23706caf004a2e94de29f04c748936b62ab5431fe73c72485814265
" ~s:"b48b9ef9cbd8bdf799b70605f00550b81b309c157332153be9707a399fbdd67f

" ; case ~msg:"0418e01236caed0f80241ce8c6307d026f5e25f4a922bbdb4aafb8d9db95a18175f9dcea9acb4d376f36ff7b7cb598e073de95ad2012eb9d11e15cb3941c6dd0dd69422e78512ebffb19cc8a403a9a7d1f1720ab0f2d25627580366093e21ac1537f93de90a94508f1d7a7a1db5a7b13c9fd00b82be044c3a35ec0451c309b82
" ~x:"6ad92911dd4fce033d7a50875e4660be08c44957b874339c2a70d915cc03e27f
" ~y:"4cf4cee4d5abc2c92db522928b6d7e436ea00884009497ed588e93281cf05b3747ca0048b917708279cd0277ce8560c22775d2aa0e7eed1bba77be45417fa7afd776b8e560679c493a520a0e626acdc83df021351669bdf9da19b12bef2926b525fa4c8e3d1f2083ea6bbb489880f594e67934d1f35581ad18e0db462a1ac944066c65dd743f35741c6cf588918d8336702329c62113e9486bfa49ca5425914526a965e3c1975824f4b39fa5fef89cf6f9ea512f7ffc9138e72dbd0f71b01a70975312eacab11118471115ee3fc810522936c9df35977509b196d867fa11f607b7ef9ab78cb748213a6763439ce5e7641b05359670612203a47d4de9c5388405
" ~k:"5aed2d19239189cd0bc8cfa3c329748c0555d8eafd5e80ff931966680e3ea454
" ~r:"83ee960e6f9026fe2454d859462ac334a13896e75179858ef40e2e9a065c536a
" ~s:"7ce8699c6ccb184d4240b8709da11451328cf1a7e0cafe6e1c8ab53d7de67d9e

" ; case ~msg:"92c949fe2342f91a387b67c1b12b1d04d0721203caed593c9c464e5fda09fdcc91d3321d2985eec08ab2026d1ec3fcfa838cb6af45290c08dc30b9c14c4445d783b6f48409a00490f4e308dbc87fd1b2f878385212e1f4c3e1cf81c56d71e73fd7a095b56b4abec15c57107420fbdfa44477078ccf4519f9f6044f0744052035
" ~x:"b0d23aab4d95446f8c6c5d496d477dd9486f50b2827f7cd19728bc96f82840c7
" ~y:"256d231ac2bae650d25999b2706d4cb63a89b1468e0df36d677535fa7a0ea890590d3290d4b50bdb399f33dc415e4469c97c6c0cee8205eec962d7153c4c85ab88f7cf80979d4a1ffd8c74e681c1d28da07732116c3210ee4b693309333686246d667074c7172035fd6091b2840b113970b45983d474f54b95d26394b7a43e81b449a2ee9423aa1c27f4592b516c12d5433e2ba724f5463b4169a2b0940e1bccd60ccab9b5a38248acb60582ab8bbc01c5e75f9ef7474273fb51aa6316e649f4f22452dc70bfd4c3da072c03ea82ee009d4272a84961c98e517ab947741d812116011dec0373ca8fbac5576c2069b067f8b005d60a36eca44f56019a64835d76
" ~k:"17a199bd383a84e22029fb90d5abc9a8a7ccd3f0a33720ca80e3161971793526
" ~r:"84cace71a80ed47494570fc84839f2e350191b74f0eefff2d7ab2c689db77bae
" ~s:"9cac33594e1934b68f62aca05ca040f3c82110c10b7379878b7894b0919a0f2f

" ; case ~msg:"df6a4eb7cad4ff9bdd8356d3568fcf0285c1a4e3c3109faa091b58a9bd907c629d54aa7a23a74870545a0942a2d23914f2f167d96573f06f35ea05ef704cac8014dd21b961d3dacf7b930bbd7e35550f721094c86333e03ed4dab7bc1b6416add9578d279edaee37504fd25ec0c5e8a37ac9ec19bfb1e3778ed6d9c6b6e35ec7
" ~x:"679ef48b643be394677d17e837a0be6d4d8027a900b686aed7c4b12634fea76a
" ~y:"2f4b0c01e4b15eb5ee7afa9824093330738be2f3f06c42b2b7c6968fa54b987c184e7fa89eff16da02b93ff61b9ce48eebe7eab0f7e203ad11c71e7b297d23f2d5a5998272c30c2e1724b5e963bfd6f83239f874d88ea089435b896dd2109b6a14b2d848f9ed7e92143c0649f97f4f2eb05b8c5a07e99e497dbc752d443eba93d7f3dcdc3240a2714ea0e3e7627f216e4701148dd21192f274f1ed5df05c60b1576d3a0b7f69a776b5010404acd5afafd3d70f57763f2b778d0c361e5f7f0bbe17aafaa5cd393329171d06ec032039a9ffb37c3ab8cd858ea788a7b9f501996baf959ca85c7dafe0cd3e30957640eff105894c43f866bcc422698d128dca0887
" ~k:"3cf9da6f182bade870946d3ed3b078208ea8153c45515d64f589bca72b703ebf
" ~r:"1dd2daeaf3e89fd644c6cc942311ea5056413d8a24087787675ceffd3d6c15e4
" ~s:"3e12781396558560455c4e70f610522ab2b10fc25343296818ef7ffb0378fa47

" ; case ~msg:"4f16681eaa5d97673a7cca02ee8a7374b75411e0b5704a947f04d1a5b14be0b506f31c2fa329e3ca516fa4f1626a9b5e080bda7f353f850365eac7c3d2596f502a5d70b1542276c12d4ea4a22b5325b9eb3e942e556769b796c4f524595f1cc6ce17f99f9dbf51331453228ead327b614f4438d35d61428429f78c8c9377aaaa
" ~x:"e0e576432913bc75a2e0de3b33bf3094148298612a9dc56fedd47aa569af8ac8
" ~y:"2e33604ed9e6c0f1ba403a8c3c3fe8e8f488591813aa3d2fcccdf88fe808f70adf173f0f143abdaad43b80769e30ffc5749e8ad35999953deff4f61f4ca07313609e23acae7b35f77934fdbbe1c380b2727b1c3899250af5b4399b658b7908676d64d11763785373b2169836611d72a957319936c84efd72b72f92bdd2dbe0000d8841ab6d8d0d666e79361abb23b6007348dbbe7a94936dc6b026f3b7100081f547b994e0e0778cb761ebd43a29d8764c7f962a747ecc92e4a2a628f52d8abf43f6e3278a0d32ea67c2d79d04c83387ddc709365c0a0bacc83d75c946e283e0739233581441aeddb0d7d76503d621405d27ef66fa8b5379d178617d4bb5ad59
" ~k:"53cb2d046b391193efb14a4dfafa296c2ec92293c7b3c7d19a20e68c4a1141d9
" ~r:"e2ff3fc441db4540194a7f5da1ead849c2c3c48dccf8b2c1b3b359a7b16e16ab
" ~s:"52fbdcd5c62a999aab46147fef9e18cbfc7daf680a7ddb892edfa44d285e2158

" ; case ~msg:"09e48a36523b5289ec41859faa141e2a29b3e88ab2d6351e20de001e6424b85337675f0ce26be224fa4f8df0ef9710ea285635b27b297d688e338b5461820b57be4bee21645b04957ca2f6cd7af9a6a52b3c97c5b9db1c2f7ea817cd6d3c8522d4e6a9de869aef26ec0dbdd269c79b38806927bd3a5100735e6f9f655ca94dae
" ~x:"7891b05e24823f283126d7d175a4e8c8124b3776f4f296d0ffc4b5e21bb64d70
" ~y:"7e38cb668d647ee15f71ac5d2b55c11fd4731e1a6c031dd7594d614f2f1ed25623fffdc5956f5256e635c914205a2937a6074cfe1f3e443bbeb323a23b0f0fbccf8c1770ad18ba97d0acbea1e846e12cf12c370625b1555d710905eee943539f2241b8fb490c9d6b44f36139226b4c1f00e95ffe595014f61bf579836a14212c07231a5e9e87de4a9aaf0f46f34c9229f2eabb71d40de26a1cbe10db0645cec37d48575a1154bb5acc947becb2a74b07e2a0e45b903be37502f91b07fb4ecd7f21fb130c6d639ef0fd8444fa12de859abe95548801f6a3c40e7a65fd1518221a274d7b65ed4175f66c04d919c86d2ae8c374b14709e9c8a39e1d0c4e9935540b
" ~k:"dc24b379ee2d26d5db792839795ad0d4b9622c0e3fd518df541a5f6e9cefba0f
" ~r:"e550dc65af275e47be480fd647366e2b055c79ea33ded4f5a9557121e082af26
" ~s:"e26b1a5f27cc6c87863e31ef7f1e61bea476fc5d7c25fdf22fe740f23aa9a752

" ; case ~msg:"8837bbceef577511f2d0c08f790d5d2e8562d93df3d82dd4c2827cd9a9115308114a18c452db2785561081eb523685ae2b3c8b090e0d44dd40d2fc0cdfc88d6f9063a7707df609edf0a8c55034815ea9f1d8b0bcbc92fba513ba81ee646bf98ad4eb22be26a4582b1be2899c91eebcbc9fba5825e021e99be0c9d28642d13fa4
" ~x:"116d1836a131310644aaaee6ac39b3643cd50026a6b486167cb4daac242a4e7a
" ~y:"77d7a40a7bab3f5778f85d4fc48b3e28ce28b2df9eb87cc9cf394ef28e8064f39a9690103980a66da219cb5022c101f22011a8157a7568c5ff2e978ba2201367d17c22a867865d00c2a437385627bd088bfcf7219251bf6ae158269f4ef35da7095a53c24f37d61bcfb7c043feb6e93832343f9e90ee7104c80486ecd087be1b67f18cdaaa375e039cb7ad603cb0cd855623e9fb48e4eede14ea3c76a0364aac006650d3b5cd9b474b56f8584be58a721bf34dd0808d334cd8632e808536791fcbea961f7163dad28353c115eb3e856737dbbee03436721637a47754a8a1fe0fedf547b358a73d05b769a95bde3440007c0773a3c7c8dc9714e11c3a10ee01d7
" ~k:"4a6febb624c8ebd411cfb30c6db055dec3d0d17456dc0c54bd1b43531d4f2649
" ~r:"7b6b3eaef6cd5fe6daede86d63943478c771582483be0b926ee3022d22ef912e
" ~s:"39d928b59a690450d13359a29efe20cb98bfd3fc9726f80e5148f059663ffd08" ]


let sha384_n256_cases =
  (* [mod = L=2048, N=256, SHA-384] *)
  let domain = params
~p:"a6167c16fff74e29342b8586aed3cd896f7b1635a2286ff16fdff41a06317ca6b05ca2ba7c060ad6db1561621ccb0c40b86a03619bfff32e204cbd90b79dcb5f86ebb493e3bd1988d8097fa23fa4d78fb3cddcb00c466423d8fa719873c37645fe4eecc57171bbedfe56fa9474c96385b8ba378c79972d7aaae69a2ba64cde8e5654f0f7b74550cd3447e7a472a33b4037db468dde31c348aa25e82b7fc41b837f7fc226a6103966ecd8f9d14c2d3149556d43829f137451b8d20f8520b0ce8e3d705f74d0a57ea872c2bdee9714e0b63906cddfdc28b6777d19325000f8ed5278ec5d912d102109319cba3b6469d4672909b4f0dbeec0bbb634b551ba0cf213
" ~q:"8427529044d214c07574f7b359c2e01c23fd97701b328ac8c1385b81c5373895
" ~g:"6fc232415c31200cf523af3483f8e26ace808d2f1c6a8b863ab042cc7f6b7144b2d39472c3cb4c7681d0732843503d8f858cbe476e6740324aaa295950105978c335069b919ff9a6ff4b410581b80712fe5d3e04ddb4dfd26d5e7fbca2b0c52d8d404343d57b2f9b2a26daa7ece30ceab9e1789f9751aaa9387049965af32650c6ca5b374a5ae70b3f98e053f51857d6bbb17a670e6eaaf89844d641e1e13d5a1b24d053dc6b8fd101c624786951927e426310aba9498a0042b3dc7bbc59d705f80d9b807de415f7e94c5cf9d789992d3bb8336d1d808cb86b56dde09d934bb527033922de14bf307376ab7d22fbcd616f9eda479ab214a17850bdd0802a871c"
  in
  let case = case_of ~domain ~hash:Digestif.sha384 in

[ case ~msg:"8c78cffdcf25d8230b835b30512684c9b252115870b603d1b4ba2eb5d35b33f26d96b684126ec34fff67dfe5c8c856acfe3a9ff45ae11d415f30449bcdc3bf9a9fb5a7e48afeaba6d0b0fc9bce0197eb2bf7a840249d4e550c5a25dc1c71370e67933edad2362fae6fad1efba5c08dc1931ca2841b44b78c0c63a1665ffac860
" ~x:"459eb1588e9f7dd4f286677a7415cb25a1b46e7a7cfadc8a45100383e20da69d
" ~y:"5ca7151bca0e457bbc46f59f71d81ab16688dc0eb7e4d17b166c3326c5b12c5bdebb3613224d1a754023c50b83cb5ecc139096cef28933b3b12ca31038e4089383597c59cc27b902be5da62cae7da5f4af90e9410ed1604082e2e38e25eb0b78dfac0aeb2ad3b19dc23539d2bcd755db1cc6c9805a7dd109e1c98667a5b9d52b21c2772121b8d0d2b246e5fd3da80728e85bbf0d7067d1c6baa64394a29e7fcbf80842bd4ab02b35d83f59805a104e0bd69d0079a065f59e3e6f21573a00da990b72ea537fa98caaa0a58800a7e7a0623e263d4fca65ebb8eded46efdfe7db92c9ebd38062d8f12534f015b186186ee2361d62c24e4f22b3e95da0f9062ce04d
" ~k:"2368037a1c7647c683d7e301ac79b7feebc736effe3ab1644b68308b4b28620d
" ~r:"4fd8f25c059030027381d4167c3174b6be0088c15f0a573d7ebd05960f5a1eb2
" ~s:"5f56869cee7bf64fec5d5d6ea15bb1fa1169003a87eccc1621b90a1b892226f2

" ; case ~msg:"02bb64d2d5032f54f1ac9e9ee164db83af0cb036d88d41e9b2118cfc39d1b4b4dc2c497549c7982ccacf665d1b0011268246c7c17f562ecba25e265489873e0dd9268e9b06880ba74e74b56f50c7324d29373853e3a0f3ff787eba4e5e7f9437f8ec8a5e868324e9c17fb3d0e12de2d31d438c5bf38b27167d43ae4311b11062
" ~x:"521f08c10774077ac15bc85f2f6a03d84207b4ed7bffecc35d730cdd1126877f
" ~y:"11f3a716fbda7af35bdb62d128af6f21ec2ed4896aa81e8769c6eea9c21c81aef23ae0f525269dc405accef098377f652730968a33b50f0a4c7784345280651caa034df87342ca8973ad86ff7f0f8773a94f95dd2bfa802d268dbf3a2103b1276e06db2d734399f2ab7bdcca097616fc46ed2478e52cef049d19444586e7b75d6a56741da2270f54d2c739ec8db996c71f06a39af2383c611499be0fb34809b171254ef273516c33e17e14048ef2d21d600aa153bcf7377fba9405c6b2e5f2aaf0f2f3467d7461f62e814a2c461e8ac9db0df370e18ec6eed8212acaecf1e7241bcbcbca671060e50c29f966f1ea1e92af6903f81c7ab9ee09f60577bf30c186
" ~k:"08b161571ed031152677136b54e87119133f7de56268aec07cba07667b98bcd8
" ~r:"7a5d2016afe87883491bd6cd166edddf138c1c89961e4af6876be08b0e06ad74
" ~s:"34efbda1849dedd0d1aa775dab2aa2b14c9ba0206592fbc34eb47b844646adc2

" ; case ~msg:"4f1c0053984ab55a491f3618db1be2379174a4385974825fcbe584e2b6d0702abb8298dd9184eef1740b90a5eae850e9452b4e4ab219e187860f0fb4ad2be390ef2ba7d76cdedcaf10aeaf4f25e497b4da951375b687a8d67012d3f99c7b5ca82e9bd0630dffcd635ecd8209cddb872da5bf4736309783345a35376b4fce4b91
" ~x:"6ba8f6638316dd804a24b7390f31023cd8b26e9325be90941b90d5fd3155115a
" ~y:"10e6f50fd6dbb1ca16f2df5132a4a4eabc51da4a58fe619b2225d7adab0cea3afc2db90b158b6231c8b0774e0f0d9074517f336ca053ae115671aee3c1de0f85728cff99deebc07ffc9a63631989a9277e64c54d9c25a7e739ae92f706ee237b98b8700a9df0de12d2124e2cfd81d9ec7b0469ee3a718ab15305de099d9a2f8cecb79527d016447c8f6fe4905c3718ce5234d13bf4edd7169b9d0db9a6b0fc77b7d53bdd32b07dc15bc829620db085114581608ac9e0937752095951d289855d0bcc9d421b945cc4f37f80b0cb25f1ffee9c61e567f49d21f889ecbc3f4ed337bca666ba3ba684874c883fe228ac44952a8513e12d9f0c4ed43c9b60f35225b2
" ~k:"2a4a4e014c94d8546c62f0db2fd488f5fac03073a11c3760376114ab3201930d
" ~r:"006b759fb718c34f1a6e518f834053b9f1825dd3eb8d719465c7bcc830322f4b
" ~s:"47fa59852c9ae5e181381e3457a33b25420011d6f911efa90f3eaced1dee1329

" ; case ~msg:"42199186434d6c55bcef269bee685c4e1580e243027ed128ca99492033a52954bd1ca8ecc5043820725a3c0d71a181a05aabcb4ecda7180d86855e7b4dfa9a44c7af4c98fbf1f0624058804fd8eaae4990d4d7bb75f01741ce36cfc9c137254cab065a4617d0d0cd5f58ea56868a40f3e0baf7db5d2557f4b9775c1820dc1d41
" ~x:"46d690ca6b9cc01e9a8c7bfdedc59a97eba52f097b8fdc19bc1f8c0ab5d4bfdb
" ~y:"6364a35ae994f27703319c36d90793c8f26511846ba06038995b6556e4443aa61eb0f8efcc3d47f7c5f85276ea921da0784a67998253c992975f9e13847ccad099d9c1e5c94cfb195488e1293e23b74db00603e8bd6814c94690bf0cccc1c0e47f0c6609a48e144587ece178f72c8514a43590bc4c219da95cbe8966f4404fe9c288f23cd0f973e77ec84b4b0f163b50a3c556cd1d3951faebd982af44447e60d7834b93b6d9c3ff0961fccb908312a24376eedc508f806668d6617b77491a01d5d069d6ccd5f21b5eb3c3a3d4a0479593845c72f720157b188d2dfae4401c57a600b142b6bde2a69f1a0afba2f507a63cd6df056bb5b34fdfcee012d341b3f1
" ~k:"638e5fd0885f4c9f7e5f4e6a103b2d2d9d1368c493f9822ef431f54e65a7a3be
" ~r:"2551d4f855174f7b28a782b89697d48fbc314cfeb17ec4c9902a8e557cc6f6b9
" ~s:"278b786f9e28eeccd00586b445e75f48cf2649f3f1b7bff72b0e767f3443dc58

" ; case ~msg:"4fdd888756ac68f4c29cd5b1de42756794570ca8f18ff795f6f0fc856772b6a2189b5ed4a9b7547328075b56c28ddf50b84c27205cee57b29d0b387970e89a6a2236293bbc9e399013d1dd3bd5a10ab0d259f7fda704f71cbe3b8b8752806a0c84668d85e4d739cec628dff63371d24a4b14137382759ba400df0e2c25947d18
" ~x:"49da89d1673704d1f24ac7dc799bf006aa7d606c590e5e37e38032ec51a70376
" ~y:"5b619845ba969f1ca5963fcf04c03aa40e989222774e957a54191acf9ddc407a54a161e22a5ac50ca5d61e6601cc7995bf0db38ff0fa1f77b244fe98148c81f208dca29ffa30f1131c76dbbe4303425e9180b4a48f22c757ed8e388b61bdc6d55519523d00c31a5f8376640d4688e60dcc172deece73de28437e900cb19a5311a0c9ca9af6cc6eeb6844e9b8359e3ef1cbe03784107d2d0aebec7c1d70d9385a4d2b8033851f5d5b7aa18ef570aa037fcbd3e30f2fc2013ffbfa0787be6d59ffa1616eed5e121ee4dbee04a9ede004956075465a7688701e04ec9b2153f52cafbff7ff9226e69397c7083c3aa536d7109ee430a65448b10c1818c70510a339c1
" ~k:"14dca45937cfdbca5c799f2ca50de2a44d8051e6d80af242c9f4d614419e6e07
" ~r:"4b90993d707f3371d0a0cc87255e99a8fba18c3b58ddddc1067cd394172366cc
" ~s:"4b2612d506fb85e5aff9fcd56c09bd12bf60f78ab7dfd021a742ff85dc507ae2

" ; case ~msg:"8507db5f1df9d22f447c20e4320f90d9b30722197196d1a2418d06dca41b3305f6fbe52ab58cc0b60ef1a1d257fc2fb2062fe6c5f2a25f0293ca39d0c083cfd5e4bdadf2169ad4ed178c88ecb5554ffa2b53aa4398115cde627d30144ace9325b2d79d7dce951509d734afb0ff6d9265b902672eb5884e9d8acff0ea22c76938
" ~x:"82ab2908e3d2335e07c1002764b07b1ca46d039a95b59b450b16d37ed4838872
" ~y:"438831cb0eb09aab24275454354ce42b9a2eedb31f421219def74687e6f9c92f0b1982355cadb26e095b7ca25de530aaba63e64fc23acc3d1d1f1b70cb726156ca0a799b59094bcc3b8998a4ae7744d215d63b887082f4c84128e74b9b9999c60cad3bc6bb6f727284b4311a929bbd964c9a7074e86062224dcedb58b9b598546ac95b3b434ea114ab0d678541d6caec0c56009bc347a425f167cd32a34eecb7192424d57b0e54b4a9e82f425138703ce89b189039e92a770b51497f8f10eae9c3459ed87e5101f5ab1b6271485fdb2dd3dbc4217fcf67c7e92d0096dc7da9727f5a434b7545284cd8a283070b5a49d711dffa85904311e0345a99147a168ea0
" ~k:"1ea475584982b639ada8c84e51ef72738390ed6fa44395f11428dc5fd794a81e
" ~r:"1d2781f5f9d08ab2feb1683942c2c29a66318839a7dfef9aee9cd7a89efe2ab0
" ~s:"3adc7be968502ead10feec191e212ea0e07d449006e7f22ddf869a9fae711834

" ; case ~msg:"c7844960966584c8e3a59dc37df37b7eb3ad333148a32b86c1ec18072f3b316c59cdef98ba4dc46f532a4280200c225fac6cd1adf0a45382c2d88054e447740454976e5272330c7487eb42a095f7314139938c7419193b1c128054c1bbf10d0634e22c6e02d8e12279cac0bfa01d3058e0f8d5547ba0f71529c27e0084d4bde7
" ~x:"0c76bd647c6fafe7da1029b9bf36a9f042195691a26f36bbe0eca3d4b1e6cbb4
" ~y:"2de9d27f1a030199ffbba770e08aeb1ff3708edf8ebb3a8e664e3bd1511db126ed87bc44c2d2af40b9d512c50a4d6c10b23e3ca61819f5841cbf5d0bd6c88d46f1ac6474ec20b9100b328cc155879166f46b6d71140b0cfb2b0725b64a38d70a91ca8f0e3baeec6125262c52a95d5ca5d5ff6f4482b1825006cd469f9e7f31769a73eddb5f7017f18bc747ae4fce450c4274f4abb960577d13b6a77dd99e67d11edb413e428e50726f7052e53565fa1d6fde91859573c9289289ffef0598802808ecc5501cb300e06405ed0febc3df23f40a1f6532410f7d9049b920216f7d5c7a728c8dd63a8d0060fb53b3543d62a636661750fd43775e80b509004351475f
" ~k:"71e12996d8aaa7cb1e730713fa441098347ca95eb39362c5a78ee6e847469c7c
" ~r:"09e654b17ab775959628e7cad0e27053ee495bcc29cc2a5e3b029660a77b1330
" ~s:"261ad41d6bce6d04d891a43c16ec2a8114e51f0e47b48b1dd1f3d626150338fb

" ; case ~msg:"6f3f74388cc90b29c109ecbda08c79349dffdeb90722974d79d640620949448f66ae673eaf4d4af8c43da673a45ed152ea66fc97166baa7ce8beb666bd57ca43da6801c0ee5a5a9b50c5047935d7a8552c381d93eaf03cbbbb88ed0d3b5a2521b67612a4405120ef0205e89aeb48d577bcda3ad20e0a7cd07f8c9b215c845dd8
" ~x:"34c0d0de98c85be291b68a5b8c7fb3536b6f7447e8565ead9b002417f56f4616
" ~y:"080ca412bd197c5aafa2c6df5933a6210fa54089826828d5496b453609a56b7d55d232fbe650dd9f62c05c050c026a8717a78b5db01614a19301c610d2b9964a7e3357c722a4c553273bf27f871b4b9241678c334e20827a5f511fe9319a075d12753ac0960df60870a08a12f09b9d3593781781a0cd75e9d81cc6b9b0d506d100fe972165b68297e6070db2d8b6ea32176d1562084f6a06e08e2929155b255d33853de6549e79f8b56049a1d02f29166d5f91cfbde5aaf6bcae56f5d2d90a9b4e8f6f450080cae8256c6619e9155523c2b2052255a8f6d9f53d8a897be5b0476002410bf798256f62bb1a81827c2c3fc4ecf9abfd77e74174787370864f05f9
" ~k:"8086cc691e7e793a5c2a81bd3d5a1ff5ae261d9336b33f103d983a817f7eaf7b
" ~r:"43993b68e847f6ba61d5ad4dc8f5ad70dabc317a7b6811c23e7f215f95415ed5
" ~s:"1ea727afdb907d1d5b2337c1ecea46c71eb0fc8363af23865a345202a762a7c5

" ; case ~msg:"74a433c2d313f66232324df875b82563805d7ed682b266eaf962375e422b3abbfe3dce7f3c1960a1e4100f333e168d021968b48397e8cce9005e951fdcb096a9abea342cb5b08bab79ef0c431dd3a43de7d5bd6b86bea8872ba038b43a236a7356b03f89b09004ba2def663e6d299763b6cafcb6b150a57f82b890ff6d56f832
" ~x:"5c1a80e926de194995195c4cee9a2e874c7f6af0fa8a4b2df5432f9cfc86b643
" ~y:"444fafab58db4d6f5283c3443d6478b5b78daa631bd4c3d9a28ed17281da4c1c2ef4d5ed576d66bfe5314e11fe68abffe4df406f6033edb84f36a38a3ce614601bc25841f9419afb2867d991e87b44c4b744e39b64079d9aad4b585d79c8e21c8f90990540fec8ae981f7483dc5523d216088a55cf2380ea8eb5246781290559ea1b208ad4d0f5871cb4d13cdca6ef34fdf2de63e209aa320cdf14185b8f5f60ccf93f398c1a6cf8b3ce3d98daf05e4cf90c39801ce35f01ec76a9f6035ce1b5ba107a5f66cf253b71fba3833e9969c314eb6d500005749231f799b0c79a555a10cdd69f8eec4c117d7c8b4ec6f60a1ee557b70c0dea380af53b92fdde8823ca
" ~k:"13dcb7c12aeb75a417a93a22ce94618716996c3350909cfbff6d38b603d377f6
" ~r:"3bda5b0c9e3da22f0b3e29356a2f7ddace6e9b24a063eb3f5a7d755f2eeaffb5
" ~s:"4cbb815320314a06538d2a6740e6bf9d022eac9aa25c7508f659f0f7c1f59c45

" ; case ~msg:"f4eadfea117fd3d670cea28aa9d2602c951ed843e2e8cb2864074c8c9bccb0606ced83ae2980598cc3e1b047fca8659127406d8f59f5b7bbfe8ece6d3e42f87f4e42ebe92adaa1e6e92ced3dcacc2e0b2c98eade7c9c99da887e74db5a59132c1d7df7cde866cb2f3ca750852ba53e265e62bf7a93fd693e4a13751e186e9d6b
" ~x:"6abf7cc887544bf8d3256fb210848eb46281526b1e8cdf6c9204c4c46a747435
" ~y:"104f44fd7669607644ec55e6ca4096c9a279472752a1753dbb9f2a6941b8122274c87d16f63d75dda9ebcfd6584b0cb374fd17581353d2a246ec0b378de60e9613131683c0568bb54d74457ad73de859a4f02445344d13ee928f3cda5134202a9388e64cf05f8190049df4e777709838d0c9d3bcb37eecdc38c1a5d2b471c4b910cfaa9a9ba81f69b4b45c40344029958fa40000e56881bc6a14864330d5b351c161208676cb852bf47970268d37d4bfe97b3b26ef5b785f50ebc8c47949dc9bd0b2e673fb040e26789f3f5cdbce8e4b78389992bb83eeb2b063e9e1db06a9ede933faef7e635effe5e1b1e21153dc6934197efa1fd68f18a40ed569746c8374
" ~k:"0711c4621a8bcd40ff3e8b95728ce67a000e1fa33741246d420b046bdec48657
" ~r:"36c086070368265f736e7bbad54aaf2482d26161f8057a97a4b8cd2b4ddd7855
" ~s:"31d99d736ea67014fe59cb2212c47eb920f2af44e32b65db15af83cbe8e6aa70

" ; case ~msg:"cbc37afc75177a8386dce2c40c33b8f5dedc23113b4512cb96790f2dd74066103e0c45a9c6176ff96b7d719162003cee10fad6ccc198550a389275d21e708b6961523272ecd5efab5680ed741c2de025b02bbdc56315a442e437c43e3b378e6d62ea8878fd9789858a8c68a504bff49516e762a22ae513a2dceba9253b36f553
" ~x:"7c6ee86f45ddf8b87f8884f59aad9e320b73b246a80b26a645188a40a9bca62d
" ~y:"356cc7370c840fa26b0d106c47a626e028a0c967c093810b520639bdda0d339b7fc29adc0d9036b9710358ef9f8c6c05252b278281b2afe7953886429e85d228fb5474acfd65213151e9da0aef86a66f9f9c59fa88fd48cc3addc83d7adf4afb1665049ed094020219c01958b697f22e652152e53bf4e8f68f476a58181ddd3f64344e9b87a08c5d0de49e7b3c2995840c200084e90a76d2c05f8b5c68e77192d0676b4219d4579cb2de0f2a93a916b4f9cfe0d8113dc4bbd97ed12d8ce0447fcf9df12e922c6383ca69c9de9ad320f9c5331adb6eb1d223079196a2939cc0a7259c512c478c943fe05736710e273e4b5867174de72e703b5e7bf7afdbc06427
" ~k:"685a19da2ee3dd94fe9726a32e712fac05eeffe11e3dd9f60e6f90af7c13e23a
" ~r:"5645ef65e8e9236d874d459e7a5809923c05d64b22757bfc5b5621079e84819c
" ~s:"65f4c8febaf3e9d46581b17685c4f2ec9b956421d034a2c1aaabee94b787a4f1

" ; case ~msg:"8eb3685c3f406c5615e88accf4c0c7d2071b6c7bde5244994f73dc04f3cc0ab7e2b6664a1994e6eec52b62790a04328e436a2b4af3cbe3ba6e4c8f363a39b2529ef554c0c627f9f6b255928a39a465e60ac50ccf01f32c7ba483640344b6a8f583c90876b84d19554b0a4baabc2c240e296b12c819410cacffe7a7464419bee0
" ~x:"7e5e3d7255a629c39f88b6046fe0039159e44c2d2309b112ab05c61561d9e44a
" ~y:"94ba486977f5982f2ae75e986b7e194461cc3d65cdbf26f936805d12d7f850aad7580206d7dc544cd12ca1891c9dc406c949e52b9febfa88836f1566d521a110bb545e07ba28caf07e1bbfa3b176cc917cc4bb45dae7f873b72dfa9000e9ab6083e705c0167d853dda114c429fd812a05961fc2e78ba9e68ccdb9dc67b116f10532034d9f0f7d39901dc643127c4309058f8ebf43b28a5ce534e29d6227c4ec27ccf777b0008df5ce8b8a19b5771725cb0f9f2a62bb41f0106c390803a307c60acbed6c2e1e0db5036e0e79ddcc3f718b29ca5aa022f2f0bbe815f9c0eb504fc9ff8d18a2da999023af8105cddfc6794dfdcc41333bccd446ad7b82a0a7bfe38
" ~k:"3966daabf7854949475ff47f3932393a73f21e275b3baad861a92a3ab322e376
" ~r:"27b4e3c3a45efa6131c3d005ca924dff11fdccf409c2a6993fcb505477b6e400
" ~s:"68a085bd130c4ec08aa9673c495ba5afd46c9ddad2052ba7ab396329d900d86c

" ; case ~msg:"f2b02ac627b3f66baf4ebaa52b899adfd7071af53e78923182d8b4d5f3a9474251308b4dbd15fb6b657be65028a189353912d7c16d6d4989985c15cedc4343f0ceb680617bc7278511f9068abd613718a862513ee514fdf80cd25b6f84c48851e6a7850feaea57ea20deb1123ca4206bde8a93ff999ef789583e2c850d9e0635
" ~x:"7e52070b03aba0af4cad1cba0a733618e3adb7de873efba013878fa76331b5e1
" ~y:"4e160d6970683f4d84eb88c55ba2da58d77f6374fc5127273d65e8ef96ccfff51df69b0e2fdf3e98f6d35e6a3dd9f7edd90bbae4c6581cd02ad01336c0086d4248eb1373480789f7d8333b831db3bae0bdb49789aab93cde1faf1ce88dcdc7a1a4f86143ce44f851ace459a5528c96195f4438ee7c1856ac61fd5035d839d62e48a1ab6bd23ad52f1f6ffed19826b6d7f6491cfb05003176f29079455443f0ab482150fac8e32a3902a4096775f342edee2daf4c4f338d455b4ea35d3975f72be85e98e87158486b4c3d6ec37a3703f63a3e19272ba5255089aacd30fa3979b458df616f57b7502b4291384562041f6188db503f3df7f5981da5705eb0f1d242
" ~k:"57c141f543386db3bd6a97121f93b47e38891796f02565058ec6a5ce65f7a212
" ~r:"6433bd33db0ac8261c691af3a27f52cdd4a65d799939faf279ac41788e7528a6
" ~s:"04cfdcb993382e8fd2db8d90dca80e94b17b432009852cd3f86625159e837c19

" ; case ~msg:"2b4365a4ac6854c972da7347af1cecc6edcbae9d533b74fbe6db5712163a6ce984f9d7a4c54b44dd7555e5c2d2f3d098f31d517f8ebd330199a54b15297e5adee1bdf391581f1019b1ad72dcccd5484b51d275a368c69a7662e79f9b29c9a3084c94ae76da04f958c7d36cecc5d41d77f2302ff28f2ed9c66a0662cabf51c842
" ~x:"35b701b7d59aad55eb4299e0f9e0348baec875eaf62d2174bce92dd23302a81a
" ~y:"58e635eec80bde1eb7bf2da20600617af29f0a191705676bc10f7553f7611126e4c4d44bcf14f7a9f48da6e1b1e54d0a715724af5bca93867090f9bfc92741dfe1dd4f06075ec2a9262da81e0dcabfcab9e694ddca86d0e1cfaa321e2b5818182eb620bd5d16bc27a2da035d4bc17807cfe8ae3038c5bbb8a023fb232814b91b99749f519de39aa0f434313323b1b58202c59119b0be217617047c9e2ea453d608562cb96c4f0851a7965b164f9bbe151f9c508ca209f1af659e363804c8d8fa1ad700e20866ec9a1e505b74bbab70cb472308431a3e87272febf7cce2c20ec37f5d68b4e47bf3741013723936db7c9b0f3ded964acb7f8ac9c5a6b4f28de198
" ~k:"1c6ceff82adebf8c81bb4842b90dbe2a12c9d07c3a9d4990d44106a1768bb082
" ~r:"00a7c664c544cd7b61749410dda33bb3a47c3eb5a9a7be5fba201a390cecfaef
" ~s:"6fbbda967b584bd9ec6a0ae76e0c552b3d42bf0e9cf2939caf6123f6e86046f6

" ; case ~msg:"cab1d17666b0c9658cc78cfcba17a08e2989d3c202c8b5085531404d928c618b6e230b25c46a5b58437e4335fc040020ba00c863182325940f00aad330145e666d07e9e9d876137010932ae520d9188ca3d7993c905395219c55846d19b8fcdb1d0c1586b9b51097afd6972ae1472b0e20453f8fbd5d6aa9e4a9a9b3dc37dd8f
" ~x:"1ca2b291707ce4f70e366ee97b5da158a1c985ba4f252c572f0fb329e43f9cb9
" ~y:"5022c8a6fa79b7aa11a3d7af5acebb2ef8c50b28d8f0e3a556196562d34131fb44f22c3be3f9895e35eee70aa53b6c67920c540ba6c1085b0ea818b12aea811f2dfaeb6daed976e362430798fdcca3912a0891e7d1c83b748af1e7689e038b490eb73f7fe6e0612e8f238580e78833b20727a602768ab2d59dda36e75146fa4d3664f7b0cef7be877afdcdba23004ee313a69fd61c326759e7e779ad750f7a5cad9fb2dd80a8eea6dcbda0195dcc17b38ad6f0e2ab68cfc69b15c572f85f20c3679c15a83099cf08a379055f8fbdd8f590d43bd12f75baf0eccd6c077ac7589aab8171e8875db0122e6c78617c13586143a7ebe904a7822bacf48a7527f7fa4e
" ~k:"4f1e2aae323c5309b3ee5d3b73e5d4090c75da17765559e118bfd1460c312859
" ~r:"7b8b75ac8514c68de0caa98e9de0b9607253d8088d3feadf92b83ffc26e088ce
" ~s:"4b10e17ff64a0eb72f70a863d00a9bf331bbb515ba3a9fef72753ad7f0df0be5" ]


let sha512_n256_cases =
  (* [mod = L=2048, N=256, SHA-512] *)
  let domain = params
~p:"f63da3be9a9616196c6556f3ce6fd8b98bdda9137473da46fed970e2b8d147387a81922065d528a7d6433ebc5e35b15c67ea35a5a5bff5b9cef1cd1e6fe31dda52838da3aa89b9b4e8d9d3c0732ccc4f238ce1b416c4ca93f2c6800e5f4ed41c4f7615cec5531b98680b20dc63f73e70d803aacfaece33d45fa0e39d77c8508209528b9046b5917010791234397e412d22bc0b8d67cbd1cd28a32c2460a0bd86aaba0eea80e16e3245643171e34221760c203a56b8207a1009e6c1a2f6cda85f85c4f9e410b9499233c0ee072e465af4fb4fb9282c5c10e8234fd630ea92f0aae6b97a520db34475707b79a4c175265c0356ccbca827e3837df3d6d0576d9079
" ~q:"9b7463f8269f0b909abed10991684f36a64ac864e0d6d717c0ef21577a4c3907
" ~g:"972a75f606e8aa3a91ff08fd131a20f5963251304e3d1431b712fa0803d527fd710fb7eb27e52904971cd43ca977199a24dbeeb4b7bc2ba075d3b72eb6b2c5ad8f0e8b8f48c50b554c7e0711f4c7416330806672498f430292724bf98a8ea48c7f53d7b31d8b7528b1a6f087d2c27c335202835b1e314225b37aef8bfcec7d80920c4a460a3d68344ded75ed9ee867fa2a6945063894f563b68633b8b39f83a1aaaf5a96c7f422687e7c84cf8fb8cc5f4504dff087bcb26a95bbf8583f03b3a0e43a356b2bd7e25cdddf7a015300faecc6793c5ee99b6327cb8456e32d9115339d5a6b712b7f9d0301acb05133e3115e454d3a6dd24a1693c94aab5406504bf7"
  in
  let case = case_of ~domain ~hash:Digestif.sha512 in

[ case ~msg:"8ab01510cfa33cfa5bcff003bba39996fa727693abf6ac010bb959b0b59a15306c0c3a1921af2a76717aa55b39fa3723f4c3229ca9acf6b741614bb551cde8a7220ab97d4b453bec1e05a0eaa42e382bbc7b9b84f8237dc8964ee5b66e9b2a4ca61cf675140efef54fb327a665def8d57ab097e8c53c643fcb58209c4215b608
" ~x:"5f6e545daef6cd1b8d9848dd98758807236ac0b7ff053b32c703eaa3b1147557
" ~y:"41197ce2233d7e48c803cd64c78f657923b9e36b871401f8661c21d8ba38c6b9b3239db767b11d1d401e5faecbf7a45860cc5f1a54d60286b7d6e1c99fd5b8c84ed851c5357d41ad60163f224d78c996143fff89dd3a8fe123dae1f621427fd8cce76ed138d68fa248f374ae233249625b93f3dd5937d15e541b7effa4df4fea7d52faced615bfe0348418ff93e69a20a52e55c76cc30f307f84e71e4aabc0825eca3a95b4bd58ebfb0029d23a169e9d80ba7d1c5fd35395e6602e089aa9918f08bae35ae1cac7af33694129e98f0dadadd90eaeb6eed25024390b1a60af794734c397b0f509865b134b2867c115d6f489b6dd7e3c82994b45dce2a23c6bc902
" ~k:"5fe61afddbdf04449b24295a52a1a037d3f31441a3cec138b7f0102db86ef132
" ~r:"6a47ea57ceaecc116d7190ff6c6dd9831ab75b4bf6cb291083e4268b486ed245
" ~s:"017355f698a32abe9a4d4a7dda7c85950cddc348ab8a6751e72fddc01aa5d1f0

" ; case ~msg:"b2f56948a336982a5bcb4bb5d79e3fe5c36081bd286e6e021ab29b522f0be5ff5e81e638f23d0781c268a89b09332575cb31c0804bbd34c80589fb11570fc65b3f67612605a9411cdab3ac00ff3fce33ab22c46d26bf9c3fc5ad2d9018deb9b669b50fbfbaf8bed6230c7bd621d564fb1af953f0e82c5b5520ab97baccf58d6e
" ~x:"91e01626208863a954eb8987f8e987c8e6213536bb18f5afe3bd66a525bbadfc
" ~y:"72b84eb6a60c686f74f376e26b2e47e44a6d5dd92c06fde49faad0af9b11e43147ce9308ef3501a752e7bf18e9e6df3c0a49c44cd2515a05508f8060a61e6e6f1b2ecf14b338cf0fd8b7ccbe678d52dbdf20352c155a2bd517d827d6cefbf48c5679c998298e2186ef1098160dfb65914506a177943a4a058282382d327ad36f88301be693c02000c72463e682421a0237804dbb27335c78e8495fac7842d2aafebf90f3c3605f758615df989fdbd06e23e4ad6974b62384f0aa01027db89ac3dcb01cb5258cdbd9c19372a6c4aadf27298062ac9a16de2eb076e167ad7c65d0505c8fcecf359bb5d05cd22e7d48629af539fe7f60e23e957c84c7a61ac92bf8
" ~k:"6aff566d97cc48ef6bac507d64973c95da14fd704d3a5332aaaca2bdf21e894e
" ~r:"43704e96cc8d63e6f5b7e118cb7c030d0bd563b8f7a1a304b368a6c66d7e7fa8
" ~s:"490da43fd0f19fec4ee081cce25df6b2720b1a76b023c15704dd03ef1c3e48a7

" ; case ~msg:"9ae8479327b8b8a57f570f6ec76a1ac6f02b198c6048a1f096e6ce5630b6caf363176413d88033b1cd07f4d3960a12dbae8a659174bb87c37aca6ec56ed5a6619b8ba676b650d97c6a21af023985dc361fa234b2b3c17e77703ba99ae3211260da10a60f240eeef478f2641184a281716ae57888117dba992853f494ac3caa45
" ~x:"8a5624694a25209a5fb3983ecac3fedf508e0b23e878f60a18ec0e897c379f7b
" ~y:"ce348b5cb3d36808422a5016dd5873df79f3cbb5e1b458e8c1110226047543d965769a112adb4fced0d146230962a8d413225cc70d810d40e6a72e6dc80db509400c09d263d66206966ed51ab65930a2aac99fcce3a398b64d59097683d2baa57682705abc32eb8c32d6f1e7d94ca17ed7067822cd20fba3795ed1843c01b0d7551c7c4c759d53a4191483bdc6e3121c2bc12607701f43e3ba382c6766819db07ef9c59586937514772c2eccde4c54d92575734c45a8e832c4417b43a92c9abd152259cc0a969bac64b237bb3a0826ae72919d7c2dd2efdf03e83701980c2a8f50ce6e44d7cc8848645bf40aefdf24fa7a6dce5a3b9aca6f017618a64d91ce4b
" ~k:"86c3ce567e7995a61bc00e088ff2f2a425433a453252b1a729d8d85ed506bdec
" ~r:"0091d750ad9a4f29573fd457a5891b68d4b6c15703a2bc192c7c620c4e4c4529
" ~s:"92c409c8977975a417d9f5e0e2dc70683a53a95662ad270ae35d496567a9a2fc

" ; case ~msg:"e5a1a344c25ba0cbbcffe6800135f2ede81049180fb2759fd9e1af3b816ad5436a24faf29cf3ad91cf413332f454f74a9d4f5efe76cf02512c273cd525f04afdb5c24b0588d611d72153680d1e3995e0aa750e9077b0752bd4442bf7bfa8dba38e1c5e7ddd687f55aa54c138c7e6d5f064f3ec55942dc192dd996e553633afd6
" ~x:"976fb067157b214a80658e7ed2f566911b35b1671e5c0bdd55ff5811e822bf82
" ~y:"3859d4735c14baeec14b79cc2693ffcac900a2c26ec634a8e977d206ad6ec7b13f2d450ef04782ec0abb0da48f000628cec1f6e9a727bb59d7c0f0d743f513ac0925beb61bf3ad75824fffae1eb783eb1b68fc40d28770e280fde23844a144d4b1a95409b755c7ff2e5c67811f3b1c2eb96cb159a642d84dd7b5dccc2c0aef06d1cd54eac94a11273f9498f1e7a7cd79c108e496dcf573ef3a6610b7731ab14c162ce8377cb9b90788e356f51f4b51a1ec8bd86bd88fd4c38e62cad619ab8941bcb98a2f35ee512f4f8ffdd5ee70caed8467156b893b3532a0a2aa5199ceaecc5b194bc057964cf450668c44f27ec80de21ea1a415ee6a6569832394f6b405d1
" ~k:"1ef4f08defdb5c59a3df3358e083ce804c969d046ab67f2f938eb1a8f06a5d0a
" ~r:"443644e127e381b17bb66c53509718a58a30f927425806a62840119e78c293b7
" ~s:"3f01e5d1e9fdb1cfda25eff3caccf4edf599fea277201cf2b01ffd7cb1a9a727

" ; case ~msg:"b88c212070be398a1f81e85dfd71dc2424a38ae38a9d61085186504f4c2cbfa492b76dbcc051cefde0616a7e3310b4bf17244de7d10f847ce2a9f665948e76724d8f1f4bb3a61919b2ec7dc47ad8a72cb5998b79fe3a156395e4ae88e682b1dd16c52d64cb4b31c39d4a42a21e6242dc0cdbb0acf3d47182638c5f216dc6e8b1
" ~x:"02163cda612e84eb5ea9e4e068b14c10dad073409154d86fea6aaede59538d2e
" ~y:"541c690f4ca0c42e5267646f78ef42fd68c363375b2e983be444e4819e63cdc129018bd3b8c6da8b707c196c35c93eabee10e875c41fd925bb3ce80696935d16313fd3a26858eccf2d507fc2a10950525c670dadc883dc6779ac1ce866d8820395f3541c863018337a6be944ddc644aaa6c007197d7a5f9aa53a5e1180ad51c98be9d561a85fe9734160ca35e4fadb02527ba0fa58041b4d96385f7f8ff6ae756add4968c0c2799c0d680f66c8ce96f498228738e3e87b7c866344db7d5a4ec3282431aee5951d9b4c83ec2a0cda36cb2e2c437363ceba4e8e9f6128439d12c51868d0cb1f61e53a68d4e71c5a9e7de43c6dfca26f1741aca916e4282653bfc1
" ~k:"42cc30e9591b42486ce9998ab7594ddabc5328ca2e931e08c75b76bbe1f8b978
" ~r:"078a7146a2c509b97a6a8c963baf1fbfbd1a2a5aa214a15ea45763f0e7930beb
" ~s:"2979cbf59adb70f28ac4fcb69297498f8163764c62b31963da9c8f9c0c43e075

" ; case ~msg:"4adf1ed4fbb5b82d7a2b1a2938430753a6207da1cc049574f0a19314272f9a80c6a53498b78e5c0b7401ce485fd4baebc966da6c1fcb025816cfae32b58aa87f5e8885054735f93df19ed32c819786d4109dbda047d68c0589330715e10522643bbe27e32c0dc9c58336be305b4c0c981b40e0eeda0de461d8441c02c18ceac5
" ~x:"5b44bfbb69277fbe497ec729838886e7a787f336c246551526b660a7603d167e
" ~y:"8b6927fe293ac9111ba406125d6ebfbc30f96cbfd696fcac7dded42305c6105453accb1b0ca6f0f31601f8c34f96bb8ee4ccf149923a12821dfaa2a3859a39cf82567609b2060ff609232e90261d66cf31fb9264671f3f1bff6c8a958e5cd015dcc02dfd2f02fb6a443c2bf45abf13862059df98066e00311bb6438b7fe2d91e287553d25411f0fba47417c2902f978c57257ae4eaa3f99317d5adee0f9adf4d41e41072552b3f51eb9936a7f63cc28b466fab6429d06868d18ca09aba634093767192049b02bcb752eb674c98a86869d6726f742e57ef8c3d4531171c64f03e10a4e44039a44d407ebfc6b56a7cdf6b17394b53b5604347c51cf375551b7306
" ~k:"5f02472e007874056abe7194e80845b81baeaf4f6f564d3640373757f4252f57
" ~r:"5b201116d8bbc87db99001707b567e7c3451d802fa6c679bf3db3456711a1913
" ~s:"5be7e4c493fd5d19b771373141294daad97656a3dbe3fd2abbd3b6c62c166126

" ; case ~msg:"bd491cf68b34f7ba9afe0c6ef5f2b7956ef964465f28b2797bc1d6e670a6d81730ee2993d0b4aa96905157025d775ba104e7c19b3b372e852026b1286cbc6a48a10cb9378e97ad966f9cf03917ee8db75b6264e9b0a48a0ae10c2f46444710d4234126ce456b9fd11ab7a3504948d046d5f438d893d9b1052b8fac9547415472
" ~x:"609a372d3844ca8224dfe780b425e1a7c00d09957a862de6f640af57c086dd6d
" ~y:"a92e446510764ee1cf81c6b59b5160a7608ff8952d045dd69f034fdfef93f633607ec209b106c6ac8f0cc6ffa64bb9a4484560b838d6f24c993a954efc9d5ee16656aaba2a0d5a94e7a346c7e501af83f131db9e0cab8789fab19bd591ec227f39b349be7f8d0df58ca0396efb1e76549335904b88ec21cd3265c543c4e80e9dde7cb5c9ea8cdda23d96ef1c3839ade8ed4a5cd5fd98b79bceeed9c641c5a7758d0529aceaf27b5014f13dfcaa267a14a0841b36897b6e1e8917b7f7cbf7cff1d1953ac43cc04ab06cf111e006497eb42f28cbc905d6f1cd5d8394857983e1c9e552015a451d0c13a6848a8fc56b79dec1723a8067ff18931c852ceb81affec1
" ~k:"8c70ac971b83f159d2e6ec26bca1463aadbc8e9987593f49a9258f7f0de9cb38
" ~r:"0cf526d8a0f9c912d143f3f8afded4598b2a5aaf200e0749ea27defeb7f28f3a
" ~s:"877a9066f6c5ae78251d9d140bcf39ae912d18bf131bdc7e9d61012daaa4292c

" ; case ~msg:"c00a8a2fffd10bc2eab63b8e375d0c10f9dfae2848ba42afe6085aeec26e21af3eaa493ce4b3d95a31fa502a60ab88e805f4fdf889ed91c15421718084cd0d644795749b1a6b183d74782d52c7babf7400393cee698af5dc010c0ff7f5acdf0208f93ee7e4ef58da123dfde7f0a34e209bbaec61007293fd11afa60b6522c45d
" ~x:"683e924893dbbd751e0a3f910867471a6410fef562cca9f464943abd88e0430f
" ~y:"7560105b8586c4532bf1b51e0d2cf9a713a5ea5d40e262ce01ebdaf1ee53d857129e1529a0f8dff63e86202c111c6eb289439cb15cd59fc218abe619c9516250f127fafe9a53076274f306f0b7871cffbd156b1a8819795f0a9955864756650274b83e67caa4e215f833afd5a77d0594b21b4b54356a98a56a0bf6177faffb9fdfd888d6538a1ce76059854bd1f0a20761281d7b751757c6ecc9b1e8131196d0669597213ae73edb9965da9ff372420851155011f691a03a7f1e2040291575b86f595998a06ef79f4eadbae2bd9e2e477dd72684d8efdc1e835f7f0f5c93635c181b96cc7c0eaa27ee62c9227ed9485a8c822b3224e9e2b7acc10956f3d49a6f
" ~k:"3bf5f524e3a3903c149958d10ae68f0a87a03821445a98b0b9d08a3689738853
" ~r:"98fee10c85ab46d334758734819e68b5046439cd0b66be26d43760613ac77b8c
" ~s:"665fab98dd437e06a4f877ee218986e37c2cb2d237e598d98f1b7d4e829a846b

" ; case ~msg:"27f01b47d15f7d196f2667b75ed15b89d7443fb4fab068f4adb67175ca70071d52e270f68964f9fb0e0e14ed5d2954a33d93807acf3c82500e8b9f5fc5510cc3bd6aaa1daac8309128ef4c0b4cac026425aefdd7e69c22c32e5f8d2a6e8f2ea291ac33da6c71a1953e443c0ea206568aadef2b96466cbf76bf149d89d86f529f
" ~x:"285dcba140162fc203651c5ff7f1155341436c5c5c98e1e9df192b3c948a16ca
" ~y:"38fa994a1f61ab79ee7a7e6f689c38f6c2826f06647b1663cd812adb36d7fd7ccc50e9a90d02bf7c3f12a228c692c056fb3bd608f51aa401022c839791e6a678185cd31d88cc661af29e5d238142181dd3f6e7c8b05785221e62fdb36c71e07f51d732e7e0cab520a7f2fc5b1831b0a6ba280e00321cb9a025db6538abd672463dbff5ca81993676bcbaf0f6e9c754f24d654ee7879bc03d7d4bc8e8ca58fb9b3929a3c38365cd2e205729e9def0a00108dffe9407271e17d355ec4b29003e0caf0c5b2acb9bd8e52d4410baa9b97a49874c14beebf03abf28a9ec59bc1738b8dd4223d47aa336acbca7662fc69a6fefeecffd47f6737ecda331d1ba5cdf023d
" ~k:"4029e06b437cbcf8e0788a393ba3aad0d182564ab6a53565eec1755c4f4b6e2f
" ~r:"8dad02c02ad34fe4e758ff5c81d5384c40d2c49d0ac777bad1cdebc58ec01cfd
" ~s:"0fe4e1f6875c113f1c17a0f0ed228d44213f8d7e2f15567e57ceb2e8b1098f7d

" ; case ~msg:"73cc5e4a188d2814466941389014ea45a1a06525d2069cf4883ebcb5f22ab128c00f041cf69fd94b33fdade78548f6523c838b87ccd868f3d3d0a9a000f278ba54048b9cadac7a99d98def51713191ad83e5232e3e86497245c80bc710fdd7faaad88ce92c894f8cad3de0075caba337a222cb7a3d7c2d937bcfe4b6e69d388d
" ~x:"742242f1cde89559dadae5e2cea28cf402c60ea9af2a5282202281f55a0d4d04
" ~y:"5266427ad4c1cf3ea229377ad397c7d5613512fc27f2ce37407d2cea8e1999aebb8f3767ee96cb927ebdd43b8dbc10ba2c47843d3f43368d9e442bf51ebcf20b48b543a4c388bb3ae3e4027acb657d1bf74abeb8b998421308770f70b3f7b1d910219a1210260340123b95dba187e00cb067f7e37792341202554bfc8a235fc01ecb099ec3615a67a3610d4d8c2dad16087024f5973eb18400c29c05d6984d1c15c159422827c0dbb2bf4509d710c4972ee93be7283aadd991ae8ef0e97312118f195d304fbe96d5aebfb21203eae6117831f9be9099d3d476b83f65ab225f8be493a8ad21620f259d8a44200810c8e562aea8e7a6bc238c129b19f2531a6af0
" ~k:"57d9723e0d17ed96a3a77ad47be6eafc06a5aa01b59b89be70a756d37dd0df2b
" ~r:"77236b33b04285425775ee3f658b3761295cbff8e4bc05abdd22e3d78b1b6da2
" ~s:"43fdbd936ab40459f6843056ca77e125b6ec5ad945041c1f6a2770be9dfcc682

" ; case ~msg:"c0746befd2afc6ca15cdb145c18462c515bd42794c4c7ee513cd9aeb0fc6fc3048b6c7231634984a1be824c775f9c9b028255f5b3c3d8fa08d47aba07755b5f1b5b008933eff35838f15a02ba9366c1036d3ff19e88199ef86a88227272cf4e4e00ffad9c36bebac30578b00214fb29bae43cf555ed431a2f24922430b1496fb
" ~x:"47c0c6f4e6b56cdf1e1d9b63ff3739edec9c3d5a7c990492a1c72aa1494fcf9c
" ~y:"431eee49090ad58f4a874c2eb5897969fafe3274bd7486b65e3519e4309d636ace6864d5ca4d8448a357cafac15ac3cb3bd7b2755b3cb6db0af1a4e91b2d1fcb28561b170faf2e0690071bc0f6e42b2d82abe5646ddb8f9b99ee1daf5906036f395d824d080bfaea103048b3f44d0636bc7a6a88e9b004a363b99d24a89b6e97379b20bacf48c7ae2e9bf7e281fe3b4d7eb947a102396d523a1e85ce17fd25f271f3c221a5681e9fb77d64d6241039ac8a85da32741bacf00660e421fe850a0fe73a08ee3a9b069c6d9114c1975272127468f9008552ea4cdf9d96561ea69a646695242500f2318bda82da633ef1ae0497014a637b15a572ddddec070d19d884
" ~k:"3598a6006fa3f8b8f9b7ff96ba06bf3837a1a1a92892e4a268c75285bfa6d660
" ~r:"796d7dba322d92a083da7a588fb6238dc86b1fc5104ed600c9b4c688edf805e9
" ~s:"012c1ff4de8ee386b951275e250581d661d030a4d8fe115432288ab0a4bd46cb

" ; case ~msg:"b8b915cf4ea3b0c4cdcd8b2a06479e71bb4797294b6c41ca870d3cb2ec2cb5a49f6bfe5bcd10be609ed3e1882a312395fc991345aba5b566e67960b42913db669041ea30c29947edde7bdcfc0896b97660740d6c79f0088665f51dadcfa07f7be44821d60a8ffde4e5cb1f98139ff91c9c6f3126596344c5f7eff40049d3f9ae
" ~x:"0eb74b5186697af279ce72da74f1ebf59921ed425da0f3eea17517eadddb7c90
" ~y:"1b3722764264e17994f3343bf260c73575d106f6307f2eaa3f7dcd5af804463ddb6bbe38a38f5ab5a8ae6701317cf6c267049fc9b84078241f82d3c6b7e5beba5c1427030297f1df258148e5f9eb41eb20a86877fcc06e5373cd50562613d307649539d28cb52418d42fd59758b61185e792992b5a581229b43403d793b04d878eb9b9d12ea10d2e64d153d3fa41881fe79a67ac408a5348d79239567dca96e1ead3c6ac22dbcdbcb5185bf8ace57660a4252104e5047cac87851d2815b12ae8ae96ab2f33345ea1cf5f2e58a4ddcba26265c06df65afcc6e852b3f910c8778de28a9f098158ed0eca652dda2f9f4ac8a17a9b252410ec5973a6063b642568f1
" ~k:"37128d19b2108a8e8fdf2cac984d45851078a194bb9946a4db260f27b4650439
" ~r:"6b4b0e1e7cbdefedb1df1f529ece47891f7b9e959a3f8556ba4bef7bb9856560
" ~s:"7e933b44ede6b2e941b60c37dcd1568284def229c0a2bb9093f4829000c4409a

" ; case ~msg:"dffd458a808f1889d7f3d6197f0e41920ad731124cee308cb90d2361b23fee969c0e105835549e5d0a3f7690d5862d4cd6ccb33ad18094c85c9650d75b248496390a0b89e7dc7dc0d3a6130dd97789ebf105f8e55d8f0a1162fb3c6b529e2a80dd51e9045ef8ec42ca4bc46abb6539588b531c9799560cf4ea806c3d93d043e5
" ~x:"36ff71ed608f351c736042f3b638a89666007cefe8ab487e512d76fedce1ff35
" ~y:"ea437ad0ee9264de8792b677207e547090b32d6ab460b4d589d842ed0a0b4fb4c635e4443bf60e46cba8d226f659c76d2ca01c69707ba6d977255c4584b74740a7cdec4c973e3d16ab6af60cd3123ca12ed5971e69eaffa3da0770d8e1228889cd6825e1b85846f4f7ecdb33f1e5c7acd6b2add1308c5cec439728d0cc625eb89df34fb9c0dd4568f979deead286c50145903a0dccca7239874b4683d367ed31696eecada90dced8a9b1e01364b8794660c60f40590794c95a614c04563c92d444b5ecf01286b1bffe9ed9ef915b4db820ea5c9a5b3dedcf89a3e2c37871d21b763990c7bbf44418f91cdbce4361eeb227516cb344409d2c651f0dc29ec82623
" ~k:"91797ee940a167a57de7619334638f1b3ba63f9065b69f56dc04e4020a1682eb
" ~r:"3152fc286fed44f28b1af2d537592c5691d6798caed90591b5888b0d6fe6bb07
" ~s:"7bff61a8676f0df189654f25c5812b341dd17f4f44667789cc887c191bf47202

" ; case ~msg:"a6516019727d95639db038f90306a8d94fac5243dc7b67c3568d63d85dead1cfddbb2b330b619589bd582af15f0811177504fd5b7aad7b298647a3f64797e3da5fe5bf87b65c2ddec576a8f40660686b808ba42e54bfd0e9e48082d6904f8e19050e54ea4797a2f401ff7c9f3d217b526c03be9201c0dc1b0e8e054bbb32c382
" ~x:"396102ad116ca2e419b9229667a31737344d0d7854cac8930af18e12a9e2d63e
" ~y:"9779eb5338dcae7377b1847018ce72c1ed4c55292a963f81608ef332050f0a484519aa96b18bcce8e1b49c11a200c1ab4a75726bcc842485df6314e5c39fec622d819434294dbe1eb647885ce841527c03481b7f22ee586d8c2b1a8471a2757bffbdd9c26f125065685509ff0e4c8b826d73c6e12f6d4b9319cdfa72c069e07b2d2c254b330c06f488d6598c7476ce0f3330c97ec36b7c10871388472451a34afb7b4d4e251f9f72a4a38a6851aab86507b283e890c31ba96d0a1e5572637b2d8467060c0736d11d0744e332a19f59ae2920894e9cffacfeda64ae1ff4869882df3b690c7ce8fe0eb88171e4f2ab8624e6ac77dc8907613235163e0a2c7d9fd6
" ~k:"79753fbd43773b6757c01663b8f5ef642801aa5ccbf32082c780f71a22c4cb0d
" ~r:"7222d5eb392460defe8fe3df18fa534f3060235f1e8dce5370762ec6fc11e690
" ~s:"4351c428031cd9af567b1163037a4e376962620c4ec23c43b7105879f95bf614

" ; case ~msg:"1ffa7cf55f92f234a24bd3296744d543a433c907c1f77d8b706f4b6262d096e2dfe713fa9ca80e68579396fc11a12c0331cfb7745d96b005204e483fbf8f9fdc458e2ca8613406069df5f44918eff8c5f54b8b4d972e07a4b8e06d8426a70874cefe6e93404c1eb381c2d0701c37f85afb1601a09fff8ecfdaf6cb64ad9bd8b7
" ~x:"24039963cc5ac26a977728b852414f60a287174186ea812e00a5c8a8a5355daf
" ~y:"18e298e6301389d48644674f8339487a8651b0768dee425905e803ab357c7f9fa05dd5e2ee84bfe105a092716274557e063d086e78b781a43c56a4e0ea115c5cfeac57a4c9b7e1effb89413689928f1546feb30738586d36ffe338083ee2bf5c5bd344bc3db2a7977de2b1ab5ba006d9ee93ef8688a7d10cafe27af3e671013a816984196bfacf002335fe7414423ed8bdc80327372b0d460866480bdf073c9def7977131b06e28d14ae1a816d3222ebaadcc8d7c300aa820e0328af66f742061aff5d4b7176a994ad69b390bbdd619fce047dc7d15a48ea71afa72040bb14eeaf4a2b23d99b4d977beb6d806101021eb0c3a0e31e54579e58c953b55b6e3245
" ~k:"2ae1af11ff810141c37b1c23796e54f027b4eb7c2f0c412b6c83076de3d4aba1
" ~r:"21d70ed955b09ea302fb792978d12501071a2e8e2cc8f659decd3df24e37c466
" ~s:"2cdaaee2a5a3dd74a67795f93ac1d8416223836c76f7fe31c72ec6170925fd73" ]

let sha1_n256_cases2 =
  (* [mod = L=3072, N=256, SHA-1] *)

  let domain = params
~p:"fd5a6c56dd290f7dd84a29de17126eb4e4487b3eff0a44abe5c59792d2e1200b9c3db44d528b9f7d2248032e4ba0f7bfc4fafc706be511db2276c0b7ecffd38da2e1c2f237a75390c1e4d3239cba8e20e55840ecb05df5f01a1b6977ad1906f2cb544ccfb93b901ad0966b1832ad2dab526244a3156c905c01ac51cb73b9dcd9860d56175a425d846485d9b1f44a8a0c2578e6cf61947bc1a1392fdd320b16a9d70455fe436f2d47ded8e8e605f7486eb578ea7fc4ffd13c07f9996af159fd411e9451403278dd1141a8c926b35c96384bbd6bee09c46f44c36b1ffc7197f5e925dbe0544a68e6ab8c18e426a466b392f9c27dd79fefa9ca163cc5a375539a8559f277f657a535d1964c6a5e91683ef5698ebaa01ef818dbf72cb04c3ff092d188866f25cd405108f566b087f73d2d5beb51fac6de84ae5161a66af9602c7e4bfc146f4820bdfc092faeac69133e4a08a5b202a12498a22e57bad54674ed4b510109d52b5f74e70e1f6f82161718cd4cf00cc9f1958acc8bddcdfbd1fbe46cd1
" ~q:"800000000000000000000000334a26dd8f49c6811ce81bb1342b06e980f64b75
" ~g:"99ab030a21a5c9818174872167641c81c1e03c9b274cfbc27bc472542927766de5fa0539b3b73f3f16ac866a9aec8b445ded97fbff08834ed98c77e7fc89e5dc657bef766ff7fbf8e76873e17bee412762d56fe1141760ab4d25bafd4b6ef25b49a3506632d1f8e10770930760ec1325932c5a4baf9e90154264ddf442ec5c41fed95d11525151dbcfb3758149bad81c62b9cff7816b8f953b8b7c022590d1584e921dc955f5328ac72983ed5cf0d04056fe0d531e62f8f6c9ab3c0fcd44e14860b7311d2561c77c1d32f6c69dc8f77968c9d881ad9db5e0c114fda8628bca0335eb7fb9e15e625aabab58fc01194c81bf6fb2ce54077b82250e57c6a7b25deb6ee39d4b686a5c307a7612b2d85ee92512413dea297e44f317be7ceb70a3328af0b401001a418562b8ffe4e9771b4b4a8e0b40c791349d5d4e459fe620a1a2fc72e2f6ca28567d4c2632bbde1b49864c06bb12619f132c1da8f571ef613eac739f66ab3914cb3fa1ab86e05e5082ebaa24ebeea4cf51beefc27df512fe3fee7d"
  in
  let case = case_of ~domain ~hash:Digestif.sha1 in

[ case ~msg:"ca84af5c9adbc0044db00d7acfb1b493aab0388ffbad47b38cd3e9e3111cfe2cda2a45f751c46862f05bdcec4b698adfd2e1606e484c3be4ac0c379d4fbc7c2cda43e922811d7f6c33040e8e65d5f317684b90e26387cf931fe7c2f515058d753b08137ff2c6b79c910de8283149e6872cb66f7e02e66f2371785129569362f1
" ~x:"433cfd0532ccfd8cdd1b25920d2bb7396987b766240379035b0e86527ce9c52d
" ~y:"e7c2ee18c3aa362c0182c6a56c2584628083c73e045beda8d653690c9c2f6544edf9702c57c455273905336a5f5171107a313cd7d0b0f50f8d3342c60219f22a9023394059d05f464c4496d55dab6eb0898527ff4cf5678e7b5bfb5e18d92c4a9d73288cce14530fc4702f6d0397ec39a880c4a72d358730c56633386ede028023c1791f3164d1574e7823c79b8a3ca1343ea166ba6f02b7ff7e9ef2198db107f7cc159f3b6a1c00a78c355c566deb0ac6fde3f633cb9177a1fbc6c1766ca021d5fec470101abb440d2f06982181a8c92b7cdd765336b9a1e1ab70283d6db0a963fb648c37c4e29a74c37577291049ab47cdbc104c04db966681ea8ebb9f00cf4c4a5462117379575fbda4b801979451fa94b19b4e93656705c0f734f3e0914bb96c1e2b8a0fb68faf14296efdf3300ad95bcde8b67cc4b26e6488eef925cfaeac6f0d6567e8b41355f89d1c2b8fe687bfa2df5e287e1305b89b8c388c26196090ac0351abc561aadc797da8ccea4146c3e96095ebce353e0da4c55019052caa
" ~k:"40f503abd70fd49a76c67a83e08b062b3fd465ad92be433c080e5f295bb9f559
" ~r:"21ca148cdf44be4ae93b2f353b8e512d03ad96dafa80623fde4922a95f032732
" ~s:"73e48b77a3aa44307483c2dd895cb51db2112177c185c59cb1dcff32fda02a4f

" ; case ~msg:"3ccad0018519a898f87d8ce5f28c0d93ab16c51addf4173322cbc49d48ca9ea37ebe8bc9d8c1b3f783f8cf59cf3fcba10a393eb2ddd989ce258e73788ce74b0ce8223d24e993cfeafa49cc8ec1b8ecee017d83a11bb7034c779206c364ac463cfed3047e1a2bf5c591773b1d882b310bfba2db87893c89a5442c0845bf644e21
" ~x:"306c1304b380b7c3e09e7a4b489c64a295582bb3e03ce526f13d7482ef8263f0
" ~y:"3750d36353bfd2e9973e26a555bcf0d834d3d6620cb66579199e040ce8eccfaee660046e78df66e8ff641523046adcf425b8319db2447680194c3a386b5201dd1ac6bf3e66394e939eaaaca4fd3f386fcfe1d5ef4524b06c5ed9a15746f24baef1eec41e683bf35371084495d4da8e727aeba307fba000a769a234e3c4609704b3ba4dfd6a8644fba56083dac848751b52a8c2cdc47946cd21ea24383cc6244f000918e9a23276b606c5688565c44ddf7788181b789565a6becd257123bb81a2cbf9db7fa384e0ca41804ed7cd3c9ca0e1f8bb390bff50213b0629682409933770f6e03a5c4e7e89ade90255609786f6b2fc5a7aa7566bcf7f725aead4cf456c5f5ed7dc3e91e20d94d1aa2f6568c97abdf21e0ba8cbfb6561305cb45175b1abd7f39b9a11c797926b944f5d13c3d70e0b2a8ca18e1f5cda8ce6ac43ecbc1fef881f5eef5a842fd5984ad1e321a317005ad478cb47c9cff61267f1d496fded0a48328d629b7b200c441634ee908879011745bcab6660e15583748014d6de2fe2
" ~k:"223e52fc516c0a79f55a5474321264fcce78c050cf79b3d9961b37e24d7f32d3
" ~r:"1ef7723345b2013b71104ceedbe7a9cad430018968bb295b672c2b57b9a108b9
" ~s:"72852da485c0836a8ebdbc4c996f7f6cb65e99391ce06b19a7187618e9a95584

" ; case ~msg:"1fc98288857fb3a83ab507465a53c079ed66679cafdfb8653bfdebb03020fe86a943182d4f1377d58eca3c7710d32e210d8d03728bc69e1b8003944ffedaa1b69ae6cc506302bd6917019f588cc29501cc8263572ebc0feb153877174bcfdbad4a58659175d2de71d5f5019c46d112b6631cf0c3f912aac83140cd56cdf903ee
" ~x:"047a5e52039da40523feffe63312887e4d1ecdf64f32abb31dfe680bd1513077
" ~y:"335372770c0e8e591a367de99833bde6f01240bc6e236a5b4e36233e120b8ee6d1c19c77f4cdbc294d3278c3d4cf73ed9e8ea5032b0524a391cf293b35ee7e023430222216d9f18b45022f4d5f9385f6384d9faf1a0ffa4a800da23b937651a09e82c22285b9de6a408e23386ffa67abb9d1c71cda7bc0c93525fcd79e83153e746070782467858b697ad14914673033fedb2d7a105ad2d438daaa35b503b518314ac370fc5b1112d4fe514e5835d9a86de25e6b35691392d1cd04836d4126b295b8a89f217d581258af95277b8b91c31e6b0d23a7c52b0ce2641cf1a252838b6e28e226cfc4fa9dc914c5f675fc900ed680dc1aa9e1d17193c432af4032ebab954191327083c59a5f64c1ea18107ce4d7211d1c22f04805ed548fc22df4b162f30b6ff3a7f7c38a5a95fe824d2961180e98b30208dc7ea7071f792261d45c7bb7b911f3b19c3ee0171a326c033cf5fc2bf79de7d5115ac568e04789cb44e08f3a8627a1b1f3762342b49b7679bd7edbe47bc3ee9c3f02db15d53256a5ea2847
" ~k:"69a22d61b152af35c1b43deda88d5ad456d38df75b318b82712b2690a5f2f647
" ~r:"5dd0c7e8a3993b9de0676a579c897ea39943a43dbec5996e58c1985b541d7c1a
" ~s:"67971001822a08a2148a6b1adb50274a57dafe896fb04a12a6f99707555306ac

" ; case ~msg:"fda9765cc91a9db922aec7b13fc32ac4ec4e3b8534f9e95af96e8ebeab89d847dcd150444868cfaf4213f8d8baa6b1d0886224e2afd0aeb93d59b886572088d05bf721c7adfb54da47c6c4851204a7a92a11deb39ba17cf6c07fb7ce8ba350a99d018d4ea64bd56d1d9f8f7d88157f190fcb372acf6f8d31cf7b795b36c10f5e
" ~x:"7f6ce353841963c8a6ff3405713e361ef9f1e0765e665195e7c147dd98120c4f
" ~y:"0539cc992ca70f913537b1211dd326d85f7531baa6be0583ba45b9571baa81cd5828050dcd9ab7a203bc4fe1d874f76ec1f34d935579ed2132255789d7e6010cf504b4c7f586d44a716600acf8a04ad30cb7ca055d7223f9761cdaebfdf7ef72bdea3dfcd020069a969c560160f0534676beff5ea611fcbc0fd47c867f3163e1371e1de67a1a3c3e37168bf0be79c09f45f2bc43517aeea0100a2a25d148ff1990c06143fa253d8306f48d77362224bc3efe93389e922def0fd11d1992f550ed8294b6136547ffd612b0bc8e4ee90b3100bb89922177147be0083281bf663f8370417fa790d4105eb98cc126f5005b7c08bc211dea2898aa653c3d2b51fc67732bff5644e804aae69200c416035aa0ba5a14cc439b569f462117b7dfcf3f2cc13e723a93ff9533208f20241daf36cd16066be3dba20117cb145d756f5a6f79ce2356a051647aedd645bfa6faf8f80a6fdf3eec42ddd42bb2e7b7738296e23978c8bd63b8045953e06cef1263bfe03be2f61b16007df1eb198567a7bc6bff274e
" ~k:"01cd3cdd3feb4d1a995103b1520fef17f60cd9370ad59b3efe9383a2c0126f00
" ~r:"55c2b27e769fac99b47bc0a54ff1821c7a46be6001ab664fb68fb1bafc04446f
" ~s:"3059db42a399c428f3cfbb102d6c0409b06f20068d1ca8cbea4858ac6e5de1d3

" ; case ~msg:"e49a12b8d761ef7afbcb1c377eedf629d08cc509a8753a5b92e26a2397365156e7c081bcb4686695575c6a64f5d77dfd550b04df390aa55e0d051c759f197a751a6041e2dd0959f902f2e359a167d880c49cfa81e7196fa1604ad32a8017071f098d4cb346b39266fbe75659dfc6607bf0d829640782cf3e12e38376c5a99282
" ~x:"28d9bc1d9aaba882427ee26c262bd4003aaeba422bf053b0dde14bb6d6d74bdc
" ~y:"ee7fff18822ff454a207f9db542d24298bb5edb11d80ddc6ddb9bfae0c952d4fe8d9db0f1a86e8a0f2193af7caaee7264d74106de5af0a6c14f710bb863eb7dc167a1e4378b6cdb7ab6841c664e9824529119773578ef55b7c35ed221ef070dd4690b4c12f27673e5d1fe964ffe29da57e2d1acd21ef13e0669fa97668bb199b56a3a53e10469133022081cdf62648100dca267c4f6a3ca3a75b573bb1b39c8a4e1fcf81269e9e1b10c63f5ba4fe75cf7139d038d02f5f534aa081fce732cd5051609bc06f18874dd01121d3c179f0c3f0399c185eebdc34635b3139f1ca50fbffb3b0ad12e481c1a64682143793f072c7db8b5b9eef41ccdd66b904139d644442a92f6255edb9bc1234e27d07a6ba32b1f14cdf98a22c6a12300dff50ac1b65568b6e915541bb386ec725da444467ca25e81448cb78375146ad2078a830e7d905de9ad7d89559c9d430cf5f419ce945704a426ab264016ed87c90d97f51a7d6e1ee2f51bbb3a8de81391697b0e422df9e5d3551e93374e5f38016b296d53bc2
" ~k:"0650ebc3e21bf1d90ffb3ef5a707013dfce78fbd2c21a0da9c8106d1fa98a46f
" ~r:"21d34df605237975db31b864f98c9ab6e465dbf0b3fc5868d67cd6cb3a13963b
" ~s:"70c48807d62d1fe74d58959347ab12c97b500d20607ed2a95d8a388fee265812

" ; case ~msg:"28f7a067a0ea7f0a4d797cea3939f66b281ed19cc98b8563ef375798b40614f4dd85ac2fcfccbc5ebf0ac93228c0b72937a481ca4f9df7a7e5d2e5da9af04874dcec35035f6a7db493793aa2361fb66ef2eedb7574d04e2147c357298a2adf99aca1eebe00cefa44b39157eb1e94aa8aa98d545151fbb4de67070b3904cce930
" ~x:"0c5088f5d337802770e6f98349d53461f13161020ab9a241efed4faed2e569ee
" ~y:"cdb9922d69e99c7f34a9210e2afc5be0115da4aaf682d9ea37788e0b6caa6fde13c88e51f558820668b59d14c06d2cbe6549d3f06d10dbeee46f59154cd467ae19e16be25e6f6cd238ccd1947fc581562d30ca329bb327258ca4aeb901f814414058b6f169a45ff55e40232d7870499ae78c0513777140752d55f0a470761bdcff5a6609cca2d1809f184b298718071d216a14ad01f56cccced23969607b62d4d140c9ef28507674f59fecc7e7ce8ad2636a5c53f070ad317c8cd0231f500a790ef69ac786000faf68e7b7854d6eb26499a9d524cbf8f373ca41dd6a2fa5198eba2a8e228f5ab29be9f6d450f7f5a149aeb20d8a277971fa6e64a0de36c8750afc381961756975621f287a3950f88402c5081fe0c54f44f9fa7c50df906b264098853646b3d05a4f04c6f1bbc6c440f7e7358d3a72b29f7643f4406b7db173690d40aa2938eaf01874d2ba8094cc5be1145b2b2ee9e7cf15bf398e50832d95017430b1869938732cdd1df593f5db2b2bd71308d8c253d254ef39b475e249d890
" ~k:"2be962ca1f82b879255e20d6971e633ba68582a5137bc55058fa42b48ddf0566
" ~r:"12de252da2593c5969a6496ae808d851cad1ded2959ea89057a92e5ec91c5f95
" ~s:"165338075e6a4fea0b238f9fac904b7b33dbee5a552646dfbed827f6d28d6492

" ; case ~msg:"0e156b0bd84595155ef4fc213dfc7e46bf27a89c275723e0984076b027c49cb2eee6ac866d75333581cca6f897e11418fb37ba5cab1391cd237e2c6ab3f11a055d3bd03f425baaabe5a6a34eba4b118af73edd610787cb8eaf476bd217048208ea4c1d0591372947a1c0ef94696568983424fd1d802fc911e7bf71224afdbdd9
" ~x:"2c4d972bb3539876b8f32cc645cff0d4be877175f31a028b9bfe973f0651789a
" ~y:"d0973641d56d8baaa2c2d430501dff44eaf9a3657478799134b0f335ae94ff2791dfd494401332486ba637683e704bd985f526919e661a2280d99bd48262b6c9305e0c8fd879cd0a836128d88eddae51fcfb51f744b23d2d2d27a2cc1ea5a9d5e0a7faf4227a2adbfb7ed45d6aa9c33798abf07bc69efc5fdde5dc5c780196257093eca75468b161cba44bcd142b21fae9edc6ab327830c28e1b3d2d7c812d8aec3a1952627a040110872e148e15de5c7b4ca24f086336daecbbf9816cdb9dc730db8a66a1929abecd4b09a039a19bffa45ffc85ddf0be3277bf075bbb46f07bf0daea24897e07044b5ee37f9f44fee757188170da22924fa15ed9c07f113cdf37a8c48648e586fb55a0c35f3b63a69667244193c70d94bbe36d043b25a041fba92a2042e2eef767e7cd18dd1c1b5ca4878fe774c833cb5c5aff9f67bfd6cfbf2dfc63b8842ad2d49cebcaff4c3927f3199c106d0c149a9b1b49bef1d6f8143d93d25df9db1b5b37d522e723ff64d9ee52e476206712a38246dd926271f5590e
" ~k:"5e12de89504bc84836c14fc47628a517e898fa46769eba2b36e7c69e580a6473
" ~r:"5ef5d78c421ae5a63978bcbf7d2037b5022bc47be7b293806580ad5b4de27a4e
" ~s:"67ccb2833c1d32c68e91ae3890b4c9a6e5229b22a5799168c0046ead92573c85

" ; case ~msg:"849c5337d88b3b247df573eb0d665548b6423763d5571f8acb5e61e316d7cdc208cda5b39a1944a717587e58e21b86ed222b8ee265105a32baff3692dcf7b8713d0b539262a5bd9a954cb7143ee66f8764db6236136cb1cb3b34a87cbd3fee3b11288bc94ac99179c681a469d62d9bcd91d40332a650a5bce33b6026884ef94a
" ~x:"680878e382b713d4dc9bed8b7eae880f1054f58e70cda2717577fed3c63393dc
" ~y:"1e8d4d6fef9905d639e2564d87dcbe0d8f99bde38082ff091a977f2affcab86505aeffe6ef1ddbacf15d9165b006ac0517434aaa65db210452fb2ff4c990b87f25fee7ad5b26ad87749575190089a56cdbceee6782ceaaf569814bb9e658ff50aebf6f3c9791893e5d6ada5fdf8c4720fafa184cc84a84f5fca79d899636e007bd0e1a89da094a378edb6d72240cc2d1d7098b53ba4837a5d0d7d02019b952712e4f1420e58af23d1377cd6d5f3989b3d60b5fc572043b96c4f7beb7137c0894fa99d727a5a88a5d5dcbf2da7b0b2d83db88747fb0ccaa8991d24fccdef42111ff402ed0d9bdb8a4ad13f8fcff6a1df56c82a5f88f575f49a06275a9e66067f15daec402ed877048499909b9e76e5fde52feac944e1de7894cf13c515299acc6442d90f027317b0713805a9512256bcaa7963b9429a510c5869792c1e29082921d0e7d0cefffc34d30762fb83e2abb7821fab4ca89d08b497f75e3149a5cd3d23b29bc52137d8be9c4a95c6376f62ed64fdc159b1bb6c842bd07f8cf03f7f2eb
" ~k:"48ea48cad85abe488665eb75359217b63387427093318bdfb5d7d8092d342caa
" ~r:"11b16351f8f72031ba2a772000ac8726a479e1be4523a9eefabe23947a1df0d9
" ~s:"2660fbb44e29e7687c10e29de96fa1ab03c087ccce086cddab48ec63774141c1

" ; case ~msg:"4c37a4c8b41109240c4f53d87277d3c790b2f071105d15aa10bd0f7709da274ccea1961e0b99635b31acd2c80530d2b403d7110ad7cd0e3572518909c136e73e57d38c1c7443e58a257f0736b9f6f51da8fd1ae9213e8193003d69583381f020cce7fc59ba1b1ed5541dbef6b59925750d50b6515a977aa4325d5fade42f8287
" ~x:"6bc051fba93b92859a8a06eb361f348f5e50d091c55b998476ecaa1777f26fb8
" ~y:"88a4dd593b64a4ebcb27ffea3de9a7ed7801f9672b5c8dc27b383d6cba58b4f00181634d05eb490282ce4e57f0940373d3a7bd7e9ccaa9bb2965322ab5fb21b4327b47ef4e2b42424c1383bbd8558b506a7bf5537b049fff35c558bcc739b760443728c090c34d6d4eba81e24e42394f8fb826f7c92ca71a9dab16e999274726b0c5d8f72fb9141870dac0bb9ec0429802b629ad71ae0560e5862ecf3eaba9c2a584885b32c684f6d55fd1b090d93d036a4e9858a4d89b9b5750849d926c519120131d456fce9d247341eb17336ca9729a9080ac5b1272fbf707526afc8ae6a8c661ef3c151845f6ee0902de9abb4322afe585e59d6d418e87d7cdce4897ccac81d013fd72dae1a5557762527312587ca676f0e0676000fc0c76b8265842f2db7e18e621c0e3c2ca9295e9e36ec8ce1c85097ca5fffa62e7b896bb16836d063386b1e663ef29ec1702965a7e0562d2d282f80952d7476b322ffa7929a453a638ea3bede802ff5f8f566085a6e2a2414ef7a6f117ac8628486b23603b1408faae
" ~k:"40c6be904308e25af6616fe77c23e6e6570ac32ba5bf54aa81f6773a5071a904
" ~r:"23e13a35777c189ae56509c7afb411b31307737e2ffc8db3f208940c5e76edb3
" ~s:"0544758362cbb61d66b66826958aca63af1b8ad615a49ba557923959b68f8228

" ; case ~msg:"443473d615bedcba2c8d9a9a45a28c428d7f1a26ab14705627d9ad13f53b767cbb60be523fc21a99c373bd7761817b314290f2f6a80e06e12cce238954c648ace50f3b0dfdf71dc308e1a8ee1159fc1f19b73ab6015d186d9b6bad965a9ad62e440a9ced13550a444b5f0400b96e2d238e9e3dc6e6de12f44205d4fd57f60e9d
" ~x:"0bdf6ed048358dcc9a2dc555c3d45ac394571135ab36168d9fc4ffe4a3529a80
" ~y:"0f4ec6e2baaea9c81e90700519f2f05f545ddc0ae9bd3a091e8b6ba5255c15fce5ef3c046771c5f31bb01de4377e142831ac1749903f9317c7b01a990714985f9251198c829073205924c568050acd6dcca757618cd2809bb7aab64db1e86ca92eeb854120c9d89fb9363596be9cbbaf8eacae2f18f3ed483589eb466a5144824feb1f88c30cfcbb7628f7cb4159ce32e7c2ed04d0ff0481c958e5ff744522944cf32020389b32959b5e12f80f08064908a270f8695a3f99e75e7e85ba3b3c773f04ef9e09e76b6c47302e41d50ead04541e0fca4a42502722265f82ff60ef46aa7547f9de249135dd077f24a4e7e03be2e3094772767a9760883c520816fae637c030956ea25f0a869e4a00a4a8017bcb72b2f2fd83643bdc01d8ff2868d3caf100ae8b818b926c96a850bd69d8931dbfdcff31c67c537c4f5959d04b744a346647066dcc61f63be6251b590d688ae3c9b53f392007d8584e4624ffd2941650a31dcd5abfae7ca120b11c8d0194be96e8dd09b643d5685d1065d98f39b6ed7c
" ~k:"63066e05d16e79fd013a6ae456aa3f036e9d58675b3c9e08a412420a64c1f977
" ~r:"76bb0ecb9faec7c971137ea6feacf1792073ae80be1ca8ed9cec2a5ca6cd510f
" ~s:"34920246730e0974fb0faa57e77fc50ab78726c8e51579a0ef5ebe3fce3ba7cb

" ; case ~msg:"cee06f792332080d6e73b3f02f5ec16996b66995beab4a2ba092f40d85c8ac1accf54fba068128c8cdbada209360776a7706455015e73e92c624ada1dfa62ec794cf2a1a9294f3fb55994bc5211add1c685d9a54acd5bcd830d9a4fcff29aec5001c3b2b2a9706046f38bfe48e8522768f1c6f08a8e240e123ed30e20fc46c19
" ~x:"530e54bf51b3ea012e76206b5ff53c1d5ae6f3433f2a50b66b6947e84dd52171
" ~y:"5c9205fb649d3b4ba2d44c80a925e30d27b05bd339f1ce35e0d0419a91ed31fd108c51a2a62cf9d0adfd877d27cf5575e43ac7bfcfceec5673736cae089516df8eb1ea6b563198b24a6e2522f320b123bfb250d43b600df9298e121b6c5d2e637a989215e095e603ee6d4e8a2dcd17b908918aa514c86a33d8c717578d861261da43f73250ff2be746c6916fc72871fb42a279d22595051b8ac04afbf2013063e31661b117c5d094b4c232b22f21d2c65d6361290c08f12befd1f5a2b9b5259af0435b97b4328297c252d813499f5209dfa35e9198de68501af4ca8658942d059bb62b8e55a3ce6120a78ee098132e8d2dc3757f7e60f8c08c4e43feac67abcddd1ea2f016839fb1a0f797b8b137ab43b64508ef69f6ae0f3abc4ed682aa7e38fa5146fec62e01e0951a6e81152de43171ca8869fa1a42a4fb2d8ae512c005fd97d12bb13f299ab9f5321ee2fc39b28e61c9ebcb91ecd2b610fd8291f538a00d06d057c3e79422a931279fed9d93b0b6533fae441e98413025fb4fa73cdefa80
" ~k:"4e500b513c2e24ad17fcb8cc0d6a8c54e654e00a892545a33cb5af8877589520
" ~r:"6d02536db546f2bb1f65ff0b91b964802b38d171e678054ee41f2b8563809cfa
" ~s:"6bc51120e35c955ab8f717f8930d8cc8def8505415cf159d2516f96578842f31

" ; case ~msg:"58aba24e9481d1151b574b146ac21b17110ed0b9bfaa55a4e2e06dcdc18bd10cdfafac047189f5ba9f10377affb40a514d528a3483fe8e64b831ea0cd076ce583942b938a4b257d0b5a92412e01dfda8217d5f8054596a61d5737d8ad8112ae228220e3bff60e2e891d03d53fb14f14dd91975dc15d6b7bd62e99d74ef3839fd
" ~x:"7bc2316b6301b772b6742d0c50f2c1c39bbede01448026b6a201793bfe7dc3e3
" ~y:"2adbc07d8aee284ec982c4b95e1ec3ec3f5fd5172368ddf83f9a3c69655291dee6b99ed713e5a1fec338239b8199c5a3bb2b5e2e7f23fc795058a9ac70ebdff2b3daaffa389e97fee35f174961f12d634e8b8250b8b770b8d7113d0fbc020b7b108f8d6b2d7cb6c59e2e151015145a8e374f9b7396e970d91e3c1f85ce23dcae12b2f53741cfc2350b582ca87f0ff9ab50ad0ca2879e216e61a5c358970a3c3528dcd9ece6b83d525b31fe687696a2a2c65e34f2854fea6ff92244d27500f7da946c37169756f4a4664b2909611549ad2b93ebacebfc270ecf4204e6641dbce05da2c000a4ca5ac885406ba155807494706180d54cc012ce06e734024f4ccd882bddd2257afb5c287bc3a8570edf21a20afeda0c762ad696fba177a5f2f9d609355cb91d72ccac8bb9e7c3cff1834d86b0772aec741d7b4b3c3e43bb26ec9f5e86b8685ea5c625b6aea450a46e85e380b158de6aaa2701ffad0c7d1ed0df355d09d06fe1758b2f27a5d02aa283aec9fd12d3b62d504dca0b6632e89fc55fb083
" ~k:"0c2a0d2b326ad63e869384e3e2e32fcff8db83285fa0a5b9a7b13589a7dd7fc1
" ~r:"2c462d49344f3ad03b6798f96452f7d66351cead919e8201b7665c877f8255bb
" ~s:"50e8908a1c6684a2caa8aafb432cda4b7699008c72d8d622c3da4171e51cfdbf

" ; case ~msg:"e10604ca00728e533621dbb6618b0c877c4902a2ed79aaf40a4daa34d6cc216ad4648daab6cc1e18451bb94e6a1c0c6f9d0d883962eebd507da099788008da23205e3b4e90fad9ae857074ffeac63430c0facbae489c54c957db09d53e12b656cc278615a3a5612af4c2f168bdeb118a42a2a67103fac321adf5688b05848f7c
" ~x:"6dd500e2aece9ae331df269c26a4e5d58fc6be3963f5002e36bd9cd04c1adbbd
" ~y:"b0448d43c520377b7df214969f59ffd4e0010c12d7e5fa8f241e9ce1c634439c94700ed5742a8322d405dd05de9953447831c7674e5ae1b89041fb8f2ec1054b928c64ab862f021a55ebce838d2a3d2c7645ec7c0a1a4603617e4f508929144c1ef2b039bc78b59362d5ba9537906e66c8e9c9a3c68e71b35d88b8bac86cacbebd962c66e18129637fad2d98d21e45a32672649492f131bae88c9989bd6372e17492bedff4d9b091b3dd00ebca6bcc49148480589f9593e32795299f3de7e09d88bc0ed27b7ef22ef7d202207fb5ce8c91712c3bd5e758d2822809ea5d2cb288332aa0760368259281a34447ff5a98c9c97c1d58383cd14f6d59bb5e5763217b2337ec2321268197f02cecd0d9fd93db39f8059a38bbb35792ba0d4ed1bad95a05b481c39f6adc9017dec1d662b0803f2ecf0845935765f9356db536c8c11887d9e44b73b6996ae7ac24fcad9c23017e5c2aca88b5a136b6307298b85ff010f964b7477a4f980800e69d3cc0f438aff7f2df8ac61d64435ffaf5e46633609e87
" ~k:"65243ccac0a014b9e52638171b4a88b02a8c6e617ab9467da523487122e6650c
" ~r:"56ab9947ac94fe3df7e35801660f68753b0b620a26594cb8fd375be3ea4dbf05
" ~s:"608ed1835139af29a2e3d874df465edd8d6428f40357d9ae4904efe8bccbd035

" ; case ~msg:"8af31f66772fb0c31a8c5b28e568e6368cb66b591edfb0db867fd99e83feb3638bc80f0b14483d069e8f2e167c8b0f10cd6b45d039b7d6f833bd58d99b00597aeef82fa3aae2e55ded62ab660810de0fe1c92d53adf98c838c18fd76a273ea12119d675af727011869943d765b96ef266270b4f89ac72edadcf707a4a21b7533
" ~x:"145abce3eaa8fa6b670afd658ba0c14fa98d2d20e1422367d4455967f9844858
" ~y:"22f3db9ea369938ed750d5ed3781368d594e62635c6b6e103d6db489a9972f398203abb973d5ad9c0dc110586978d2061483c0202738ceb01a665dd22fc568cbdff2148ae664dffbf888e4dda5a04fd3e89398b4f1ffc3a3813ae94da1f8965efbe7f300948749e9757cc7c05f6e53fdbff994c223aba2c137151b6a320f5b7f8cdd6003baa66020162990624099f3cf56d68b74e96ee09240f2cf11e3954e75b261ef9e8e3551c6c00f41e9eb17421203a4565388c321c1325f72eb10c28a9deeddcb4806f625382b37f0becf77936b7f83d26bf1ee1fe05e8a0005a4058c678eb569e339423e7c844305f4a18b1160a0c430513fad715896b62b9d6e2468232ae375f5f3c0056245eb4616ba11a6029410a955af09f07595fefa03e5516c95a4cfcd66046be2a4f7b3ab274b21c0a4f126c482c934c79dcbbd6916f3b887b2600472495c8335de121c7720f29ae56f5ccf9b99c9ce5655c5e1d15d67895af0dee586bc491a97241f7eff434bb79aad831db0695781e6b512e8702407a7d748
" ~k:"06f085f77088ec97cebe5397a588369e3dc15b70f2a5316a6dd5f94967fe3dbc
" ~r:"0fda7a8a3e5d324fc0a1c2841cd22f98757a0c6a2a465b0d9d65bda9b23b3c1a
" ~s:"40860265229085453fe58487a933edf3c28433694c7b85f6370d9a4783168237

" ; case ~msg:"e2456ef5d465731b976f2ad1fc94634c0569a0ff7566a49d47d69e60b3b6d7eb2ab25cd49c931299796bff7e9774075ea20a972e3949a29dfb50b2b5613b45c596ca5dab282ff183f564a06311a49601a1e8560d43c6a481ce713f46c6ea85bf4c16489fbd72cf552b26516298bc66942a05d5a8e6d0f6a88f3e678d310e297b
" ~x:"286d3cec1d2ad2a85f0f163245267438f7d7d62149ba9e59a18dfffaefe44358
" ~y:"dc9d68b53f35c29f7ca003a2583ec8f8ef5d78a0e45db3c884d35df4fb531a080ee3831bffd3c756ea5042c7614570fba2f6ca4870db4a453d0f793fb4d0225d94f27412dbdf43432f52cb8f867fe5f492a8876d7bd850d899ba2f0a53820c440841fe0cb76fe0444bd6c3235785a3da3081fef99f53a195314aefe955f2964c56506fcc969b67b323766d299c0b02981c72a2ce3d7524ae6f08458795fd32e31b47aa1f974e356081163cb23efd73a9e655deefe5e734ceb58e88a9dbb524eff7e11c3e30680702d8560dd8b6ad9f61e7246c6dde164e914951d6a0573152ec8bdea679dca1985bcf267304d5f1bce2f32bb9946a056857359afbaffa59bee61ad960c567efe3f1145a8a87c2491fa6b33f7e71fcdd8f1ffbcd2b89920907d1144a8cf0573f5b89217bc0598c6e1754f1ae7d9d42a608a051621419da91d11bda9bb9dfa7118e4b663e7bffe6e9946c77ce9f8086dfc822a7ef728888b31654a19b6debd2ca62f5e3b4e289810435b363ecab511f47e9e157f0f4198862ca13
" ~k:"183ca3afd082bf3de19e89faffc5cfa7dd713a873c02c723279b3091f9bc627c
" ~r:"778640ce75da584a6a83f9794c4ffdbe30411be43027758c74f89f7ccc7f3983
" ~s:"6125481e103f7803b2f16d9a4d00f881e0b367024df5822f7cbeb5711e0e4401" ]

let sha224_n256_cases2 =
  (* [mod = L=3072, N=256, SHA-224] *)
  let domain = params
~p:"f63b3cdd646d8e7ddb57216aa6eec2134d707488a1f29cfa9970645f1227ea5db2e318eea5da1687c7ed90509669345ed6134cff32203ab72aecbfa693d216aeb55d8d28a981f4abff07d1319a799be5dd746f84842817929c305b408598af12045daa2f1ccc8be4d81b513c630f017fec1658aca108a1af6120ec05e3018c4253c9dd35bce062b73d0f2a93d41c481a5c43bb97909682d39a9a60dc3c35e36375dec6ced0d2db3ba0d111bedea701a0e4753624977a9e75b70a74e2b81e38a52ab22da131b35416d3cec9663079746a763476e57598142e39861545daaf8d38a176f26c71f5afebd9c5620da80cf3452b55c37c661b4a1ec0351710b9de4a3cbe0b98b4d9ec89128d97aa7efb19db8ba43cc0be25c200f90e1506cb78ec0c336d7a95613d4204e8ed68d0f0a6c78420105a8d2d438fbd2551a64a1a0b03ffb878742f8c9979cfa87394150281998d51701d5fcfa9696a4989fd25f400955e626b1abe926c0afa69aa6981900effcdd030592f82b2042a47a9a5a8cb0283dc4d
" ~q:"80000000ba4634b5fa4da054bd0ca48ae490e57711f381193842429159ba7ca1
" ~g:"8ad4553c4e49aa24728ab5024417b132d2ca53a55d959458f2f759adb0435beeefa3a2cfcd0038e2420643fc4a4deeb5d9feaa1edf21193b40e14b42982a94f35c58b81147d7189d263c9b12fe63ab9fa5f6f03a2860c186432e3ab04f2ab0f2fb6147bd9bf7ed5d20713b9da21383e2c3a168e7d09d3d8a5a058fd23095b5acfeb864a3306be2425fa1ad32ad6d9382e603b03c68af4af0246397102c4155cba811abf99da7839e77b2eac9970588ca1d0a2361723a164ac9229c2e80dcfa8db4f9e29803effb3168c7fed7a3a6de40dda19a0536af9b5b7afaefb9c70d6ae8df12da658f6236043aea873db29ceb6f07d108f5225687bd0c30e3084e2090b45ae2f92a97b8ecb7a9705c4956b8b31c4a3d61107c84e47adda6c80d5d22dab3d859220f9d5aab13677ae3df168f0c176d176b54506c639853f04ddef2722f39c18e5ce426e14562ad8ff26247af88870efb72c0cce836de8fee67a662378245b502bf1f83099988a093ce7cdc81364c78b1f4a51b800df6137c71d65e6b089a"
  in
  let case = case_of ~domain ~hash:Digestif.sha224 in

[ case ~msg:"957973fc3f3fe3f559065be5d4a0c281cf17959018b9a670d2b3706d41d5812e37301005f8b70ebd2fba3c40a3f377a751b6cb9693e3cb00d92888247d07921d3c1e9257ce08733b8926e0df7bdb6e855f1f851075d4e628d110d42b643b54876e5faa3611477ee68371562555269ed62a9271bad50cc4d46038de2dd41920c2
" ~x:"524a7ea5977f8102b3552930477f5f042401165d4637dcd8b9d13df4f3aae5d0
" ~y:"42243539e49db9ea19d98d97f6f2a94b23529812df889eaabcfeda01ce4c759487fb89bc82da75fe1c9134361f86de47d16d8eee80e56ac502178e8ed8129477af8bfbd8262c5edd937e1a86c0f0e7b2afe7bcbddfcb5814ced0b756a76ca178423bb4d578c5da183712d968582640aa0ec7e9fb56bfd960d7a57549747d8fb7ade47cfe816c1e57da6633dacc537de060813964bb5b2757a312f9da3d84e60aff98170051d3d90e380b8bcc1986c58ff9dc91e8827d4f9f5fc4b2b2e743cf9389ff02dec01f5d434b430d162e891c3355f91855339f8df58300e4c993ae4df8c4318b5c4bd05283ca4b46b7d2fb0f6476bf15907f50dd4141aa7acac9daa62eccd3a67357122060b6cece0446a93eb230ad93bc9a4d1b1efeeca1e3fc83c119785035b439509ffb7968b1a448b7bd8315753fdf04a256eca1562a11b096c90a36b353659cbde4420e17e90b94c43c7519c60641ceec056f897b97d6bb1861268e0dc79b7c3b6b7639c255bf06865737459126cb465bc1da4a043a1963da7d63
" ~k:"29e4d7790e181b4767903fe0eb37757f33f13337c33588c1fdbfba0e655ab621
" ~r:"2e59d5f30f73781d38255b70dedeeb38ae78df4f002c1f747c08deadc6530155
" ~s:"615c55b2df0ca28c60a6b385c58fa036df8c4b2f4f1935730bf8f4f0bed13610

" ; case ~msg:"54071aca28969749ce2e2dc855052019bec27d0dd6a310219311b4b6d822467b22b3f02fb8313993fc77c4af1d76ab9db99b0b2b78204aa45f4032a7d945f93d55bcb8a6bbd47f98299a0929710461419edbe1132dc22575f5afbe7078cf5f05b231000f4a0f9f367d9025ed3ae1786e0183eac93ea96b55304a8c2dbf690821
" ~x:"4b2d62d0e7b88436737d03d6f64dd6a6dd0757021817169ba373e3a31bc12cb7
" ~y:"ef78152efd88130a4fecfe235037de2309b1e2f322d4f4154756caa8f0b3e41be45c80d895de56389257c391307286be8e8709b80186e2724172b0f2974be591584916fc0e750c0caf83d839b5c248f5de658668665f004bab8ad310118835957c02da6ae9a2a79da039adc884f9eb8b62e379e27f549e7f8aff8ad2fc276ece15f0423528a09e31b26421df93573bec7a4d6c2cbfbe5ce0fce0702088fb384ad1dc35bb2c1c742d43d79ad136e71057cb9f22ca042e61d2c5cc4ccf5b75a7379922bc4fd88372d2a8f6a2750865f91c143412a3fc61e4ad4abd03dc1ca0fc4297ab107a1963533a3d80a24ae2ec4146e8265acfd4446fc28103c5047c17796c4148b8e658e44e9b1c259d63c97f0e766fba8d9a7394cdb734508bfa09ae42d2da3068e2c85af2065f618ec3f3c73d73a750c13644c96e3dbbb7474325af48d1d145c28d69f22cbb4a9073059a9c40891804c73a229f01cef0678cf4855d18f900f0253acd6b3ee53dd96c4c92afff1f3087eeb4fba86d2e9495c5f734a46ca2
" ~k:"5af719a9e5d8567dc26576782e8f247517fad5ac5de0f7115c5158748fc73b40
" ~r:"0debcf6c88504a882a0191e6fa4c774c10858362629428aff24c22e3364baa15
" ~s:"53d8c1dbb3a2c1023521b705005ce6350bcf66c093588c35d768fca295a4a9ce

" ; case ~msg:"49d5f20acf1e9d59a656bd163fe46fc868476ccd926377a40ed3d7476e9eb7a8a70c4b88b16e799148d25fa23bd0c91611b76c9665f5722f404fd90efdb8ad14b759c349ff6c830642d51076ccbdc57f152fba41c6a7f3cd3905fa7c857265ffc7596a64dc69490a932b95adbc79a3b4f21b2c6fb5d5835d8bcae5d44d912a0a
" ~x:"4abef24f715cfb3ba6e39c26b07ca46b700aac69fb8df3c0e09be08df90e44e2
" ~y:"8ff13022080316bea49b89a06dd5a971d86e0c9a3af414258a8f485088b66cc38cdea02cdd62096c00eb0d1c2ee662cff16f6d2d30440b2ad9e897b9eb939b1299ff879557f163f17c8ac60d0c6e998b3a044b43fbfac7b0cc30a579a6bda1b4ff598a531f9e37cc1901a7b08e794a7401d0f8ca4be55bff7b176321828575a477686a98b4b17266e101601f436e554b9e4288057970fa3463343e7e52a58ca145ec9befd7be31ea766ed74ac178bccdfee9d29565e7935e8d70c3eb091e3e3b3e6e77716931ed729c49b96443606098bd0810989e0e6f253cf3ec38294231b711b09a941609acc8976819076543926ec4e06f3e4d7f123c2b8771e54589e04524e3b4f950da560a25d12172d4ebdadc1719400d91cf0264708714479200c50ef00ec0e604909a546c95eb2fa53c65ee72ad53f149c938dc2193496db07af3b30a1f439708aa115c8dd47c81c1bc68ea3abd9026113c01eb05558b8a2be9093476f01247bfbeb3f2858b13e6228b98205fa710b6af1c5f71480dee401d7472d7
" ~k:"5472e89286e6ccbed316fe7564e3eae899ed7bfc55ca7fb6fbc392d191304bfa
" ~r:"19a73b049b164dbf7fb2826f4253617cf1c5bb46ffc5204efa00002a79e23c0b
" ~s:"7be137c109e68f337b5a21cb591a87af1cb8681419f875ff8f041e829991fe28

" ; case ~msg:"1190853efb7e04cd4947c1ea5b1b5d9e0ac5e6df1dd050877308f1b2c7e0a4917e588103d28c0f6e8b72d967aa06aca68a986d807740f2dddde7281e550af4f637eadf61f880c7351b486615096f6ba50d8754bbf9ba1c49a3485815ef06b3cd761b5586c3fc2b464c6fe12c160ab0f6f446fabf74212430cec15e75a57b102e
" ~x:"7b2510c73ea6447bc319de79afebcf45482917042a3ca3c1cb1c97d1a1216b2b
" ~y:"9bb81c80d2b8a601a09e22475d70d1dc5513409fb4668b176c76b3aa1af8630ac7790a4444ab823787f6f569bdf02b9eef5e7bb21a88e3d3296857e91919f3c473add16bcd763f31a2f9844d7cbd8d480672a036c4b104be66acd66e6ef0e8a744b3d878090d1de9f105560247c62153e117efa55ec61c177cd82f8d72c51d253f4dc7336f79826025619fb2103f91144f90f6a689abcc51c68affd28462578b183eec942058f48abf546f738940a6c26d301c4b90ca40ea49c117d61147e8683989baed7a221c4f22092f72b1ed604b6aa94ff6a574b4215bd6f8e9d7b638afa435a3346589a61b1d1db2989d7b45f3234545e8a22d605ad6cb036ef791f625d2c6a995eda3e0cafce704a2bf15ab5dfad0162104592d23f52aa0fea1f432f0a308d16a45e1f41f823262074e9173754ceba70cd8a370dbab1a14f84159116da73d3a9cf82594cb3af95797cf444272850589acc6bca471d076335d67c461db602395bfb17c39bfa24df140c0ac4388db0534a50dfd261374f81b310f751d16
" ~k:"0fd7617bfdc671127a1d7465f683b98d8951a741f85d43cf5a5bef9232a16ae8
" ~r:"587d7f4454d59418a7527570f28f1b07451f3baf28f5cabe0310c4d79e4253a5
" ~s:"18839404aaad59ff24d6accec3b7cc6ac7003dd4adf96b77bab068ae72f25f61

" ; case ~msg:"b1cb430c5a1d72788c795ab567a84c7f5977965933a5bf238058f2fc818880d25b4ddef9635481fd9fdd4598aecec3764fa73093a225d4e4ebcf01e4b75bdc1841dc01652c4d9916afa24b89c2d6854b72eaa7b1f3089d1a919210831ac80f99835790ce64abc34270cd4551d31b8f5348ce8a70df60b88e085a984acac665a7
" ~x:"403b2137ade39c1e5b817ffbd0bc3448024089fc1925550b5b860403e7ba65bc
" ~y:"a81a54be0685f33505aed9591f333a74a842995da5135fa48f5053fac29fff08afd9b901c3df1347204a3f133a7dff6b1adbab077526b638a63837d7844339d48fe107af08ed62e87de547ced84df9a2ccc45876b29bc5361ce8a9a21b81d4f85d3b671c9b44b5483f2610efa01751d3a07fd694e46653ac47ac64a910b7fc421f07e5de54e898789989091e9ed58b7c04e9e1dced60475dc693a0eb4015ed658110b82f8e720dc7afff69cea7b8e56b8a9755bf1e2933d083608377504cab52d38cce1ba82f84c26265e693f18cf52e930dc0d8bc9d41f4d28b32b7405cb1fce88a55be40dca1b1a351aa7d77fa6ef84c776fa301dba2e236933d89c8b944f53403414df0d434db72caa749fbcd566d76f4f6f0bc40e42a29aebe6210e89fa0ca8b6ac08a4cac65c590503533c3e4f1b3c5bde868e79d9da918b72d1b098a7278769546b78450e00e46dd400efe97c884db9612baaaeee2486f64cd8302a4c32d8fdb873fe0afffd7bb74811220b01339dfc5e567c766af2805ec1c30126399
" ~k:"2c1ca8b5ce7247dca6173fbaf854d00020ded6300311f53ebec8eccef9570d07
" ~r:"60d2763f0138076e9e0e20f83e4aa2e9aa352c19ca79e3726303fe89b12e27f2
" ~s:"07e08d916c8a10ba269dc460ee9d83f86a7b3d98621bb7324a6a7e607238baa3

" ; case ~msg:"3bb9430eea6979129be745d5ae6babd4966e3abf7d9ee5856f2caae6014cb340eebd28bd9f391eb46b3a2b8a4cdc224e5508532ca08cb104aff677133cf4393a20fe4499967dfa64515455930c659d43bbee2340b14a3b3342d4b9a466b889e850dff4b2a51d389ca32fb6a5f433ed93032be4e563695797b8c1e1e019184172
" ~x:"0d3fc8fc4c59971a963e8e41d26a86499c962615c64abe011e88e590bddd3b0b
" ~y:"75b765eca4ebde0b6564c3137f16cdae00eeadd2d0b2cb83cd1500cd05ed0dd16730c9501c8a353a64634d065f6137ffcf9563d96127906fb17d5a79ad291024a4a6fb7e7d080219a6231ca158b65f5202912ddcb8dd1f018c9b0e76b3a476336c5041bc502f8acb748f136c3d78cb2c429c8f1ac17b63dd7e9e57b607f9debe571459df3688cf4c11fa1e84533aecda2dfece05f4bdb268cc7b0c8fe7af5a633a83515ada95f31824d6a3c7122fdcd12f54992cbe64d1d6bdbd0ab5ae4d19aa52609750a1de186afab5a16398da473d128882b065e873809fae0bbdc01a9c73b5c6ee65857fa794a15058ddfb24a9a17a0408646f2009dda610c8291ae148a18c173f836b197c78ede5654895b45a3419e9c3177f2503a93ce526be14ad919939ebe3f2d07f006a0b022d6a623c6017f0c76619f0780531d5390d4239b2f900efb44c9530c7d9b3e84a70c904b179ad0c4f909250f7ccf83c5f42d6437cbc9f03fbae8131a12d33e01721e650aee91e1c893f5e7e039e0d585cd7cd7495c40d
" ~k:"32d9aa04b104b5d7b59a122b368fe0cf476e28098b898662a78efee764545ea5
" ~r:"7416729a1f60208b7f837480fba81840e45b338ab9846e9bbb9168229f64bcea
" ~s:"58eb904076a3ac6907d750ff6cdfaa465435e9982ecbdf72197b09bb6df1373a

" ; case ~msg:"55a69fc16f6b753d0bf65e844d067859f51dd329279980196063fb59f89bd778a9244f932c2adb6811183612105d1c527e8302dfee5042cfce5dbeab165a396f5a4c21339be1021b7ecec66f2177f94243ef6261608c56919679d44863cf9d2afc6010fc2bf821b931ca3970d69b1e622a908389db5049d718e357071063aef8
" ~x:"3dd224f00ee1d4648c600b10ba05ff36ad2c06ddc5a9f0112e0331ae958f36af
" ~y:"6146a51deb79957a83b2c7a3204b5c34ae4f8e0db60f0c07e70803f22bf99a39647263db9e285d72f6270ee10f18584c39081d2544d40502c50df1e35a457600b5569d61e8126c055f7b964572e9f3282e4d9745006955c24261c68d7c0cb3f08b0b0d8eaa971e1a631c68a3a914d35efe89f76b9c2116afb7bd1989e202e092b5b570eaefcc933542e650d92c033b5973821d6d77cfc243f744da80b56eaea7650bf50802516228ad6d5b0d4e889c575e3678ffdb1c289e59d9ff7f84a3d63d39d6888dbe213e2c3b3114085e006ad74505739fce826f963284dc4e2b01ec2f9233d3470e82d872ed944e62961f64134e8080daf2df494a76240ac0cd22f9afae7e80d3cf3efbe055147f62ff8c6192e388b49e47d9feaf19eccd65dca9991638ebd7b048077707adab1cb2a4358eefc4aab8251fb0f9d5f0b09f299c720d3a8c00a5a4d84feec040057040b709cc0ed185a832537bc4b2df0ec1f77169ac96e91282de21f342d5429ec3d66ad9d336c440949a1211217bf54aad93bb4b0a43
" ~k:"7969d08c0cafe4019b64ad3e6614be0aaabc2c2be61b3b3dcdd10d5f75fa24bb
" ~r:"136f93dcc7d33e559b8db0af13e00c7190928bff5086eedfd11706e6f2349ad0
" ~s:"32b95b9b147c7d1ac2a2f0057fc0538a4b7c9cd4652e6783e5d7e3534655631a

" ; case ~msg:"1567890c69e578a27d6208913dfbc20eddc61f5feed457400693dd170f8067bf290b11150780684c20d5cfd2bf1d536dd3b70025883fb41703436fd09c0a141125784f9091151303ef80cd345e5a7d2854335c2984538c5cd739b007248cd99f1dbcd3148cb0ff0db633f8cafc7a0b99c61e784d0303a5120307d3fb3c4c219e
" ~x:"3ac374b2a4940d92ab35ebb8e59677fbf95980fe632ffbb1db4f385ee4e13a15
" ~y:"5c53d13a1bee17a28720b7089646d07a3fd58b9b2b23ec94af3144830746177b0d2073707b6b84901ffaa7a4165ceff2425640fcfe5d17650a44a168ebd769c833445f1b2d26434c228c1e2edf1704d711a86257be25235a7cea1e5cbac412235b7596d1dfa0398081a4f18151cbb51dc62c226a2abcaf3335e86ab54608040ee814e443b64398213ba60d7b5a3c8ea78ec6b98934c89aca05b97df5f65bc574a30acddd09f73cec14528be49a2fbeca70291b1b29f7042c594994da128fda22b3ed3a935a1a00575ff1ffd193c4cac53a2a2d4b0c510228a76a74333607d15b568614427144b4174da358e383f658c60b45710036f54f93f17bc808b302674e838c1dfd7f816f7ea44b0d97386e4e1634c9539568dd6ae1c28f25b27aa94499ae389a0926c8fa62956c6e24dced0afb0491dd9fac0516d27fd4d2dd0150ee6b4cff7bfd575043d701daad0f1b942a0e4c61956b32a68c9078f6077fa9945198d447a5bf3c47b7288427edc6f99655aeadf8de18515714c6b9c0d4ce5ab092c2
" ~k:"7ca690c92c8d4a3ac1d5255a2e5a12922093b8b2ee95906eab29b67f84fd21cc
" ~r:"4947d36e7426f1441be5a75dc9cd845450c61104f19ed40ce33e252fa2c26268
" ~s:"356879deb1daef01da04750d58e598db47aaaff50b1cf42d87334a615780ff8c

" ; case ~msg:"4f7d894dfb7d82040a9fed6c26a7d27a9a1511388c113c64715a06dc46fcf4f904070a6ed95bdd8dc1730a27645d37eb3b02847cb1c631ec0c67b2ee07b8805b34dd9b84e6ab3f9afb9246994ea579567a8f4af7feb86898cc9cb534c387993c6ec16584ac85bed36bbc2c305770f21163686167dd53fe562362ff549d903539
" ~x:"2c14cd975bc163f9740dcb4a5ba9d8529c5a075016e02400dbfede8dd4f0d245
" ~y:"00967478358d7c1696ceb92be850f5538ad8543e151aadd84caba1b72f3636a2092a86b6462873903d5bf17f612b45b5133eac1630bf07c0371423d2e5d7147ceacc9baa8cb3b04cbc3cbda429ab40d7e592730dc477b0a95f1fb5ed5d91e14b9d5a1ac8d403a55a658d1c383bb598053be238cd82386968aedb811586fa2a14119324896f2111b9bc7cff666d37affe76041d98f362daa09ff65e82e865eb29c5d4710ca7800886887d383da0cb599b225fdd210a3d70929d35fb9ca807e56c91c0851252b95c07b6b120b3b650418e0f54f45736f82018d09294462dde6eeafcb15a2a728577faf3ef3eb13db044965ea3892f7eb0884e47766089d2a43abc62a3c375831c20848dfde8f83c249a8e27f2897cafcf5a06b7c3591e09b42f82849d498664f485de26c788e559ad5b15f999db927f81f54b96e997b9096b2a7e3e756f5a9aab54c160cfc2e64492179487c98d0aa38308d67428f3a113228bc6dcdf7ab93cbb1da225c72c636f49d27442cf3cf2f9c49b90ac8bafe740dbbfd5
" ~k:"141936264e075533a96952808935238d715e7cbd840c016ee7a9f508608e4808
" ~r:"0940724855a0671d60147dc61fd283190134a68c178114d59ab58da73a1c8182
" ~s:"43f194b97078dc9b84c8e8e867a74bafdc2211706ae110b5aec0b99ede1ffed8

" ; case ~msg:"9b62a74bc49ef4ff5c62165e7d25521f135c836bc4ef023fb4bb1d6b42c6291071eae0b465c59231cb297cac6d145875fd84f5729f79f92218522b9e55cb70d471030d36291a24925ab731a2d4458cff677079d207ce865b3d5526009238861d64506a92b76baff59b37b863087558d5965d76685f0fbd1fab1b1f9561f8f69c
" ~x:"70e12e51a254831bdec081a8882e5a24d78b48b6dd163727b93f803734e06a3e
" ~y:"75d7d9a5dbdd47cecd12f69ab212dfe08a9656e2bca92c81db2d268a293a325e511cd5aa1ba59deef2ab6311665dda58230d48f1416371de1a8364b38f5ad599c472d363a18a2c13d572cf849be2fef9a166e838aa58b721ecfc4b361fdab1d0876b78e2e8f23ef1c82cc0e1700fa015a4007b1d7b535c82d23c129d1d1c9c4afe875a06c05f71f078cb8d9060f4d936671faee217d4045525d570b0c8ca0c4e8b55dfe9b780ba69c9d8cda10c50fd61c4e7214b943c1c29797b099f57a4c648597cedd9d909bc584a9b754b209515dbfa0fecce2ad05c848e99dca21a6d0d5f2dac2361e4c0eaf96df199ad2888d671974ef05d65c92788434ab42f1f1f79edc49ec1fa921395bd0feb6a9e6a0622e8255b0ef6937b89d0cccd5852872d2b0ab5d79c2f198bff6b8aa38acee21d6c3add5562d84d968758d93e8c1d611f7d6182b62e44f57df342899bb564a794d13915882143d9df45f8f21cc030af3397e9e949683ddd8d8da9909cc1139619e4b7b252aadd02c66a5e20105adf26f2f021
" ~k:"5de7fe70b5c60ec0ba66ade4fb6b0c925d1d56d26d6f57c5d12d07b5f6f800ed
" ~r:"01e3de398b018a694780ddc6ca12b78dc55e7ad9fdddb5a3f5b2cad0103253dd
" ~s:"03c98280abe3050a67f88ef29fb214a80124f47321c62e41e3905b8532f4936c

" ; case ~msg:"6c66051e04c2e6aaa43de9aa42cd9f61e8329c124ed3031b67452db4c435db291d756ba6ef90ab06307cb8d70f3496792e633bf5ac985c37c43bdb4e455c7f761a5ee450567f85cc977e968e7fa42a428c1a7e915c82874865631d8078899377255947c344618297b83c96114d11d74d8cd579b553667cac1d97aea4d1684987
" ~x:"11d2f10293c3884b1e28a600dd71b2ed37bea133255a0f97e641f9530bb4693c
" ~y:"ed2e10a44316d677467d79947bec9e405d30f32d860a1ce46b366845df9ad222b0f992f5844571b196a310d587fffa74bd510215f3bdafa1c93d1b1315246fd2f794c4da07bd722a98dda9a02ad4255b6d5267738256cb8639a145c28404562add2bc7691dac12600ba9f8ebe00614ee3fc6e6b2484d9c5c7090b3f3b134ba19909864563040fe8752d6c6ab95111fe1014bf7bbe4e674c9d03bb8d229e4b5f6a6e471c678b0265e88ccad7960fffae700f3a75e61a24ea882b970535eb7017e16c48ce9e2bc8357f7f0889c871d0b4ce29d279afd1d114998d1eb6fe4a5661e429b1327f0a39e9ef00a41a74fe479b90fdda21d9315555afe227274c11a71c0d10c9e5dfc89750eda53c6a8b52a5272c75526375e5fb91ff75d028df7aa2bceb5fdf6f8e3bc1ec3f1e226d04df1d842e4c8f458988cb7415f0d2ca4498b0cd67e8b085b008fc4ca064393a0df517f0b4833ea4051ac3f1de5686dccb7bba8bd939092d6d78fa08f5bf9bf6f13d7aef72f047fcc47a88223df6e1a62d218169f
" ~k:"2621703fb8f5094bc68eea72d5b5caf26f8ea3a173158b8d3e7f9565296767f0
" ~r:"31f2c86287e572ff4d07421a58dc7b3d727de113769952b6d8d736088b36a825
" ~s:"30acbd1c4cd6aa666ee52b0bdc41fc3b239b60d57e279b3f5483c4d54bdd97a6

" ; case ~msg:"5f8d7f283af00384a519769029d208b61eee0e1cb21ce9fb80e9d8596b894580da7ab3457429e72dfa64e7cb839414de344da21cff55b1b3a83189d208ad2089b35abd78e2416bceb66466762fd7ab9c234c4aec3872cbc8443c92b8ce4ee4595425e746e4b6f7972ebd5d065fb3fdc5e329e8a87ed3cddbe279d57227ae4b13
" ~x:"1de925bf532a50dc7a10984bd1dbd90500ec9ad22df0f2d6f185fd1ba8060d37
" ~y:"aa4d065270c38bdf996b1f5f1ee4b67a76ef1e7b134ea21fd0a6137521245052e74954b96544c700d40f36248ff29a712a098d80ca12e28fdd7901a622dd0988e1c4d67de4c497a957882ce992fcb08c5b85c6858447ed6fcbad26d8c40485f0a89d9d020fe233e89319038455644c828d608df9707c63170dd0618c0baef3eca8d1455460a2eb25faff444f803bca297bb680e5f0fd06e887ed50c8060f55d0160ec64517086f4e1d624ab7d12df1b5947017e622ebbcd6f4eddb0a41dcba82743efdc5804288d2863f54003eea12753246e6e0357df05501b195fdf3a7761c4c3acf26537bf98b32f2e72ff1e0159d046bbc053171e3d518344f0537f2e7200bcdd957ec96365caf55fcd246afe771709ecec28348a356a1d4eb136a176adb5fa102f5fa5c969f90896462e0677afc606a948b25587c10316d22e1269fc64f915a796c965b8be97e5beab047ca5198bf2ff856df740afbbc1aefafefb1ed47278b150e6a7222417d3a86494bdb51edd0616899526c27acc2a818e83baf579b
" ~k:"5f0d6676776f40cfd5ca255fd8e32b10bf3472b193818914876d4c3be68a83b9
" ~r:"7187cae836823618f9a6e847055ca2bc38c86e726d02d38f4950eb6b71b36bcb
" ~s:"21f6ff4175765430e2dbed342a85d30056b28905744ece5dad79755ee3d7bbbd

" ; case ~msg:"b216a035b0ff29feaf7d4c34eeb1604155c90338006753ee2b36062d72f62b524504659f70b976c68952a62c2b9a2a00cf0066a5e5098a632df2ee56dd1a140a98f7b3ac12db3576b610d76563e4621637da1098aa20f3c83247b7278860417cecf7e137194cf1bae12bbc63a7bae02c906d503f694dea3bd534718e37704962
" ~x:"3bae9330b47aab85cec948f944ac13221ad35d859de56db56c31aae88345cbea
" ~y:"7d6b3b71b1415807d15901427e6ab002ee985ce7c8d844969c6e7294a2167b4c26171bcd646f0d1bce14df05e4ce58a3ae50b2aba5fb74455233fa6d179a0794cb26e92ca910cd1c16e5464e8fa7ba936341d3ac211ac1f8a2f2a19c148a1c3d6b00ac44c35ea345a3ff73ae9d5abcc6ab65162a53daabdf6da25f96958eaf89f559895cbec52351394f9132c9564d61aac792640f11e09aa6f6cde9ee9ca5e05fd902911163817177bf054cf2eabf7ce8f34bb1c4aded8dad93411fb276d2d0a296799661307de579641e607fdad058d9a3f194574ea76f4bec46bef8adc5d62c7390da1c45f6fc5d9a784f696f24ae7e6b27a809029418dd18a420455c2cc9695e7c0fe00219a1711468e2866b71f3f9c538789ed2843f44f2a821773c52d211dd1333b5f164ecdf6c3ffd71de6678b0c272f92355d5974eb21c3c8fbd0bca7538bbd9894750b1dd0142bea85104356f9a515ef1ab69daed98d94803ac912c770e26efa2fa0b04e11051ced2f70f06f2f05eac8029d68e12261657cf4dbcc1
" ~k:"248048e6fc52c48398f5cd2ccd8a659c4b7b76dedf54f3fb90c5bb173c5d24f7
" ~r:"67df1d510d063c9067e9759180be470c71fe09c4f133aca181bdb47bb87b2097
" ~s:"7328b887bf0d520abe6f24aff2153f40de009e2706ae043dd3aa55521d9572d6

" ; case ~msg:"6c67116fbd21a0e3ed16b3c4ca58ac49661918bfc6a7c3a6acdbcd53dd4087034fca164df8d38f7ef7db03363701409246382ee053c69c84fafa3c77ad2ce08dc7f41c34a31da496d070a99435799f269dc8effd06d31f85879c299cf7241b37b9a4cfd545086393156737cd9da2d282e7d569fcfa5cbde4bba51bd89fdcc913
" ~x:"7fa66120c5acd5bac132d07083d07968b210cd9c26c2c56d9b16d98066f5df6e
" ~y:"6a50d1125f9f3fc2f7e023c093b3608e6972acefe29c0c6ba07a2f61ed747153ada4a9b680622a842b9a820119675620c11688700b855d4b8d13bf726c36acf923256fef1b53093622d1bcbcf023848b8b8f4abf43bb6e87b84d061deb75236224ceda914b18f7ceb72708789dfb94070413b0e65c1231ad02db42decbe0e558aea06c310aa1a8d113be1f071482fc61913225f007b569b6e867cfb392725776ad71f50dc97b834a71375bac18fabf781126d06df62124064e6a723b48635e6754fc767a5094d0645974041591d0ad4828f63783356696af7ff77cd00107949fbff4709dff8a660a413f5b6c0df37ade84fcbc1d3253ba617265a10cc087606130290909a4f813341efdb611696feb5bea3d7d00a53a81f3a2043b887a776075d250c1a010ec47660087f3ef05782dd21d298d6d37559cd473008f474d8deca6817c1390180276097a81f462c0527928f93a461f4ac2d6ed8c9d6d101a2a9a29201a83d0589f57be28a727484518c7425cf5744df396a0e14a4d260a5c8d29bf
" ~k:"5771223a25f539c80481baebe7b2862156fcf26220d6e953c37f2a22bce77c0e
" ~r:"7d489ab0d44bc73271ef42e28a60e1b7ef7dd27af4045546047085da408bccc7
" ~s:"310151d943f088bb7dfdcd52d82884a7f1ee64d46f9d600d23f52f4cea4d2862

" ; case ~msg:"c8d416c1efe686637078122f798d8804f64a6e85e05f7e8e07634a309a98e92abd54061cccc319f1acd4a087b1d7dbf0b6bf2a09c5dc508ed14dcd5442056eade7691b7fb65b678ec2e137b5fbe875208a427c2a7ad90665426fbcbc7655e48a8965d23fdef11ca8092f511207a607359f94e91b197fcc993ee6ce3c37ad3b71
" ~x:"1f4a3cf1fb60360db3790a03fe55194985977c6884a5fc05a6fb5eafd53587f5
" ~y:"cc9b9d0292915d631aa0d9eb6161f924705c566ee09e74e418d88e6b67b7f57aff5170f6c42a839ba839402bfe517c287781dc97df2e0550b3862484d253152f6cff895f092358b5c4459048581309eff2f689230b4c4951db8413573b6eae85c2dc50fd6134461328e5b6439f41442b91e3a34204428d1e2c22412b012242b14f92e2d1bad626af95051bf06c74da4081b0d619e136a99c8da3a91adb3b8cf8bc5964ff655d45c75ada253aba91c64095394c701c53ddc11f388d61984c32d4326a8c627df845b4100f171bbdb252d3e28494ac173432dd5531e03040302aac8c07c9ea92a9ab67faf0c78b3ad8d454dcd428f942d8ce6e29873049fdbfa1df0e6ec224c9dd066b981a400b1f5194fee13cc5ca7ffbeca98ed0a0221377a1ae612740fce774eeed68382b32b686a25ffc016682186448207c4d9783e83da20a5e8b228a134dc3f44ecc565ab9ae162b855ecd37e6407e714045f4e83b971a5f4e304cd778f3d34137745fc6ea15b4b74d60176ef807410b1b26f68ea14f8f91
" ~k:"589da8a8ac79ad6b62b353422691f35e6474e9c605d877670dd95738b4935f06
" ~r:"7fa51231bc845fa8b668393b78a7b0408113fb77c1e36f3c78c67d65715a8b58
" ~s:"730c9e3483811c52cf295bad042acb5dd6ee90083857bee95b6392b080b5041d" ]

let sha256_n256_cases2 =

  (* [mod = L=3072, N=256, SHA-256] *)

  let domain = params
~p:"c7b86d7044218e367453d210e76433e4e27a983db1c560bb9755a8fb7d819912c56cfe002ab1ff3f72165b943c0b28ed46039a07de507d7a29f738603decd1270380a41f971f2592661a64ba2f351d9a69e51a888a05156b7fe1563c4b77ee93a44949138438a2ab8bdcfc49b4e78d1cde766e54984760057d76cd740c94a4dd25a46aa77b18e9d707d6738497d4eac364f4792d9766a16a0e234807e96b8c64d404bbdb876e39b5799ef53fe6cb9bab62ef19fdcc2bdd905beda13b9ef7ac35f1f557cb0dc458c019e2bc19a9f5dfc1e4eca9e6d466564124304a31f038605a3e342da01be1c2b545610edd2c1397a3c8396588c6329efeb4e165af5b368a39a88e4888e39f40bb3de4eb1416672f999fead37aef1ca9643ff32cdbc0fcebe628d7e46d281a989d43dd21432151af68be3f6d56acfbdb6c97d87fcb5e6291bf8b4ee1275ae0eb4383cc753903c8d29f4adb6a547e405decdff288c5f6c7aa30dcb12f84d392493a70933317c0f5e6552601fae18f17e6e5bb6bf396d32d8ab9
" ~q:"876fa09e1dc62b236ce1c3155ba48b0ccfda29f3ac5a97f7ffa1bd87b68d2a4b
" ~g:"110afebb12c7f862b6de03d47fdbc3326e0d4d31b12a8ca95b2dee2123bcc667d4f72c1e7209767d2721f95fbd9a4d03236d54174fbfaff2c4ff7deae4738b20d9f37bf0a1134c288b420af0b5792e47a92513c0413f346a4edbab2c45bdca13f5341c2b55b8ba54932b9217b5a859e553f14bb8c120fbb9d99909dff5ea68e14b379964fd3f3861e5ba5cc970c4a180eef54428703961021e7bd68cb637927b8cbee6805fa27285bfee4d1ef70e02c1a18a7cd78bef1dd9cdad45dde9cd690755050fc4662937ee1d6f4db12807ccc95bc435f11b71e7086048b1dab5913c6055012de82e43a4e50cf93feff5dcab814abc224c5e0025bd868c3fc592041bba04747c10af513fc36e4d91c63ee5253422cf4063398d77c52fcb011427cbfcfa67b1b2c2d1aa4a3da72645cb1c767036054e2f31f88665a54461c885fb3219d5ad8748a01158f6c7c0df5a8c908ba8c3e536822428886c7b500bbc15b49df746b9de5a78fe3b4f6991d0110c3cbff458039dc36261cf46af4bc2515368f4abb7"
  in
  let case = case_of ~domain ~hash:Digestif.sha256 in

[ case ~msg:"cb06e02234263c22b80e832d6dc5a1bee5ea8af3bc2da752441c04027f176158bfe68372bd67f84d489c0d49b07d4025962976be60437be1a2d01d3be0992afa5abe0980e26a9da4ae72f827b423665195cc4eed6fe85c335b32d9c03c945a86e7fa99373f0a30c6eca938b3afb6dff67adb8bece6f8cfec4b6a12ea281e2323
" ~x:"3470832055dade94e14cd8777171d18e5d06f66aeff4c61471e4eba74ee56164
" ~y:"456a105c713566234838bc070b8a751a0b57767cb75e99114a1a46641e11da1fa9f22914d808ad7148612c1ea55d25301781e9ae0c9ae36a69d87ba039ec7cd864c3ad094873e6e56709fd10d966853d611b1cff15d37fdee424506c184d62c7033358be78c2250943b6f6d043d63b317de56e5ad8d1fd97dd355abe96452f8e435485fb3b907b51900aa3f24418df50b4fcdafbf6137548c39373b8bc4ba3dabb4746ebd17b87fcd6a2f197c107b18ec5b465e6e4cb430d9c0ce78da5988441054a370792b730da9aba41a3169af26176f74e6f7c0c9c9b55b62bbe7ce38d4695d48157e660c2acb63f482f55418150e5fee43ace84c540c3ba7662ae80835c1a2d51890ea96ba206427c41ef8c38aa07d2a365e7e58380d8f4782e22ac2101af732ee22758337b253637838e16f50f56d313d07981880d685557f7d79a6db823c61f1bb3dbc5d50421a4843a6f29690e78aa0f0cff304231818b81fc4a243fc00f09a54c466d6a8c73d32a55e1abd5ec8b4e1afa32a79b01df85a81f3f5cfe
" ~k:"3d7c068a3978b2d8fe9034bcad65ad7c300c4440e4085de280e577eea72c1207
" ~r:"53bae6c6f336e2eb311c1e92d95fc449a929444ef81ec4279660b200d59433de
" ~s:"49f3a74e953e77a7941af3aefeef4ed499be209976a0edb3fa5e7cb961b0c112

" ; case ~msg:"0661c1bf79eed78ad4879e240a46b95a0db2b29bf81263b9b1676daa2554aad7222c9eb7a893048e46fbd2826ab6e8cf42ab0cd631c4c4a1a819560f73cc861a5b6465cf2880a730635ed7f49e28f7b565768f029db2a443ba0a1bd10773f26f752c83da40fcd33f32f78d24ac9820d0bf70dae568a12538affa867160c81e39
" ~x:"807675fbaaf0b6d6ba3d82063cc07327cca3f3522d396fe5d2c6599045d668c5
" ~y:"54b6818054cc000c3af61b62ef4189ba35e04845dee0015be6273392c67332e2e04510cd5b2bbf4723cd8196e025511f6623f03607e566484c330751d03c713068a77e08bde907fc57b3c021e37303373d9d811e38f14b547d2bd87d981269c677dac6ade6acbbae3014ebd381b4008637031c9b6d49ca908765472b05962f55aa361f7dd5a4260705ff5ecf7b317db1fe5d33fdbf48e6a33b3c78b14e620d93806b52e86e082fe4f54d5265e8df623b0c9a259f61b7fa2c0455fadf39693ef3977440f302067c3affbc4574224d5a22044e9bfe11d0d6ede2739c7ffe9277c8644d46beecb946f81775c116388fd6c24af02ec59f621233efe8792d6d0cd2c84333b11f07657333da4e274b8cd3914d977706e786f325e18a339b805c51b45eacb3ce241845970acb9fd1a482a564b2aeecdaeb0a0db39f33ad2991f25cf622bf22f0c4430cf94df1db59aa2d7c2004b5177b9ea69ff556dd4c07edec6259ee139b421573a11cf85d11e245e251190ba869c9cb4daf9f49451a85f38b9b903e
" ~k:"6215e72ef2d6f6e040b7b6ef4cf566a21fcc4f37783a68db445c1ddf3042a150
" ~r:"519fe4c5f9b7707ae4b36217ea1707a1871d8fce98eee9e643c45cd3eb50c5d3
" ~s:"1df224af0b51519e11d8422999b1d3ab0972064180ffc3f1114c9f876a1de3b1

" ; case ~msg:"1597353f24aaf515fd7c0b0a7453444d5f329d6c3f099113bb3a13309b053e6c123a56227a81e8b1a0c8ab4b46160cc5380df591b19d8a386d29a8e43ccab5d8c0e547fba21bffcf5ef42efb9fb2e9be6297c03d57da0b5889b3b9742ddc2c54b8373fed1f2195f5bb2329a8f1f3f8afcec25eb152e7fa819e5d36cfd3625239
" ~x:"169b11d03cba7e817da27d889cdb147ae9cb0459359bbf85a367c64c2ab556ad
" ~y:"5062aa1fdc67294cd57623cedd2808303ceb43537e3abfa1bdbc492b1aeecee61b1fd96cc055d1459ab52ddc3f2344389e5f21448a90cb36e448e60787b1ff5ab6e5549a3921496e8354646bc1fd6cd5f2359ae299c0a047fac3920512a1f411c438bafd03e95e538c6e21d1dd1f15a89d38d48f26305c2534fa8e31d054dcb00774138fb8fc61c6a8d4ae1ca46430d0e31b4b92dfb15bd6b8739fd537101e77334e6f3ce5469e82a8dbc58b3be5ca370359f4a6132fe03360b8f6be248c34220a8003772648664059f1f6a322e0c122f427efdb7d640eb5bb7f3db2d967a2159092d8f8df333ff5ba135602b9ee7e9db6ae0b95886df38d4b4a26a4b2d790c24fa214cd68d0a7ede63e7dfacaeae14d9785be693ad78d88242dfad988b7122adf5afa9efdd0c2047470c607d47b30089ff8bfc4cf5d7a8ba69a7d0ab6c54c05280d66aa4019f6362ea24a1d3f8fcd80c3eb20831b6e0db010faf826488f015f63f0b9ac6df72883efd286f0532b5beff1b9e810ff6a2b2d328af675eafc2f56
" ~k:"467e04dcd564f36cfe47c8fb9fa09cb142a99417a61797e047fcfd51e16e3e00
" ~r:"0502a6e1d8c8dfdf56eb67f9a6f6605735e4d1b0076c8b08b61daf8e7c2bf2d6
" ~s:"7c67abdcf4e580812b13d0a4edbae8a2786d6612bc866e3c13bc09f3e96616e0

" ; case ~msg:"715f296930312368a2a98d3f42810da57115f00ffc4a12029c276b10629e6bddd60bca2c535b79a5f4a006817791f7f3ad2e01a00216672ee5adec579deb07e9d2b0db222c4e01e1f819c1a52d101b1ef678cfca85655dd6b2426f1ac379a92a9c69b0f8987432d109cd9a7bc04ef287c2afb663444688601ce3c55fd90d0fa3
" ~x:"4ee80e4cf46b4e072e976893a2d1e34c03d20f3aa1785a74564d6b4654b11a54
" ~y:"b63340d6a1955731283064f6f22ad7f0e28199f6a58c57ddcb44a026c61e441318c4f8755dfd71b295e9e7babe000849c972f68d4be00954a3c29cd4b4e83ad51830080e29e7619e45d3abbf9d82fd87e97581fe909d3da1e3e96cb3f0c893af9d07f418df902e76b0bbc1c97139cbd51226ac442b3d0b0525c784ba138131421c60543e6e296069f611b9c37cf6030636eecaf41c3b4838f506c02cc84cde6b99cabd2ca578449cc1718aa418ca12a2b76f78259c1691e0b49f09dbdaf585f626cf74d73212b3427845c66f2283b60703adf1a262bb8b10ac7ac5d1ec73614fdd37ee51b71cd1fb4e6db8938382643c721fbc4cfc987bc5efbc81299b375a560cde5adae62831ca4138c399d82f1f8bc680f9c6b47eb464a1e0aac448fe3b5c25bd8c0b7afb701b0680db87ab51738f19f5b965375dd48daca07bff3885632175700c678619f194e4ee5f55aa448aeca7f7b3322f64a547315c5cee045122549fb38b8acc95da5e833022b3b894f03ccb7f73b91c1fe82ce414e1219411780e
" ~k:"3d7728ce25def9a31dfbe442fef8b162b30544065d9cb7bcc4914662a282bc10
" ~r:"4c5e990a6e24feddab48d0af4a08b45ae8092594bfb3c012fa1c325c977a3cc0
" ~s:"820b6cafa89b41c4ccbec842d7c408c65d4998ab1ac6b6bce8d4d569cdf04726

" ; case ~msg:"1fe5ad49e11c207d3d5e1923060832afbfc0aa0cb29fc0b22b3be59a598f8c703b9bf2c7347f8abde25677ea9cc60af9307d21d301fdd23c28277fce11400310033962c04ecd377fd446358a3449efd6bc05721b784ddf0e238f28608e86bd4c3d7ac631fff8be0678d37bfbac16b75bc15a50ce1397dd4ba3bffcf94d341274
" ~x:"0d690f2c87fe2cebc9f15546f05afaf6dc843b80abd2046f33de30c2e806358f
" ~y:"53fcd07399e4d31b09abeff2f096a7b2cc5cf417dee1207d8a5aabf9e8f9fb0f66be48826a3dc11e39beba2ff47b76544bcf55485acf1e3d49e19057015e49ed012a4877be741607749b6f4bf95c44ec3c9e8b893aae8d80e369978a3580371cc13de8e714092bb892e4a956ad3654032f7758fb9454a1cb56406e1bf45855108ee960107a65d45453cb482dc19049b6c83bac111756caf65bdbe5e6b270d5875b997a1722ee9d58384941aa40e810b60b83412eafd0a7428a0abb55df45680cf22656711db6bfce8bdcbb4c083a401cdb68284e0c7ec00f7de74e57146adae221e54cc4a566b05a113ddb22cbc19d881a41cd75de8cf6c7b89a5fae650df585aa70c045b84b2cbbccd0e7ab720c5896abfd356a66f3dcbbb5386be6d02ea9b3191ca275d22248aedc360ecd4057ae06ab2c2aafb50657a91c62e038eac9f5c4d88106db4c6926fb5dd2de1ec7e4e005ce184570e7e97d76422fa037621a6f6d46cb83ab6f4d434b6a8f073900cb03a7810455d19e77d4df624d08e782090ffa
" ~k:"796cef38518aed8644ec5a1b3389da5ee9b063b88e7fb4602af0709999f9a938
" ~r:"41a2c955f41413a7ab067b4f50c61e396f9febff61c1500b1a4bc69e50a51935
" ~s:"79edd751a9dc2372b40580fa4d538fbe2cda4149f6b11939ddad92c574740883

" ; case ~msg:"a326973093ce502c16473d89ba196507d92281504759cb34c6cc353d45197f915b5e736b8ff857a8b2ec99649a3224f857401898c9ea607e6a2c1d320f27564ccff5dbdacfd87a145f1a029425d76502c081ac0f6a14de5b2cad1c23a61d4e9ec6a04e1a455fd710c3c78c096753c0b7f1511e8ba5f5f1af4f0741fee88b77eb
" ~x:"37d0fa99e5eed0fb51c6e690f0ac556ae74cab9a84d887a07363599b198475dd
" ~y:"92915db21c2c3e57fcccb7dfdce28a12aaf6dd10581193b98b7d51a728c38516e39ef5cfb1ff9fa1659c9bee56d4ebc1cd69646c3cc3f7caae0c42d9cca9219148e4998c2ddc89eb9a3edcfa6f457129007a9344013dd123aff197bfcd3db1d9e2199bcea16165a4c34ed2ac32167abd167704ead31d5fc2860b834d44f86cb530dad9e887013ca4d6e883008c286d206b6c7cb252d1328b503ae0679b502ec1646f69f2602d5e3d631d4a5a63fc7a5d06f27926a4d6b1ef2f77ddff3d850d3d9f58a958c3f4f12cf029f148386c5b8a71bae9094dec85279b1e387799d26b2a6a0e0dbf06497366e4903e559e70975dedc7d4934d4e2d3d2cd305ab826402ea8f2778e26625119e7b0c24c45dd9c05a3890dd1d9d930bd0bb409366b07a47ce572ed5bcd5f63c467d49c56811fc3e401341b9a4531f776debdea540a34c7cca3c3fb2ea99c5fa9f9fdfde918a94f74e080d1986b68fc1e3fb978054872ced97bafd96731e6d4f1c4a91278c383d4761c974100974522f7b6e8a2884d5b3bbf6
" ~k:"0b5c9b613708ea26bea151a0dd4222bb573d950588483483cd2b8ab537469e53
" ~r:"73f1922e26d9b8068b68f83c2bd5dbbb5960403b49223c02a42ce6cf3810db66
" ~s:"3ad30be9a60f6d4227039456c9827d5424858a02a8e6d3891772cf80a5e4ee21

" ; case ~msg:"7504382fb7fba1dab3c93bd31b16e73d9ae1d027dd23166b3b94c7124183faf3963c420be5205a1f44a9a9026c6ef77e7c4ef1ec4845fef6e5ea2487ce012ff53f9450fceb0d3ac62f2102d717e3287db3714717a28cd8b7fc64556a86173e6e7f479f8a8dcd895429cd7f0f5304ef6aaf275d94a7f4b30acc1071787ca5f062
" ~x:"1c21aa2ef3b11d31f3c94a278859cb74bc40daf5993dbd774b32ea3ca24bf162
" ~y:"2055bbe89da0a0c488c3dbf29531f1f7cd3fb55a26efc540c2eddcccea1615dd923fea4c8d0c95a5af7e1e7816048f2ae85323a96411e7d1ad62c4ca675b63df9dba31c1c76803fb1c8292465ad0a7e49ba3756a8ad4c6ce86fd30b8b28e08c4b4777e079faff10ff852f7d891a984198dd04977972108c52ce8bdb115646224a79337746e3647213198f1127430f5608733d88204a62be6eaee84629fc7282acef4c4f5d3adbe72410b1edfb74be16b2d675cca891bd8cef205178902b99271b48041abe33ac119ad6b756a477a63063aae8a17ccfbe2acae3c0a3c630c13ade197cf3d05a9fa9d6899c0a3f9487e6148732dc63e907ef79488df3373b8a213705d69dcce6ed9a2209f59ebc58bbbeb08054510b5a65169d0fc1d4d10bda68aa7eceae2e72f0339a2eaaea08303064dd6588414ee7705df3ab974debef588f4e31fd6a8f25979c9f521d2343120e40794f41a4601be579183b877e6a8f6c0ab7ce8480e7fbff467a581df570af89929bc4b56397b787df4d729e65f9b98ee7e
" ~k:"44efaf7a15a1eb2a7ba04fd4717e938fe738666040b3d81560497ce166f31e86
" ~r:"56e21a7ab61f9eabbff47c75e5f68c31873a9e1f2e1db662731182f9a029b8f6
" ~s:"2f24c52f7baae29c0b4633a3855233180eba80611dbc7e88e23548a520b60f66

" ; case ~msg:"0c0f7b0f9955bb54f16e4e39ad9bfd1deb04b8e8b38e674da455696bdf7cf28e24114ad00513d8dd4e5c895d351ea913fee516b646820087721d9a0b5ecd769b3825739123544e7058b66d2342b04462d5d173cdb00ef6aca604aaa438b8868d15dd6624abb8d19384db48bdaa66471413a894d3610bc97d848a59e2c69c0c0a
" ~x:"34651f5844cbf85960e987190eda4ca1fcf32d8cbe1ad08dd5aa36fbd0d42000
" ~y:"45ef384ed817386668e1b90b42f1d423ad9b17ea870119c0932ac2f515f546a3b6b80a612ee66dfc00cc4d9e3b5dd15303d5ebc0aa40cbcd7746f54a3ffea23aea0704ae9cf5ad6145629c61d158db6ee39ac899bbda59794b1769a92982082b77a1d48856427b78bb6e077e27335f115bb842e53251f699f04488beaf83a6c4aa6a4b76370cefc9099c0a45bcf973242df2a01ef68e66c87effd7f98f441e94a09a2830076c2895f997afd0a909b45b3c059177000236c501bfaa56da800ecf08701d212016b525f30d63ccf3afea09df39e1cfab7bf45de1a39ac7f28de0037ec552e2ea10c6b56a5db8c13fcbf73d2e50d58b4f3cf278506f1eaf0873e9ee9465cbaff4ae626f3aa109fce49e55d57fe881c50f7279262621282adcf379141c9b2c39813faf823a7ec077c6e6bf953f130aca58f36e7a87ab1aaeea5eeb4402fa9e26ef8938c8f38a6c040809f4d04c81e2948387d7be813a973a9c95176700117de2f33e61940387f851a73dfa4ac5c984ec97918c967bfedd886d1bb705
" ~k:"52c16c3e7b17f3e73d7965f584bfd7ca036423b0d42cc00e58d1ccbc419d33b2
" ~r:"77563b3b48fc9ee0dbea79fc74dd6c69b72c4270918e6a1be2c998177023b40f
" ~s:"099cdd62dc044a57ea25d1b5c1f6ed84d11bacbb0975976d5821c414b5416bde

" ; case ~msg:"c67735698ae7bbaeb6f321a1088617382a5c92092151ec364582962c9c0ed9ed8fc790cde0d9744d4e38970a8482401c0f61e91805f4984b8cfdf9dc8093a5c6681dac13809bc41d167d3e11bc99698a4bc07fd248a67491e8641081ff1e97871745157cf930195a35a14d0883a26db442e4edb962aa6187b8d1c7791d61bd25
" ~x:"2ad20d2e78a9ec234f99a4b2ff52faf492c3e3242ae6c04ea8a37d5f10fce6db
" ~y:"0d3b3c3df072b5f5129118132bb7bca3c52f51df36767f1152387ec00df65c728f0cffc1cb6f224258cb6d3e90f79dd976b5a180b83903d210f0c4dab82eb72a1f8997bf09301d0f7c89075d552c81fd9585b0b1b1291744d21bd1edcb511217c2962e1a6de9bb01c2b9698ff55ea75dcfe456be481cb6f064fed4bff874eb1c9b7451979f7de7011baf5a47c976a179aee909d25ca87fd5e3c75df778e21272937c5ba7806aefa70647221e5f7cc32ab8015921a5a95ecbb3ca4b667249d0f34dd2d8ba86dc158f9e8425176e988048efd9f7b7cc53e9fcdb29ad2412ab4ca6ebbde6f4efca5945b53b2753bfc4eabe6280235620c4464f6940acca1a94659a527aa14cc7c5467382a54fe479656dfbc11923094fe8019a08c3ce7e99a28f086bdaaf0faac6ee16190dca8e94bf87657058495ad07931c89008ca1e565076256a93cb2468aa7122758b8e174f6a80f41a90fc92f05bf1f1f47da185b2f25a1abf5e0ccc6613e3aef87193400d751b4c87b44d9bdf5c0e207f0f6a7dc2113799
" ~k:"654dff8f0500b52adbb70fb7bb7aec4b4820963706964c19c8320e161c3ba365
" ~r:"42c902c5826874774550464c4bb736f2af7fd2a347f27c65bae11820eeb752aa
" ~s:"6411b45947a43c5b01c2f6cefcd41cab73fcb6ea0f2a35a21475563055316e3e

" ; case ~msg:"eb6a0359c6e46e09a42c554705bcfc5c0c022670b2f6c1a5bfe14ea805759ca2256153fdf815057ca9bd5f4cf837e14fdba3ad17612ccd19fde00764ba2e8ecd8f5a185cb26512f7457259c2f0670852741e7393b40c8bab673be2fa519b48a95dee6552365fdb7ddb632b1b33f1a5290b828da5965e82d874f79cdb928814fb
" ~x:"0d06d405d228c0ed860b9e21bae570a6fd940cc702dd6e9a0846e7b2a4be47a5
" ~y:"6db83b06c698ed80122ec4a2183370ed7dbd6ea44dbb4542149568570c53521d3399ab44fe2babd49068e11953c5d38f7ffe3bcbe4cbceb91c155ac8741dcf226a59ede10b050b9f3743f29689266ce6ee020ca17f9fa0e75b3f7158a65cef9fac76c88786b5e377afeacb9b3dda55be922da0ef958aa556abfb43067a414e915e31af5f5370881ed97b25b4bfecbe082a145d02717af800e77e28963cc0a6a1c11b02835e14bdba1a8c9ce4bfeb06aaebd760d7c43cf56ba212d0c75da026176535f982e8d749f20c2a8d5f53875d893374d859b7cee58b0eb319d3313cb8d17602f47e120d1a24a0f8a63cfe45a5028cc0937bbe89f6b3b7cdcaa7dcd5ec5f3ed2aa9f3aa8e91a496a8bad7874dd34bd8f2a9591997d54f92d5864216c953646840b378c7a05215ecd97b6ba944ca18597b7a54832ec98c1cac0003d50d5a05312cbc852d507cc973ecb56f424e8a1c198bcdbafaa6f928fd27a7c91f84bc234f2532639a8aa2196f8fc2b7111b3d0b1153165a0e0525d4ea595f89aec33b6
" ~k:"521906f186797e7f5ce85112ab2457ddc030d6f34be361929f4d373dda576e08
" ~r:"0493db0c18a3882709b3cc9f8dbe05454506c04c3a12a41d599d201d7615b6d8
" ~s:"7494b4d1b2f3ae22797255a1d0662746352a3d0532290402068594cfe48c23a3

" ; case ~msg:"5c59b209bbc0a1e010cb108db4101b8e2d04ced91299a8742322102e0d578c3698422b43d19d331608188bed4c7edc03a442f89aae60f4e7ee9b6325de3a8bb702918c21343bc9b266f2ebcf5a620336a7bc99ae3685f19080db46f24a501228c5bbfd9c0b4b0abecbfbd676c359607ce292cffd52d26af80b22e3c4d516ba0f
" ~x:"4214d55b0a058c3dee2751407d9296168fed9f255e5c68273e1e5aeb3e504e67
" ~y:"be31fd5d62db690bcdbc09e453d4417f82e8621ad717cab94648201a74f6ffdfab965311e8ff35c4a0b5dda339b435f17317175ac642f785129e151694ea8b244627e300ceb0f3be08f91c0f527f2e0df7c9f55492d1329b7d9689634c8a4f5210157e2419e615d9431736f804b1641103371e7ffe7200e74296127d59a8f97d41af11d70c3fd02531f7b811daa7516aa2f2a9ba70dcb704f3fee47f2cbed65c1e3d06c8814e1b28abe29f3d056792efdf9ac9307ed0106c5a328721af0e202b6df737ec4d82143dd2505e103ad845863c45869e69abd9e02c7b6eaaff9e2e12bc188138688c0be3e6941c37c7ddc9b6d289f7cc8fde42bc3c14e3ee521635f32f54280d119ccedfc51090a0ad006b2427604014ea4d0e0cd1efbce09c7f8e9981f969aed6d481cafb329f995343541d36686de6cb8e4b1e7e3727abd5c1e3ffa6936ad44b926063561512c0e9ac787f8eb791f963f790ba1b21dfe1b8d31d4c16b152a6de65bf54ab0f0d1e3d450317b1cf0c4e331d18587accb6960ccd04dd
" ~k:"574cca3bd87ec1994449da2f2324a3945fa3047791274367bac0f12d4c064343
" ~r:"7fc9bab3505adcd1b1c8127e2d1fbcd0e15eaac314250dc1c684fcc47fda2993
" ~s:"70f2007edd68fb9dfe19a63eee4d5a977291abd235ed26e4291476ca5d0c8171

" ; case ~msg:"c805d18c0bb53d32b57cb652f5b0e5293be492a1c88dfbec5baf47ee093e2df06918994e5cacbc3dfff229abd31fab7a95ade2fb53adaa7dff51f6c8581c69eb5b090baec38607ee9435447ad8137455b6ba179fc53ac094f97e3e29d0724cd10811f142d67d1cfcd5c3d1e9b411dac38f6e1c0c14dc9a50d84bcf00ece8a603
" ~x:"3cc7e58577382500cb461c0ab8ff01ece8fa766b66f8be746e347ed2ebc18ebb
" ~y:"2b6e1a8d4482b41697bbbe50b55b3dcdecea8d2e2eb5cf27b892bcbcabfb253c19486fa77c98c15add4149925b5501e5a5ef45b32ad09a872462a0f41d048af4e530660a3864937ba6a9eb0734e90fda3c9b6fcd30c9078771295a93802d9e1992a4eee9af7a0413880f33bc0b62036203286844bc384187ec51a33d390eaac0cc3328098a847509129bda735909fc7a11893ad0ec61276b7a5dcd4e626d9ba67610eaf0af876afc0419fa4f009aa5f913a1c73798c2707eeb8fa77f4ee058229a0ad37e845739668d95de226760898c02d06f155f82dc16360c3abca3780bcdb79446c8343583dc0f6925434b0dae7b59cb26b10008f86570ca0350de340b275524f00551310f1d095db8480b4acc489cf5e2947eb92904ebfd0d978bbfb5d0c6a1a9db50cc6917949c71854632b4408bade5195d40dcaf61fe950eff0c8997c374f1d465c80bc65adda636433e94f22c5fbcf09e99666a535919ee6f88154934f11377a9a9e021f2d7ecaba32510e92bf5ad67fa8b3d70dd2092b1389e3193
" ~k:"179c02ec8f18fd88146120fcc51628f23e250ad694aa47bd691c0f442a63a92d
" ~r:"38208c0985624bb9d62713bc7150942cbc92b8e8a36ef6d1ec4d08d1d9a5715f
" ~s:"65d2ba787ed4c08beabf24343d06ed61872d6d684a3bc70307fcb7e20df931da

" ; case ~msg:"9e0c66a4f120e85aea064e7a8ba132cf30a45de2889f3547384e4e84f45b3572bb0423b834de9f2c9636faffdb6331924f0d2f5b6876145d9cae110ab0cf6fc90c2eeff98c61fa186cc3952b57299a73678f4585bb18fbb84ef4166779ff10eed14d47ae528e03298dbb97cf4f88b7e6d0959b5894550a3e2e356947d25ffe73
" ~x:"5c5791dd648703f29099736146f5b1b5e35dc71a74d6eed312d37aeb6d389ef0
" ~y:"a62adbdaa5a55a2d1e439b5489cd6c8fcb23e9c64fbfae7c83e9d5599319bf3f06c3c290b989a638940b1d0b7e8bf6741319ab4c38d46e77ebd4945e25cb89cbb64e44b9474bc7c9d9f61a36e57eb6afab6c7a149afe02c1cd685483208c55feecb0d0bd96697b437991059267d76a488465faab4a7e17592329567005faa421e011d67f4da75accb627537e933e9ef0be3c70f21ed3f8c3b3d7d769bb611f82f2baa10fbc7313ad0819048d353d679736c4d14bca9985ecd37041afffb291a7d909c7458181d01592e6c90c0e34b49461ede66c5ac002671a4985546a6075df95b523f166d2e0d1f5da77baff5a24df775cc9d367f2a0728c4802d797041788c56cb871290332c1361f8da8897b5b8e25d4a93594ac648bc53c9d85b4fcdd7ab0f5a3ee9c25cc14ba6543b078859524ec7f0b61cdb209cc51c40aa9af082ea9c1d4b91b2c1f6dc11cd879fb3865d879fe000f0e0b4b233dbd01c9c98d01a664746577a64bf28d88256b76de2babf14961113733b1bb555325c09d8ec9189fca
" ~k:"8327daa2fbd001858dea53d2dc0cb005e0ae5fb15bebc0c5efd33371637ef318
" ~r:"4e35f586fad4f512863c485ec61ed01629aa1399b16fef4d80cb332752b1da92
" ~s:"262dfe6ac72a2f6044f62698e42dd2f92b1f9a91be42b5fdd293b1bf9a145f00

" ; case ~msg:"ed88d7076c5f6a5e0f947543d5fe746afca9b2c4d06655da4607685c799c210be4aaee0e6ed19713814182c7f7d584ddbed488c8e3239ddd810555ad6316d1db37fd9553ad74e3ceef9eeefaf54563602f5547aad4161e9384edab655a898416db53f71237ac5a1485711182bc5bfff72460252784ab1bba23634a36be77533f
" ~x:"22b44bd6d23ee65ebc2e88030f837ef65593eeef0966239a92d5126cde867a13
" ~y:"3e1ce8780f39444c2130dbf9d80ca4b25817dc16d08e2cdaca0b56cd2abdb9ef5adb741ccc1abecf62806ad7e87636f52831c6dea48e0729b904e5a0615d7ab4450104208a5ddfdb2f2569146ee83ac9aa27b4d066355fc53dc1a3683211ad3efad1ae69b8a7737bbd89f5ff48482e2c56edaa776e43b2a0ba62e513862da290288f07f84ca5a06837d19e9b186dc8d36952966e08f7213340186d31fd41a2d1455a083aee62127a28dfe4da6c876a5a6f36c45245dee6f6566b8318d3d01943b2adf8ce94ea01a01ba41a6e286820a96707cbd4002875b79d9fe2db6cc3f808ef0f71380ea9a73fc7e36850d022ffac131636367886a6e9965759d73f03ace69704b52144f67b678e2fa201c19bb37b00377daabc9377adcbddea2816cbb50b26ad2e429ea0576e7721b3b75c4fedb31fdf1f0c6c2eaa135f52c9a97f0df5fb25ef28848bdd7390cd054003722582d94e90a3bbe85beb34701271b4bb48bdf9b3d0e1bb5623445c7828c937a423be512c1177c9c0b5b0b6b0e1f639d330e051
" ~k:"0c37eecd48682f897accf43b3e4a538cccdfd784625a6cc046dc54b093d16162
" ~r:"2e7cb404a6daaa8e00760dafc95b4eb5545683224a61a1bcd6128bc4e7ac535e
" ~s:"3a70b3a97e06e63b89d56ed5232346461c1a3b6b145d89043a48d666de0256d5

" ; case ~msg:"9e440052ed927321948388776d3719be068739dc2d6c64c5937176b2005c2d70a9389e6a655663366c0970a8e2e3117ecef257e951ac81c0731dfcd4fbdb1241bc249adde9cb398c7d15e381368ad3d24edee23397c15a5a356e787d8f2fe9be76260bd363e17006281c199fe5b710f9dfcac52895e392f7384d71bb83053ffc
" ~x:"680883caf23665e813572c1e4230218edf53b3a5167f56a7d80e53e7d3ad1df9
" ~y:"89e859fc63a263bcc051bc2ef58cc919ee537385cb3636d83a624a4230d4b0024ec5e28bcb884667cd2bf8c28451b64de097f2194cbb8c6e1cecbd6f9fbd576481555d0f0e8f13752f2472f7619d052318424310f69d50de78ad6c457b98c611f8481d4543031a73f83d1e852c1f2038a6435e571f776bbb5cf978a9b2c88f05d134fd5ff4656a69d6fe6b667da6da54be48386250394c75b495689fd4628f666424eb080094448d41b706292e51e75386543e5fcce6a6f3aac03a7d6d5c2551ca6b5b85fadc86bff14c79a1602fb0c1d43d88d5679021e826062ecf186aaaaefc312eab9f9e2da120a8d7d08ba09aa9abf4e34f6d88c4c314c59c36ba57f928d88d5d70fe48ac6700f5cf607a55e3646dd03d47e96ad869f7ba2bcc7d65a99c3221d4909d1f22e4ccba815fa5b720570e42f8626c31d99f60cd6a015391fab3537446f747c0111293c5bd6b5dab2bc3d5137d2124029eed12db71bdf794de1a2ec5070d83f87195264ff09cb48cddb5e852b233570f1b70cd457cf864e2ef3b
" ~k:"69e6cb5bcf8cae88c96e464a9b26c6e1bbac1e229909e27542278a50c66959f1
" ~r:"37c34f9cce916df3deff26be08a4e6bbae0661fbbb5d81d6039f00b1e5632b67
" ~s:"3f4a2932917e6bb088599a269d7b590769acf9807dc5a9420a95e12c7364c5fa" ]

let sha384_n256_cases2 =
  (* [mod = L=3072, N=256, SHA-384] *)
  let domain = params
~p:"a410d23ed9ad9964d3e401cb9317a25213f75712acbc5c12191abf3f1c0e723e2333b49eb1f95b0f9748d952f04a5ae358859d384403ce364aa3f58dd9769909b45048548c55872a6afbb3b15c54882f96c20df1b2df164f0bac849ca17ad2df63abd75c881922e79a5009f00b7d631622e90e7fa4e980618575e1d6bd1a72d5b6a50f4f6a68b793937c4af95fc11541759a1736577d9448b87792dff07232415512e933755e12250d466e9cc8df150727d747e51fea7964158326b1365d580cb190f4518291598221fdf36c6305c8b8a8ed05663dd7b006e945f592abbecae460f77c71b6ec649d3fd5394202ed7bbbd040f7b8fd57cb06a99be254fa25d71a3760734046c2a0db383e02397913ae67ce65870d9f6c6f67a9d00497be1d763b21937cf9cbf9a24ef97bbcaa07916f8894e5b7fb03258821ac46140965b23c5409ca49026efb2bf95bce025c4183a5f659bf6aaeef56d7933bb29697d7d541348c871fa01f869678b2e34506f6dc0a4c132b689a0ed27dc3c8d53702aa584877
" ~q:"abc67417725cf28fc7640d5de43825f416ebfa80e191c42ee886303338f56045
" ~g:"867d5fb72f5936d1a14ed3b60499662f3124686ef108c5b3da6663a0e86197ec2cc4c9460193a74ff16028ac9441b0c7d27c2272d483ac7cd794d598416c4ff9099a61679d417d478ce5dd974bf349a14575afe74a88b12dd5f6d1cbd3f91ddd597ed68e79eba402613130c224b94ac28714a1f1c552475a5d29cfcdd8e08a6b1d65661e28ef313514d1408f5abd3e06ebe3a7d814d1ede316bf495273ca1d574f42b482eea30db53466f454b51a175a0b89b3c05dda006e719a2e6371669080d768cc038cdfb8098e9aad9b8d83d4b759f43ac9d22b353ed88a33723550150de0361b7a376f37b45d437f71cb711f2847de671ad1059516a1d45755224a15d37b4aeada3f58c69a136daef0636fe38e3752064afe598433e80089fda24b144a462734bef8f77638845b00e59ce7fa4f1daf487a2cada11eaba72bb23e1df6b66a183edd226c440272dd9b06bec0e57f1a0822d2e00212064b6dba64562085f5a75929afa5fe509e0b78e630aaf12f91e4980c9b0d6f7e059a2ea3e23479d930"
  in
  let case = case_of ~domain ~hash:Digestif.sha384 in

[ case ~msg:"ed9a64d3109ef8a9292956b946873ca4bd887ce624b81be81b82c69c67aaddf5655f70fe4768114db2834c71787f858e5165da1a7fa961d855ad7e5bc4b7be31b97dbe770798ef7966152b14b86ae35625a28aee5663b9ef3067cbdfbabd87197e5c842d3092eb88dca57c6c8ad4c00a19ddf2e1967b59bd06ccaef933bc28e7
" ~x:"6d4c934391b7f6fb6e19e3141f8c0018ef5726118a11064358c7d35b37737377
" ~y:"1f0a5c75e7985d6e70e4fbfda51a10b925f6accb600d7c6510db90ec367b93bb069bd286e8f979b22ef0702f717a8755c18309c87dae3fe82cc3dc8f4b7aa3d5f3876f4d4b3eb68bfe910c43076d6cd0d39fc88dde78f09480db55234e6c8ca59fe2700efec04feee6b4e8ee2413721858be7190dbe905f456edcab55b2dc2916dc1e8731988d9ef8b619abcf8955aa960ef02b3f02a8dc649369222af50f1338ed28d667f3f10cae2a3c28a3c1d08df639c81ada13c8fd198c6dae3d62a3fe9f04c985c65f610c06cb8faea68edb80de6cf07a8e89c00218185a952b23572e34df07ce5b4261e5de427eb503ee1baf5992db6d438b47434c40c22657bc163e7953fa33eff39dc2734607039aadd6ac27e4367131041f845ffa1a13f556bfba2307a5c78f2ccf11298c762e08871968e48dc3d1569d09965cd09da43cf0309a16af1e20fee7da3dc21b364c4615cd5123fa5f9b23cfc4ffd9cfdcea670623840b062d4648d2eba786ad3f7ae337a4284324ace236f9f7174fbf442b99043002f
" ~k:"40b5cc685c3d1f59072228af9551683b5b8c8ff65240114ad2dacfccf3928057
" ~r:"7695698a14755db4206e850b4f5f19c540b07d07e08aac591e20081646e6eedc
" ~s:"3dae01154ecff7b19007a953f185f0663ef7f2537f0b15e04fb343c961f36de2

" ; case ~msg:"4bfd28a0a79c94dbd667c275ef77a235d8ead7c698d42fb7f7c1fd3c8c2dc48d0dda2408dea56325d69283692a523d281ffea856ffd9f8417eafbea606d862dc5897bdf241f3e8e49aded5eadc7295e5afbf96b3975d0e25daa2433612e120f659036b807c1853c03c90fade2c19dcd923492ecc906cafc57a95da6f20dd59d6
" ~x:"95fded7e5e949602c1123d80f89503cc5fb7454be3173af495a18709c1c2506e
" ~y:"6c778bcb146582277633931bfd029e69c9e8c0ae9e24913fa734554f24f64aa64fd9bc608ef677a1d4829aa8a8564c2ff0ffa2fa6a0c1a2ccb606dda018bf095f8c897d7a43349beb9807b7b118f8de8856b164b8d8babdc17b48f3a2b972ce537ab4e7a7d9ba5d7e6fa3698aca91973cd1787ef7b6b4d0410de59cd3143e0f3acfdaabe56b371b4354d0d32dbd1b5ca6a872054f3e6566319d5d50b2cf54c123ffc929007ad1857ba13b7c403f551c2fa4109c44e19ef97afb62a6103356fcc2ef451e736261010b0ef58ae07a0c801ff75ebaf6cdd763f8df2f83f0ebbda40845b2f42d3feeac071fc626ee5b51f9bc1a130514f2204971b4b7261b4bd783ff75775aa73a63d7ebe990b939b0f44a909ec390036f297c3563f64d142c14ea43c5d3c6def4a3a9ccf6274182b939b886501aeb4efb23d0073434cec6a915a67e24cbb2354c9bb1089af487eab5d8e499a632e6c61492ea15d2c444c269de33271a90042468de2767f0dcf7a66424a3a40a63eebd19cb89c8d74c58504c4e103
" ~k:"6bd1eede564ecb1b3fbbf2d96e334ab4cc002e6624e2cb8448d8608fe0e8c43b
" ~r:"37c3f7556d6e5acf7989f0baa770c2450deebd4d5f58b61e17b4b2b926b58031
" ~s:"a61d86365f10ca5e1ee2c4bf276f2374e88b5a2d1acd8ecc11e97785b4fd9931

" ; case ~msg:"e3fc751b6978fcf40f09606ee4263e1660ff20e9c63a7138f078ae3e3e603dfcad172f3c7cb3f3545fc23bc30c37c8439c7b238341f29148276ea2122ea8ed0feacb149de17cfd33b8c9408aee8ab0ea8ba4a2b2ea237418bc3165369c8cd420242f8d32bcabe0c352e21f65de80d587ba2713cea6e53ca524aec365bdf21adc
" ~x:"3a09006faedec91446995a393b034b0c7ff3fcd05cda2e9e3b2f98e3a4bbb9f5
" ~y:"1349bbf16d375c392a9acd5bdce655f14d616274388a45cd372925c507ac129fe61b998e25127f210926ad1191583eee8c4190026ba0a95894be3f0ad5d05886c59a3c7a0044f7e2bd9bbe28bf9366d034db424f34960e30a8e7888f927d0bf984b0ff99ea271871124aa12e0c0e19624e533cb4149cedb3e11d321600dc07b32e531a615c8f7fd7f33a071caaa76433d1aab0b710fa7ba3ddb0175ced4e558d5117afc7542b9b07a8fe8e4b08a1de456443553fe87a4c2455ded72f98544d6c41d6ef66b7142a4aa9aa1d3d20f700010389e417840782fad682153d569f944d3d3ad1d88db5bfba3499e4c3660b76b44da4b0e6727ebc3f22b2a0aaf62dc2a29db8babcacc2169c2b8674054c89fd770db98b12af2d933becbeca9f22444b527aa894b3765292dcffaf3408e699495df79b98d957fdba7e4c8e7ace3f987a95dcb2e777fa2d1304479a6d137efcb0c404e6d8ed39d6afba2549f3ee2b9a45f324567c0227319dc59bcadfcfdf1566f356f7c2ba6db21cca2a8fb2fbeaf31cb7
" ~k:"4212971feb32e25fbb22845ab8c9333cb2a265f003542838a128a25108a88365
" ~r:"2d3f3c605eca8fec37a76d606d20fde89cb6f971a44796095a01dcf8e900f5b2
" ~s:"6a43168334e5b0ea07cfa5978609e86f969d1005528ebb3ee9073d5655d54b44

" ; case ~msg:"45f656a1ef0e61de46df2ca2d8ea26640a994c30380c0cfd66be3998d89849161bbcf3bee77ad30e769f10e23aad5b4df4edc19a86fbb5abdeec8779b76be279532d7692bc586c62692fa1e3dbcce33ffddc9f97589172f64a48535693ded6bc73b2ca32469d0eaf6706d2a5f58f8d28a745dc328bcc75b3415ca93e29eabb1e
" ~x:"9f35b1038686bde07a5f517d68f562739cb7150fa47ebaf7ffd29306afd4688a
" ~y:"31a989601f32b205943a841887df3c6814cfb2258e5204d04d3928ddfaba0dffad43151e27d666d2928bedc67275440fb502ed3eafc3adc11009ee703f01eaa034aa724fcc63c59a8a5963f3352f7293ea2425ea89bbf1e41724b69f383bf10a973146ed02f55208b04833d1bb5399a67f04081590acfcfbb12105423e26091d09078c45007d436eb19f952f8798b001a3c64a3baa5496c9dbe6580781d4020bb7e4e7ae2380ce79658c10a2e57bbb8cac12087728ce43ba2b9f380e3abc2dd12a682488c6b4fb2f8dd7f3846b6a26f913ac156879ee6a1ae0ada9568521a4428ed9f741e0e79a842880019c01b34e988a7cf7e63524e8cd025453223a2660273e491968af7f4b1dc2123961de3753ab16eca5b1859a4f71172538f05a2a82a34f98ba07c1e531d82ef592e5493533416bd6c6a4c7ca3b0d2a2fff88a8f073a76c691802aaaece4e852d6650871a17cca0f5251ef22dfc8e3b261bfcbd5a22b2732aa17d7df1f7b82f6b222e5f6065bf80d04c2e5774094084e4d5ce0d3e8917
" ~k:"55d1ffc73b52b6364d660fa4658a6351142ac538fd3cfb4eec40ba07bef5418b
" ~r:"3ced0ea5f7fd588668a41efe0e90954c0930afb6be18d90752831f683cd92a9c
" ~s:"9e46ca12941745ea1a12c5a2d609884cb5792f46afaacff07237137400366868

" ; case ~msg:"c737d5ae248a96062d6afa8dcacc0384c5fbfb9d8b6052b52493c60d3edfc524b567b1f896e7447d0e24019403ed83e4889c0c4de57c70fada6c8b5a09904350a44dfaf77d60af62de3edfd8760d077473f26df2837cfc2015f227dd7d351a5350f1428f2699fd3f518326fea8aef98fc4ea673130c8079fac3895fe856c77f8
" ~x:"40dbd496fc4644be7ccb24d9dc55895c1b923a05f4da5610589d564ee8aac33f
" ~y:"6112d3cd3191d17dee7788f568815a0aab50006002c9de2bd1a9bba245ba02894b02e9247517ace698ae0a05176b62b3a025a563dda8deb7f2fc3e177ae3477448d39ae4ebe7ae8ec65a4421f754667fd6d7c2eb93f1a18d3d1a6235736bcdb74746f46d88e65dc07c2591e1f95dda5e5e20e105ee8b4ddcaaf36021290d6b6493671d8aafae145d9b90bec3cc60179bb8fc30f143c575d5d861623721b6547d3aaaade455f05fef9318abcd29bd19b12c35ca756de5108c185ece4aa1bf1a8e38809797067bd1f52b6cf2c415e73f9246bd5bfadd7b9a9d2b5369701e72147e22da7e092d9b578fb0c044a36effcbd709258500a00cff230962c44225712fc43f9e802baead7f9cb46ab4931f663c6e3ed4082d59610f01741b5f24566b01b3e3933b29e028c54bd2fc75b549fd05e64c58c9ae0ba417a9e98581db77be75233a42f771c99f0a49b494f0955202b19d6c740e866066104e463e65e4bad9a081636d05367426153f04bcb2712186dca6834388e82520d34efd8a89313b2c7e60
" ~k:"aa63e91cb3fa545c447a8b8309a569d48104e14d5d05b8951033ac8a7d711c3f
" ~r:"0041b1c756dd2e42714f9ee7edce21ea33ef49dbf452ccd9357d5f45ffab08f9
" ~s:"102c6eaad38d39c0d036335ae19dd0d75e8dcabae59b120f69cbd2b5cf48abdb

" ; case ~msg:"a6fc89a223022ee9e508725278582f56db9cd24c0d75d072a528d0c60f27171ea376e2dc28a9dc0b12e668af77dcbb381737e1ba7d9e80b9bec80bf9061b8fa10e43a7403a291624a600dd4f5c2b50c52d5c6155d52be5a325f6ad813fb3ecaf6d1f92e98cc87c26c68cbd15d548a3782bffdd1116c7c11fcabde4025fec5154
" ~x:"1b41c29364947768876ad4e7abcae59c8e61373d25274ba42ceb3d876d6ce672
" ~y:"6c1d4d6b52aa4bff35f4302330052777f51f6a0849161f906ef217b04b18545ce52ae4ae423ad1b4f8b1735ae00ab0c044a56f945da84d1cdc26e082d7acd772dfabcd18b5e13c05c2791a8dc16146e151323e4ef2ce5d64389f69d9347aa2a5bd0114de0eecdf990a440d1bf9890dd95fd640d2fb1789ca6a6dbee1836ad7cb47370b7456e49f3bac03310f8cbe61dd1cc06d78c76fec6397e608a4cac4e2c38983ce5aa9dcba074a206fa608db35f2ad3d63d95b2cb7a01c33d498767e8e68578e4e99538bf3d703e63863a25091452e73b96a3716e9cc109b66008fa5cafdbf96b7fc10c3bb89d79d45ffefc01908d247ef1d4fcb903bf5e7917af88618a52a12004798890540a5a75c65fbc057d860f4b65d8b08b8d215f056d8e5e38bf0b319e294db242a4fc79b2e106feca2556d146f5203fd72adc73a48e3a5aadbb293a2ef5862654c31539ad856a16e5716c437b474f3339cd84f0ac92bc2ca6fac10c751d099a90408def6106ca83893d87e32818d7634537a4ef667ce7f26a5cb
" ~k:"4c9ace2c908648032151f638e3c909d1f0646fe018a1c9c22a170eff64447fbe
" ~r:"48bd010c1af77b3c40db50349706d64d16cbb72db51943d345151deacd4a4133
" ~s:"0f1c4bdb4758ab3b5518d4605b9864805723d33a36116ea650546feef11c4a5e

" ; case ~msg:"2ae4ac7ce29ae7d32490d3a54b715db3f47306f84b59b33b21622a18aa2c060a4434adfa01ff1686b5d1dd3035308e92f7acc76dea969deefb98c2972b42a596e1055a5aa2c661f0b734ba4f0b341c77827d88915a5e89f95a98d63d7729874fce4ff75d7add74f4313dff784e417b2ee1fcd270c038dbbbb96a7768484b8854
" ~x:"87980da0684558f87e5864ae585864625aed61b1309c1d5f30f6477f947c44fb
" ~y:"0a84298f4768e9d7bf796d06585e8b75fbde658398a224a8ac3a49fb91235eaaa183aa8827cc2af79ea334dc8be4cc729029ab5f8161f718f7bfbe90ad2a159888523982b6d4932d8159495ba84d0ab35d7e395d14dba906a1679ae3cbb72c10ed6fa14da4d60077b0bfb591a3dec643996c396338a51d446bde6224aea16aef41f354e09a9dce9f3a00cb445a5c9cae4a6c3c1919c9e0c53082173d0ec00ae5e15aa7260750b6a03ef05a518a48615340ac20984073cea5fc990d489858949aaf6e9e347b4802afbe25a0669472bd9316ba2c23a61cc3aadf1b70d9fd9761bb035f0ca51edb2b12fcfd651cb92363ef48005a2683fd2ed8665d70588fd9a1be3aa51c958b81f13e4acfaf0d2a90aaaef21b2cc9ef2ed37bce3c47c8bcbfc1fb9f94e49bd2f1a30a88df22735a0fdf0ac6028a008b062c9560c42a476997dd21100692ef6396d5f3fb2c155328257e7b7d2bc05fabd54a81a2272993d342bec8577c64d51b4cdbe3654dae568c4da018618c3047aee06bf2621e056b335d044b
" ~k:"25b9d8fbe7e3ab7017f2b1e53da579df460dfb72ba5fe4ae4c85b8c23472bc8c
" ~r:"6b7ed3a4c2a4f78500c7e947e6175c5ca857c9d613e7790b9be0d14ec8403e5f
" ~s:"a116f3de166260d110e20e84eb8c97c3f018178608a2ea3e3e2f5ed91d43de11

" ; case ~msg:"3eade9a10fb59af36a540170737fbc536e4c5230b8f6c4b216eddd3ea92342123a3374d0c751b24b627f9ead4de26e9a7897d9bc5d58a6a3ac74cd4575b3286ec15f8453224f37179e51d9c4ad8a60bf37d71c62ad7fc53e5c7b12f4aaa2d428e5c889fd7f062c913d9b574f4b5db516c976bad588302f219fd83e18bee8e68e
" ~x:"6cf453178db0dd7f2f94f9a1f518c622c1ddee46d4b090462812e9f7b862265b
" ~y:"08a15b2384dff4f3033c87168673c567059870c8e78d2fddc7540afda8058df384d3182a42615432ff93777d3fce49c117c7bbe821e6789b5137ddf084656098aa7b0516fd30a42c8c86d94e6b268b6e13011d25eba018ca40cf8a35e1963135d5cd65a57aca8b007988a5ea75adb4d01cc0f0838ab42d3df643a7d2561cfd1fdebe3ad86ad03de317027533d523351be532bc731aaf43b8642a7da80873b80dc61b7a249e5860fd1a3eae0f8f0cf21e205d6f403cb0a103290c9e69d38cbed9e092b69f71f9172b3676f29a97133fc3e18746fedc653fbfb62c5e0afe89a8e1b8724b1a3314c4cacc4bb8f390439701a614ae9bcdafd472b0ab131667dbbf1c790f73ab9046a58932691a930b3c42e908b4d1f47ed6e2ff18d6b70bb16d1af7993bdb2ca3cb359a0b43f8dc844dea6aebaa34b8d2b6fc288419780ff980908926c46c3b0e595fa308f4e894ecb683c804c93140d91769132d37e93791b9f89d595e698f049b3a9502abc488bdd9472f1131a757f3d54b149067507d1b04a976
" ~k:"a3fb61e544d59206d334049e8554d97b6699db616871fd2b421229c28e84f73c
" ~r:"9e833ec3ded9d81ea7422bdac78422274fa35348e3fce3bbc93b3c10d70b4f1e
" ~s:"653756594eac681d48a2358a0f82a10faa7929b00fd9cd4394c32679060f96e3

" ; case ~msg:"33decfc1e06b92ed81cd30ee3771470b59e22c1564647f1aae8510729715a8ce94624a11554ac909c924aec853df64327546db85d3df597916a39353388a8b3363765281a4352701ff1af43fba6d03664127c15da7b84c04d5409c364094dc62e37983a8eb066880de8136701406e67250679300d2b97d228327c1514c0bc1ea
" ~x:"3bf2be01d154c23ccae92ae93f78ea36f70efcf7fb7eb43cdcaeb9ffb8471b10
" ~y:"16ea2e795c636c9d312159a579b8df329ffc28fecc4a4c13b16a290bd1525a53a97d72315be251d11d23ca78bbec45c0e243279b1eb6e206a9273c1e766e213648bdf90c40479df48acfd9c209a523c8b4a99a481ca8df4774b3bb29f82526520c2dc28ab314fe14140f2be1792e1ac3c759ad44f7845a2012f64ecab0b1fec0ed166bd175955704f62d9401111ffc04f804e48fe774dfd346bb41f4beca2b34a83134a3884a01729cce1abc5b8d0de3fe2654c374deb246d96ffaffc7aa2055b74e819bbeec137eb3caed1fc71f129c8ea8b763f2f57e88de0845f76ceb1841559019872a5b5a969c9cf385d6578b4f27b5b76be3ef0a8fd3ee47eed695e16f14e2a3b791f2a016d6b86ff8ec2343c6a5c80ab6224b6502eb374c8fa6510bce990d70efdfa9a0b702585595184514c78f7e905b6fd6c237333d560fcc06303637ac0b2c7f7c4da559e31f531df2e5d6c651591771d7ea4575888afc4011fa1124fbd1a282a41d933989eff91a51cd39bce7fb0d569fedcc42de48bf18ee755f
" ~k:"a0c97f80ca449fd8f69733e046664408da590dbbab6865c3275c389a478aa248
" ~r:"6f77a52169a2e880a3b55aa278f6463032dc5f81c38468224d5532f6a601f2d9
" ~s:"96b753efb4abbc8c179d03cc2a1a0c1256e23d1fa2e97cfbf55d2bb69812d100

" ; case ~msg:"6ae5a6da794f923f6d8032549b81d04ae7aa35c2099dffbdd83bb94db574faf8f95c7126db2db60fed50f740e87c359544dc2ebfbcafb094ddca69c914d27e5f3d10fa0ce32d2a1355bcf61a2574c755d7c324a2e0ed6f7719ba2f2c9f113df8d04025f4abd2e1c4b7bc18d8acec9f6d8d797cd7b042f50348eeb3f7a2922da7
" ~x:"3b4a52c8b5c386f26ac6ffabcef2df3bf8b25e6108ab540d314dd3d9245c075d
" ~y:"93106fb000c67f1111c6fd3da0f44b4ae4cb3695de2e35b241dfe88d3269b8fda25bf348008725fd613cd61aa826bd8f1aaaee22b4dc0a02842290bb7dad91af0b2854ffab16932208d272f2c081c13889db3ed0b24646c665af9f4b723898eb1ac0053f2a0f4cf22fd4e12940b5b52269484ebb8abc484c06eddbd9b1a426132f402efdcd88ab29e7e510961af8ec83a642e34015858ac3f32197601a888e16c759c94ec5b8dec0da30643b9d9db2574af29e78f9d3f6a7b4c76f45cd0b2ab5e8524935b886918b5d9e9ccb5a6853e62efad2dff83a8520985ee8442f2bdd1c5f9d48062ade6b288c8ad82a41db6c34e2deba541aaac3cd3156c975efbbc718ebd4961996b3ed1cc5c2987ab779052cdbecf51d17661b498e84371ff859f89906f426f563572f66c279ef3d036a427778463f67f8d4de623fb4b2803007871d0a349ec202a9aa1cffef70137e009303497214ada786357a4d8046255e40f89ea588000634e7f0aaf64d92aa21fff8fbe078baa96961699738b268237eab606c
" ~k:"39f68875cade6ae208d3043b010541624679df649cc5d97b09a3ebbe2c9d59be
" ~r:"8636d4d3203aa0912fbfc938be4370077ea9c75195cd2f67e6ee427cde531c40
" ~s:"93023d97efb4327e9e886e7b783741e9d2c397af9c67b91cdb8aa27f83bb025d

" ; case ~msg:"86e03bc3f4ddea6a93888ee389b15eb690822c71f9b85efaaffc52e486b1144ad7fcff3f53bf97da2481e85e0983ee1d5279e27a364d0e690f587a31535fb94eece747f8b605724adfb258c9983c9002e0c11b7976627690d58281305ea9308db74c491a28192e354b600e8376811ccefb751bb10c7d97b42ffe304bee97ecaf
" ~x:"7f56c74b495a12db963e03cfafe60ac95e8019cb212c332d1f19c64615568119
" ~y:"23ed5445391a5bb94e00c76ec80d83728d5d461be425da79f921bca27d625cb42b323971022ad4c3f05bca109910fd06ba39e95bebe794ed108d2ead297ad794f99c32c219e65fb726532715b1bc2075dd4b6949297712f91d5ba061196fb25754c34377bbbe6a37f61787ea844d359285c78e733eb65f665a6b157f832b5638d74ebe1d5dce66d528925e44eef13bf23f807da35f34d169a687758229b99a313acecfb20b142b534926d59aaa7643a79030e9335ef28abeddac8ac9471da4997e33f3e491db8668a2c3920a3b3a37225179361d5539beb33f3252244267465e48faf575cdac938133effe9d1f69f19f1b44b245a447b1fc2b859244e2e39053595cf7978933c3d468c65c231663070aeaf2ec23138d1660081a55bdc3dd3f2446176b1d6d9977a14ebd0ed4d8dfcdfc4a433118401f2c2632095ce7ae6200c74bda5d2fd3854524c3081741975a076a1b4f933ec32a2bac9171bebfdf3b355eddb1f455ecaf73396e85fb04797558ba4f2bbc49d9f2329a23b393301ae0db92
" ~k:"407180cc311aebdc1cdcb4685241597783f34076672362a24a21193c0d45d24d
" ~r:"68efaa05eb90c48c6a7a45337c29175f8ee5b19b53db4ebd83a02f53c5b2104b
" ~s:"145f13f1ae3675c521b334ce6a49fc6f502e3ac6b2b5143be0641d0d57b3c722

" ; case ~msg:"1d0954ee0de1e9ceee0532597ee434c73fe4f66635f7e72d38b67763c66817f53cf36ca0f613e01896cebc9f77a772607f4aeedd3856c73fc2f19100aa7b540ccd057f26cd9564d673228c68088e5f1abf1254a97ed1453ee558e062711ceb7643b345ad33b649affbe8a62067f9d84ed4c8506fcff578d2eba596a205267387
" ~x:"0b48499625f0c2548bf8a2fed1f6696f59df8fbe6eaf91b82385994209c2d04f
" ~y:"2f0d89ac78a61fb74f81142b17766656d1788940077808e3d880ce10ec60e2bbb158d54e020dbc5f6786c0b43cca2cb002c8ce13b291b250f399e8e02f195926978f6c5b007d4f0a66048996a9932a918b2363c4008f547adcaa7d12694baee4fbca34bc6d7e29c5049cda13698fcce61bd3b3db05d2158132dd380cf653cccdf279aa164134bfbddd7ea347760041f92c3a4cfde0092d5cb96bb8c24e98259475596f3377d59f11661bcc0d47e83cb31aae9dcb4a6f25619a29054b62aa8b421e529e61ac95a0de01c50b09e119516c2c5b3563d47eed679a1cf80ba70a50254d851a13a778e1a08da8667e46e35979c15df45cf7886dde5af9d744624b981acd252ec5ba46870b8ee4b32b1be1b944802d91d8148d38f54315a7ad4e38079ea2bed9df8fa59414ddded3a1d2308ba769ae2a652f10c2d96917edfe5874885f3c99d6912f69ae3fc3b4de82decc30edc9314f7ec9e567b7e00de21959486a887d74a5b2180293df5dbeae1e35a6e937b2506d205092cc4c3595db92fc255af5
" ~k:"1c020abb0e1d52b3ad95467f7baaf665e2281f34c342401ef1fb4c1fc2d7b2bd
" ~r:"a67210341a04cd3a4b63ebc7e6208f37e487a8c6f1134cd2601b844d6903203f
" ~s:"6b972c622cab48d85a2dde355f947a8151a17a0acf06b7f3659f868d5ece92d9

" ; case ~msg:"14f566c5fe44aaad6e8b3c627570aabdd4efb7fcfa1ab1bb74f2c6d8795e88233dac4e7d240abd5e9bbd8e1fb03a3bf50c0ca92c9aef1894f2aed600fc5873d23451d3204d75ab9581cbcf82ae8c0df0dfbd3a1f149f70660865726cdc73c015d5ddbf7513eedcd1ef17578d2719fea1e5ba39aef3fa6f00846f0fb8d9a1a436
" ~x:"7928d3edc11a890fe332c0d3759bc6ecb822438d7f604da76b4fd78590720ddb
" ~y:"a36a333900035d3453139b28356bf0124e571f55a5e4259b8b2ee1457cc3588056d6c6a645d422cac72474c5901d0a7f410df7f9b4e22f8684867d9332e2d4266a6e595e515becff7fb94d21a8a9ad7211572e44ce8448317b34c3c0b89b3097ab2ec134ec7c178c2278309cf9152b223bb937e68682f1f680c17ee59ecd0698a05c24c135d2b0238e71f807e079f175e11671308f5bd9e5a69712a9c508b3b50925d1276d552bda51cef3bd0fbd00a9d2dddf0e5ecb6b328378ea637b493846480ed75a3152d9e6a4884eebad12b07cad8d101b3d001bc99fb1eee4e98fd6fc920cb5765ec24e62abd32f975a47d50f61553e1c14775193b53b05b7d02024aace818ab659d717d11deacc9877b818a51689d239b60f7f9ed4caf7325ac0b31b316c036599ea66959d525fd16f5c1a2a809f2866ee9e99f6d8a3c42b58d33d0e5d38055c55c7bccdef310ccd3426207dbbc60faf9f2a219ab367ce84623b81104822e2c77ec5b133ce7050caed090946c1f1355d878a1317de694e686c62ffdf
" ~k:"01f77e5f125a9a1385349f77d7a32f26b1efa5b0a5d4a212753bb54d300d088e
" ~r:"12b40bd1c866ce38e7da0764d807ae82512b33b51dc908e5a5b3d7c16f0d08a5
" ~s:"5caccee2bc85e28d506a9bc6d260dbd08205b75d20690e26aa6bed30d7327099

" ; case ~msg:"60c29d99753d0847bb52e906c862a1b0628496416c14df5dcfbb5e2804f502cb0a2d163e9bc2d84122c0b3f5d0609b82ac16aa15efd55f55c8caa3d1114ac0cb83e5ff3db12a24b89aca5f0514d2ceb09b14fa916000c0f4deb016db755e88b326172144e4f1a705a800559b3da3c27af55cb32b1147460c31186d99dc1cf2e5
" ~x:"3dd64db4bd8e28e701235ad83a5d5e9dd13ee8a3b3dcb4c99c1bc95b6ae25291
" ~y:"a37397e6eafbdcf1e0158f1f4ea1cb3a1ebd739c8559a500def3b7551799d652b327101cfea0b87016db591522b9b34ed267132c5255e77653c4eb935ce0c822b4b10a5e8f3cce39ad1b9606de5be2b2d36e1c5411f06aba0461ea8dc48b649f108eba88def44daa2a5c653dccf1d8ae29205dd5c340e34b7bd698eccdcd345bd4aa5eee3c08b9162ca1804872de3c575d572f34dd48b41f8235d0f511c8dc65daeb07095c3b5dbd3a076f8eb24412f3621f492126737a9d73014defa5f5d57bdc6faf53142eb191606f2fd3dc035f4b8ae84d655cb6daaaf889005c3c334ffd7e3b0498fae2a6f8dc1bc62f3704c8f8c005c8019e0bf45b7aa8e0803b93a992675e381f61a898582950b9ce40e7cdb0300f4b26f9b44484e89c9234179b60a372fe9476f84de0ed4b93497216fb96bae43297dcdc8496c634100cf066402c7d290a7cd28cbcf8b08ad4c136db2fe992ffa045bf8cb249234f29a674762a56d20897ea5538c674a14353db64ba60fe4052a0528eb0b25887e3c5ea69b41f68b3
" ~k:"453b64f2dedfeb1419b5dbeb726a2c92b1a37ef11a7732c911d9a96184285f40
" ~r:"72cf0e18e4bc3749647cdfa62dcbd2513c7c2b1d397c1fcbc7f6a425ebb897ce
" ~s:"7b7d0a9e93340941bb55f6afa6cd63f7364963671008ede457d05b6545fab1f1

" ; case ~msg:"b3dea62a20a9ed9da990465bebe18aa71f08f93fbaee4fe5d581ffaa6fd55cbe272a115d7fa18fb9cf5662f595b7cb9bdb97a81bdc078ee3bdceb2c03722610134c3bbfd7a6f8b79ecc6a9a7709265687f9c236fc68b02203ba666e9eced5161de90c110ee7af9bf54d84a22181f17b4329348bdeefbb324962e63569f07c285
" ~x:"6327d3818c87e4c99b7e5116fc091a9da1e4c02aab9b207d61e859dda8d859ed
" ~y:"45013318b941a710b8ab1010d818c3103634658d2e3e2f413165860805e08d5c1e80add9969a3d3a0d23432c8a48cce836b24a410892099bbf53cc5a84a95e1eb3b682d2754e721efc86d3f4248baa337d6f6e5dac4759b296165918a71b31ced25bf1b05d675bfa222980608fda8f9d0eba9aa08475512d04c612133c88253bf3e27e9ffe3a8570be204f54bf8ff1c7fe42aece832050aabdd9415764b8c872697f9c8e78e2f56bd235ebbbb4b9cf8f054b60292963764536d6fd4c6cfaa1baea53546c6ffb56a04fbfaee001228280aec40e66d9dc192f9ba743bd3ffc0eaf277b6ba3d33c3697024892b0b35419534873fb7a3d594dd6ae0751a2fa430ba46237f4a55e4a678072c651fe6ad314a010fdfe8f8b5342bdabe9ae5910c6f44a51f47524a6fe8216830ccaeded26ce1f13f7f216e0b7809e9272563cab3352b8ed766650227bfe16e981b505609c41f03dca98e219d02aa7d91921edb3a89229e78c30161cc13973b35de3c87779378b8d607a19320405661312432dd8d07af2
" ~k:"94a0f6f58f004e45ce5ffffa6e63abca8daf7768cdafd517f3a5e399828b1e72
" ~r:"3ec677e91c63e65aaa174aee2791dc409244cb80c0220991dcb497397a3c5e9b
" ~s:"1de0ec466b2ad4ed1adce3bc38ee521803dc87085e2fbfc561d63844c1a9a2e6" ]

let sha512_n256_cases2 =
  (* [mod = L=3072, N=256, SHA-512] *)

  let domain = params
~p:"c1d0a6d0b5ed615dee76ac5a60dd35ecb000a202063018b1ba0a06fe7a00f765db1c59a680cecfe3ad41475badb5ad50b6147e2596b88d34656052aca79486ea6f6ec90b23e363f3ab8cdc8b93b62a070e02688ea877843a4685c2ba6db111e9addbd7ca4bce65bb10c9ceb69bf806e2ebd7e54edeb7f996a65c907b50efdf8e575bae462a219c302fef2ae81d73cee75274625b5fc29c6d60c057ed9e7b0d46ad2f57fe01f823230f31422722319ce0abf1f141f326c00fbc2be4cdb8944b6fd050bd300bdb1c5f4da72537e553e01d51239c4d461860f1fb4fd8fa79f5d5263ff62fed7008e2e0a2d36bf7b9062d0d75db226c3464b67ba24101b085f2c670c0f87ae530d98ee60c5472f4aa15fb25041e19106354da06bc2b1d322d40ed97b21fd1cdad3025c69da6ce9c7ddf3dcf1ea4d56577bfdec23071c1f05ee4077b5391e9a404eaffe12d1ea62d06acd6bf19e91a158d2066b4cd20e4c4e52ffb1d5204cd022bc7108f2c799fb468866ef1cb09bce09dfd49e4740ff8140497be61
" ~q:"bf65441c987b7737385eadec158dd01614da6f15386248e59f3cddbefc8e9dd1
" ~g:"c02ac85375fab80ba2a784b94e4d145b3be0f92090eba17bd12358cf3e03f4379584f8742252f76b1ede3fc37281420e74a963e4c088796ff2bab8db6e9a4530fc67d51f88b905ab43995aab46364cb40c1256f0466f3dbce36203ef228b35e90247e95e5115e831b126b628ee984f349911d30ffb9d613b50a84dfa1f042ba536b82d5101e711c629f9f2096dc834deec63b70f2a2315a6d27323b995aa20d3d0737075186f5049af6f512a0c38a9da06817f4b619b94520edfac85c4a6e2e186225c95a04ec3c3422b8deb284e98d24b31465802008a097c25969e826c2baa59d2cba33d6c1d9f3962330c1fcda7cfb18508fea7d0555e3a169daed353f3ee6f4bb30244319161dff6438a37ca793b24bbb1b1bc2194fc6e6ef60278157899cb03c5dd6fc91a836eb20a25c09945643d95f7bd50d206684d6ffc14d16d82d5f781225bff908392a5793b803f9b70b4dfcb394f9ed81c18e391a09eb3f93a032d81ba670cabfd6f64aa5e3374cb7c2029f45200e4f0bfd820c8bd58dc5eeb34"
  in
  let case = case_of ~domain ~hash:Digestif.sha512 in

[ case ~msg:"494180eed0951371bbaf0a850ef13679df49c1f13fe3770b6c13285bf3ad93dc4ab018aab9139d74200808e9c55bf88300324cc697efeaa641d37f3acf72d8c97bff0182a35b940150c98a03ef41a3e1487440c923a988e53ca3ce883a2fb532bb7441c122f1dc2f9d0b0bc07f26ba29a35cdf0da846a9d8eab405cbf8c8e77f
" ~x:"150b5c51ea6402276bc912322f0404f6d57ff7d32afcaa83b6dfde11abb48181
" ~y:"6da54f2b0ddb4dcce2da1edfa16ba84953d8429ce60cd111a5c65edcf7ba5b8d9387ab6881c24880b2afbdb437e9ed7ffb8e96beca7ea80d1d90f24d546112629df5c9e9661742cc872fdb3d409bc77b75b17c7e6cfff86261071c4b5c9f9898be1e9e27349b933c34fb345685f8fc6c12470d124cecf51b5d5adbf5e7a2490f8d67aac53a82ed6a2110686cf631c348bcbc4cf156f3a6980163e2feca72a45f6b3d68c10e5a2283b470b7292674490383f75fa26ccf93c0e1c8d0628ca35f2f3d9b6876505d118988957237a2fc8051cb47b410e8b7a619e73b1350a9f6a260c5f16841e7c4db53d8eaa0b4708d62f95b2a72e2f04ca14647bca6b5e3ee707fcdf758b925eb8d4e6ace4fc7443c9bc5819ff9e555be098aa055066828e21b818fedc3aac517a0ee8f9060bd86e0d4cce212ab6a3a243c5ec0274563353ca7103af085e8f41be524fbb75cda88903907df94bfd69373e288949bd0626d85c1398b3073a139d5c747d24afdae7a3e745437335d0ee993eef36a3041c912f7eb58
" ~k:"b599111b9f78402cefe7bde8bf553b6ca00d5abaf9a158aa42f2607bf78510bc
" ~r:"a40a6c905654c55fc58e99c7d1a3feea2c5be64823d4086ce811f334cfdc448d
" ~s:"6478050977ec585980454e0a2f26a03037b921ca588a78a4daff7e84d49a8a6c

" ; case ~msg:"c01c47bfa208e2f19ddda5cde5833325d16a83fbda29e666fe67ff3489803a6478a5ac17ff01edc7973d15fe4998f63bbc095fc1ac07534241c643a44444dc9a356fa812d5ca191a2f6ed162a2d5fd6d0aa898a20563d993830254db8a4bf65ba86099cc6b58a1bf6ebb01a19c79304308acebe1da09f1753a195e9ef586c7e1
" ~x:"9f11370ddb3c43e2f4162dc61f7e08dfc6e86d5d71742c6adcb5340f7bea2ada
" ~y:"970d38cd8b3f16659ec42a46a19ff06ce8495b9f477d9b7e35ae1035b08b0ee17a0c3ceedf029846e3aeb912f850881c2277f82281e7c0741d2f87e9fa5c30677fe7268cc5fd9aed29f308d9be8de92b961e39c1dbc46790c99b7e29579daf888176d5ce16db5cabfcbe4209ac4753b0e96b15d0b82c7eefb42a10de88f8a7723492a2be5451c1c6ec68ca759d8b4ee418826e71f39cd07654d00d0e0f88d0924bdb97aaca5a6346ad69fc223cd57f5bb0300477b594aa445e5ea8896cdf3bc882e8fa5523b8a332fd98e9d0a924578944d24a41cbeae3ed7b37dffb2f60c0084eaf005c1251823da41d2a5d977d8e483ddb33f73fbc27254a814b616d6a390513f0567a563ac053a76667197b4558f871b69cbf2c116ce457513f60b4f528e2dcdaa71a9a3a4cccb3738a22937bca2a042bef8a74a600acd26975c891466d7e57cc930984212ee0eaf174ebcbafbeb8cc12bc43bfdb00fd11576c439513ef5b59a88fa5a9ae963d94dafd78f81ee7b0d7fab53e41bbf65f8449a4f58b44f9e3
" ~k:"ab53984e0b154992ace73bba548185b49719bcc3b11fb150b5da279529750078
" ~r:"5bb50e4f538a6e4638206be119dbf712776154acfb4c06d65d66c80212341739
" ~s:"7b7e640cd76086d3f640d18ceb26bb53e30282afb17401e7b48aa6818934dc5c

" ; case ~msg:"47e7af22c9298ad3bfef9bee5086bedbdc513d67416d5f4e7981cddb1002cba24700c45dd6d4dcef4f81d003f0513dab4e04eb4c70d944042e1b726d8a33050d0e4f70c0a8341b75fd4e27c7948754e441208eb93fc7b7c37354252f73b838fd02d078c6a1ae073ef1233aa1c8aa2781e193ba2897ccdd8cf617ca23541ce1c5
" ~x:"232c1c88d571492779172ce6650524cb6d91174e8a23780d0fdf7c44ffd80c1a
" ~y:"75163af15cd6b228251504ba024df51df32f638e37f0f2f9d08837f8c6ecfba43eb515ccbabea11b01e1e1fd3cfe7e405fc7f8142b07315e1dc37b08c78668421e2a21fc5d811d26558c504abc4e6fddf03740b8a27fa2ebcda5460ad785706c53cd2d14093d923df942051cbba2586b4d54709d24babe2f7c61a50da8451895999166e80c0fab892a37eb6782745596b49f96e11e9a957c8ec650d2d9a40aa4b014d2e9a4c08b9d7bfeaf1ecd42785b95c0172ae21cf25c4d368bb5100b6e6d92310b28b7b1afe64d496b9c60b763cac08ac46a6bce1bbd3ac8bb76bb55b649b7594820ab6ef7dd1b09bb12852816b61e6dbefab742e0ea2cda47eac7d9d913ddd4bfd8b2eb5f01951caa4f413eb5e7a41a0685695f8331a394e06b1495c170f30ac294660e8909843f9f11c4bfa64e8792df677da0a08aae32a8a4e7067fc35eee03964e8afbdb6a421b8248add284789e4ed3cace7106c23fe6666c4b12b836e7307a55ab24d92d58ac84e71f81dc9b0b7436ad07f74994af7d0b049bd09a
" ~k:"101acd88a048a6a87c13ff23225dc2c4d2fe3fff039e072fbb268ef2dbfab9c3
" ~r:"6175473d7aa7d5ce55590c952a19897206086887fd84bf2b566926e47981c2a3
" ~s:"71d7857b6ff06ca67885fa9c9c71b8cc246d0339b6c2725247172a297e26a7b5

" ; case ~msg:"9311d8f951141713f459eb65f01880b961c0a590b36f785f1aeb880ee71300c0cbc601b3a6072193dad6ddf2028eca4c8bd7b8575187928f84bd69c5dcfb0b9d320003c3a863c09ee503e38abe07ce2e0d46b3cec926231a57defa0aebd1a6e01eef4f9b537ae1fcdf64e01434d40ab5019f3965c735411a5c19941f41febf4f
" ~x:"87bde6350da15832966fe70300e5dc66b96ec263344bcfb5de051be34d76262b
" ~y:"287ddc1969156c18420743ade0fa1271ea346c3329f9ca9b5d54ebfa21f676f9e013616239f4bbe60eaf8e1902ed9ac742d8df918876770894b512aaa25c068bde961f56c9b5b87806d7d0a9de7843d3cb0797903126a47bd9422337e3b46bb1f4f4a79fdf9cf6762157118aeee1e71116f34dafce0047f05d43c7f2cbd4cd52d614b7a945d48be44cfebf784332fe99c1ee1aa8310867df20b280da855b19029fa79ecd6dd6919a4d22b5a1400c30e62ce7acc4b28efbdb94ea23afbb64d6e5f7b3975d2ac63b1d048feea835c7f50b425ce3cb418afdf4dc84008473606574e20db5ebf86cb1ad27737d46494b2e485b26b8c95d829cf656f80f96b1a62e7c03c8f20f18dc58bf59916682e6dcc68d34c89c1b1bd6e6b1e15a7dc325e23fd7a35099831dbd75989c738020bf4dc4079ccb0bf12faf3b9d6494a379aacb1b66d07cbcebbf77a6e29aef22f4baa3df40d270b457dde64f00b53759ae57811b64e040cbd42ea90f4e2808bc81dfd663b28584cdb8199da96d3e03d03fb4133e2f
" ~k:"7d1b5d39e51af0c22a56bc57ba6bf8bb6de18f2c256bb2d6fea684add38b1f6f
" ~r:"66f729716456a2781bdb8578fa18d1e64af0edf8ec1dee0a50d25981912fc45a
" ~s:"8c3cccfe6f0cfdc0ac3a542c8e8c85210bbd7f95134c8f035d1ce16f44ab7a06

" ; case ~msg:"808603f7f8439441277913b21bef4e01c89e4113e07cacc33f65ac9849db1ad1a1cb7dd2fecd88ee4139b1638355c623821309f326c16bc658bb4821518238982e5251f7cd37807292153d2b07dddc066e003c6069c371155d2d191f15111f2089ce423f5c2a1f8534e301313c69623f62ba635adce8551733a82a8fac1a66b1
" ~x:"9464ce029452e8602214c5236d9637ce7e59f92536a07ac5ba30f639e09814d4
" ~y:"389672ec6de0b86655cb10f1199f857013b6320d52c8728fbbb5360a9701b1d6ca4f9eecb8487fb879690f85430c582d3d91ef184c8247d162b94d6dfdfe7c4ae867ac1672827970415aa67a1406ac1a6e2c6c13167719e1d1a536d10078427c211cf682051a75ee8322c1408b89d963bd8e85f9eff7bb8ce05ca42225b4bdfead6b897b0feab76c2272b487d27d4e8dcde0f19e4615f7e1114541f61d43533ce788cc4505600b83266b1bea665912196c2c84c36aa93baf5b7464a6ddf547183e2cd058bb50a12765536f0a4d3524af4f31acc609fc447e1729aab97b5a36b01764b84bc5f77f6cc584866d1a6cfb3aa8437895f777f2dc6897499f6c5f02fa1e6c1ead68f3385b733387c6b58f2d11284a63ae7c7cfee42c3f44a3c926adad8107cca1c3f944f9b9e237d9ab35c81391d7c5f5292d1a322f7a12ce108a86237ba4de3c612fa738f53194ba67bed843cd2d4330a5d194d67cf45fa05183e0cb46c2d23a1bae76755c309fa1c31605c88a9214227ce02fe915bcf0d34bce8c8e
" ~k:"5c2bb856c4d87b27e01e2ac1ae6f2fc526ab8bb49a67eda5c1d8cd4253610df3
" ~r:"98fe587e43aa96f9a9bbe8af404a08b02307b36053db87f6db25a3aa36fcc3db
" ~s:"5c94ea70f99f9ff14b8e5dd4a6688398260907176ea80e19c39b14621149f0d6

" ; case ~msg:"ce2aa3ed12c1b8843a3e11b06b5f0e5e63fe8e19c1a38ac446a48eeca8dac6d8b769d7809442c32ac82e93f686ec64347e9444c3f452823c840e8d0cd334b4152002148da16ac8859d189d87d67164c5db16195c081d2edd7d8157e2bf3b97a90b4b4784324eb8ceac4261809f674256daf007c4ab1f222f5fd28398a5b824de
" ~x:"b887c14673cbc63f04f0839ea56a76154027d7eecf41d8d0b53d4892353ae9a4
" ~y:"584fe0eb314afb866c0466c3980a2df54598d8a705dc2d1bf5102eac86312784eebd019b81a7642d4a3c4cc65dbedd8187e3593f0a9bcc06ea367009b7eb4d29b0450061378edbe163efd3f344bb36234fc86fe1c32f2c9995a07c6e957d195e8105f5179c2bd976b3127067c80ca93456c16b98dfcc7de355790f0b15cfd2ff91db09345532d46096c06b40a2304681d62857675ac50e22c7d1ab47589235419cbedd4b7d24b90531e5bfd853e88a28836ac46b6df26760985b962c6a2445809866b46126212aa263ab2a4603ff41a852c7988c2d4386241655a7222fa4e9f6eac6a144a16b059ea25b71a2138491d54ee95a9d6819977f90fe6a59e0cad81b329eba3e68277df04f9828ef6f081610b4595a92113ec6d069ffe97196d956191daabe9877377ad0416b0ee0658663377e07adb24644e8a0e3ce5fc178f152be0cd9b04071890427c6b001d59262f38fe897ce32040daa7807821c40ac8c63505bed0af070443337c9e9a64e44203c36a8ca5064d87aa0d3cd1d403aa6a24ecc
" ~k:"49548238215fed6525693bc3cca3872944a97790087fb35f329b206e6046b32a
" ~r:"54c99b21f28feee27f0e999aac6b49b6b07633e1db18a45952fcf7e73b166bdb
" ~s:"7a18588ea1456f67562d677878346fb34b684b9a8a61a721b3db0e95695ab43a

" ; case ~msg:"17b925e2a1a51c2036e225715f2f771d8f0a6d98c7ed9cacf5aa4cd30ab16afb94e21a7c953e01ca211c28782a06073fdad27713aa8c26ae9ec449aaaa8ccfda8c947172de94b3f20b54af98df152d5d3a636c736ff01bfa699d6214002dc76dbb3f3860d94e0e34edaba5f2bfd6b2bf660086be876451a50f6a2dc7c2b098b7
" ~x:"0684a8fad551c8d08beb05033185e3b4b6b6f6f4920ef9982d72d0a9c7549855
" ~y:"42a93bf44ec7d2fbd651cc1d1ac391d63cab00971a7ff7a56166768b22e611dc4d729faf8c94e7ed4d6f82b7020b7b4d2fb3591cf2295cc6e1b4be2c256c2fdda43e00051114645da91cbed5cc087085f7cecd8bace67889100bcce7928220266fd3faf2ead9c21e423c9948ec70c2d31b668cdc360ddcebdf429720607f96d851235515d6dbdf163f7ea5ddf351baa76f38663fdbfbd5871bb2157df0a43420648c10e4827f54065614623ed3abad10d317be9d49a4c66564f20dcac176b6605a2e3c3c01c362220f352e477419f2b4b238affbc3920e5bb57cebb9a74746d62cdd070f4a13af001d262def014f29b7f754fac84e02d29285b73bb20ac0c8624123a577be8d6a6b9739185e4458090ddb42b005ea4fa8b51007bd9ca5b4cf2a3dca446a87ec83c9548dab46cf3daf86db3bc69a99baed459d6a197f9bf5032c1dc3a877dd7e5c1161124a6d701324e9a9712b824a4fc3b1b353259af225813c27e820b0ba72fb4e78f5c78673924e7fa2f486030284f26cb6fa31da56f49d3f
" ~k:"4a258c125db1f7b775432b53c7a0ff47c00bf7af27abec7fcd42a2916e95e26d
" ~r:"726e4d3baf00b259f4bdca8b0a5e1cbfd37827c48373ef5029f7601a7769478c
" ~s:"903079439ebde1f766d1a8ff33e0f778d77b5e8b7b0d687443c271e8a63b5975

" ; case ~msg:"1c1169f0e790053cd7df780b5c832c64147694b4a6448ba14a426d9def7ddc78e3ed36a12da81cf9c3f245d64c859b6b4d8121d112851974df178defc977db691234d142dff99bea1957891b5d6fe8a787e96369d93c24682debd1cf3fdb64379b8c1b3b73e1bc2467dcb08b86cbd494c01477be24d7900f5a578930f4bddcb6
" ~x:"3fa44778b414ff27436e276ca4904546d3542d128f73c4463c69ff9cea2b7a41
" ~y:"7fca2268fba33bf94e76416a9e3869f8a90c3b0d2d37aacecd3f6785b9a95aeefe9324c3ab09ce61ffde37b50f82b699413f3b54f24d6c52eca62325029523deb05db138778447bc3d0d05aff7d85b5525f2b863d26486e84cde13e2e2117d3fa38a38d1073aaa794ed8eaa7b3d1daa4ac3e808c3738a9cbef3546cd79eccb4faa28b50fce57cdc24015fec390f0e7a7dc9f9c471d22b30c3e4174358f1ad0734cf79a09a639bdf3f3eabda2b47b81f92e2a4f9004dd641370338c02029bbf4971aa67483eea7a4bf7dff3889f84faa5765617ccab37d190a94c57f99d792807a6965e2113586c6c5d1a81abfd372e1c7954e2e09064df4d2d8288f5cdd8106ed84ffa798819a09a732bc204a812c0352e4e39d2ceb88f8e7d3624a5a5f3dc56ea0f9c5290788e12dc463161601ff3ab681bd0403ee03af45d5e586d84d9c901986718193e661256f402de735d2ca696ef6b594868950ae173f22d95856656a9d00610fe8c2bd725ae55d791277b1317085b67188da00645ce91bbe62e324311
" ~k:"a05b9ca1c9532bc050cd0c1150c27bc192154cf64d59dc9a949906f1ded57e35
" ~r:"1026ecee0ac31bdcdbd6103b1343f84b441fc326e1d86ad0903d0b17cfb2ff9c
" ~s:"a5d3cb2e7c39d87640c4547ac6c33afccbfc1820905ba1e5be5b262313277cb9

" ; case ~msg:"805baabdd018d9e5ebb4dc51435be632d2387869756d743788442790d55bb183e26655ae3aac86dc16a48ddd268dd15e18d8320df9a1a0a6cb2b49bc701d7a15e3fe8ddd584a75c8c9aaaecd1efe17324d6261881f3d34685b04f62e968505966c9a5feb0c39b5095e5568e40f20aa21cb2505356dc9049ce56182d94a2d94a9
" ~x:"770b99935d393eb90b583d1251696007cbeb1b35e6c3f4f9bcb62879070e0940
" ~y:"434d0612b2a8332b0ab15614e3ee9fa245131712fb2ba84f71396fff9488dca340a37e820f44c13aa87fc9df0b7aabeae2ed85a9622b8defad474ac362a7039abde33d1df732a052446aff7857bc24d8f61d258015ed2a3060a8bf9d447e7d83d7b497a8e654731969e437b3f46f83eb58f7884ff2a2390f5d821ecaa7fd09a146c55fc1180073cc5aaa607cabb944f6078a4486cf206ddc5635242def2d3e2edcbc026bb84e849518f197399c22a9009dde9afcd8769b241c75d4ccce7f93900b5f488333df47c026c4f2b2767e70d2d9dde78405e226c9952f6db1a2e55829bc8a76c7de5c2b588f3f3e93ce72fadabacb75c7c14669701e0a2ba127bac56863c8c4e7205cc0a73c429a801e9797da4f26e848982306cc3c3439f9e394ddc80b0f13e0d528190638d8b96bba3af889de373b3549fc90a6822964c22171e7601fdefbe5708988b84f3ea554d621600a876415d5bc1e557e948caace563b3702f0915a90a13aada77709eeba8c50a8629351a4787d0d58808ffb8b217c1d164f
" ~k:"424a43cfd90f7b84e9e375572f82ebce7ffb197bd3237a353bf15ddc1a17095f
" ~r:"2d63e6d2568571acfe4a931580a04b974c7aae4ca9aa9610d87be1a91c657c31
" ~s:"574b10d14dcb8f079461b29ae1b91ed6c5ef32f93cbad306697552c11748fe0c

" ; case ~msg:"be8ca5ed4c22a050d8309c7a31acf667f0c0fbaadc64a34d2b63074a763a2b8db73b24cdbaad26cc6f2c3e90df4b25bfa724fce587faa0fd65ffb719f0a0351648230d5354d721d8fa6d0d686c37f257d7d9dbd15f555d5073f8bc71c92139d1f627d743f7d6586d510d19d0d8a555d0bf79ec70596e712183880c89caf69d6f
" ~x:"9886138d837d20e8c6be853cd7de1a66a25748c7a33fd55121a27237623d68d6
" ~y:"afaff7a4d438b464f27415d2e03ed9c416db2bebfbe0ab34f14ee10644885b5a4588877150f46327c2c7a6f712670bfd6237a29452494859948f5e37c0e586656b119a0e01c81acee57c1775a3a146e8fbafc99cd203fc98195687fb94a88a4f44280b03f0895e0eca84db087c1bf7c4843c85597368e839841131e027109daa7b8172a25e11355fa9a9205ac324941a9fe492c48421f0681a47e280803e8bd91b113e0fa1597607430bcb0ad50b9408de0066d6a2324d09cf6e99133654dd64e8c8f70cd6445343758b5cd5a0e77e2d3fa1cb3f7efed76124b2881dfd2028ab5918c389b9c3978271db54a5171515ab2e85eeb10ab307130159bca5fe13cc4a959e88e9267221ac8d14ee6938e149f52ec59125b449cb55c5a0029f018770b31f08440ce6876e6600a32411722f58e6263339bd9d34e17aa574b921228926ff668ce90362c4391ecd0c037454e12fdf80c96bb7a840cd866e8570bb7d6586fbe3d1eae5332931198ba1d5d902d6b7a122dfa77018553a2dd3680a809bb06053
" ~k:"1689eba0aac66b3d0cca9ae1911602f9638937b6be17c23a187be323d0dec7be
" ~r:"9c7d40e214082bd5e71f3bf4be99789303f38e851a76f88cb90aff713080c587
" ~s:"24ca23be94c624b9d736328b53782b5feb384dc9fe6370016cc3f97d8f48b6d0

" ; case ~msg:"62f0cb1bb07f6497a1dc7a66955765a9cc403bde03fef4e16b09d7ec545b4c75d08b6e9c4c5af7232548d45445638d7194a199ef1534e81241eaa9c7e767fd54e2caceea4d2f7215d37baad6b05e28ea093497e2e8e1db6e41a5eb13ffa4caa27108f2263a74cf54bd5b6a6b62284bac99fd7977aaa8ffff18fa8a70ab0debdf
" ~x:"badece34257da3d7b8713f8f0f9f0107b1909c7f99a765ad8405d8c2a20310ee
" ~y:"73554a69e1a09f6191f0aded542a077ee8c814265d745d9ae5c792f442c5fa47b34643d3ba1d5147161898de5188a80714ee36512a618a33e40300ff1187e553f54433e17466af486472bc0778af55ba7346c961d7f13ac6d8d6ac9a42092c01579ee2170590cbc3b45eef795b5d9e5d0a8449439ab307c14c5674c4a7a3eaf8b240ef36dd21f43cced58c2dcf23c314364e8e314e9671e80813d185801358d5df61d7e7ec0dd69e90c2cc75c1c3543efeca82b2ec6ec59e6c99bcd1a8631c6228e216884082da119125cb0a80c8fe344afe66e0f20646432465f3e0096a17725a8867b3bdba3c69a1aacbb8d64755b7f2a3df0a49ba0b2114e112d4cae0ad6d8d0fd618e54d53f07ba109b75a54a989618b2863e4415e176e0bfd88dbf36553ca853bb36316c66eb93da34ff3ae74cd5f187f49bf38af0f393b2d7f854df192ade2df6b39a176d2152c912bba248d84a5b0aa4084a18bb64fd136973f73b413d77db275ea5ece93ce2fa00d7c8887b7e50b00649d0353a7f58cc63f6b5fbdfc
" ~k:"2d468a99e315c158a1af18abd4d58872d6e281dcd4c9b0b43298eddf346496d7
" ~r:"54ff5d3dc8767856a10f54088882e28c110980ef9b204eb5f162dbef73a37c73
" ~s:"57ed0748427c089d6395528b2b4555c01b4c1341ab5fb99c64d1cc247a41c3a8

" ; case ~msg:"baeb12a1ebd8057a99a0137ee60f60eed10d26f1eab22ae2d9adbc3e5ffc3252abf62b614707ad2546141bed779f0cfad9544a74e562da549e2f7b286efb615449b0946dc7c498d8f12150b2eacbd27157966f592ad5f3e43a24c60b7e06630b82a4fdb699119dbd878b13a98bf22a7b3dc7efdd992ce6b8a950e61299c5663b
" ~x:"bd3006cf5d3ac04a8a5128140df6025d9942d78544e9b27efe28b2ca1f79e313
" ~y:"00728e23e74bb82de0e1315d58164a5cecc8951d89e88da702f5b878020fd8d2a1791b3e8ab770e084ac2397d297971ca8708a30a4097d86740153ee2db6ab6343c5b6cc2c8a7fa59082a8d659931cc48a0433a033dbb2fff3aa545686f922c7063da1d52d9688142ec64a1002948e5da89165d9df8eed9aa469b61ee0210b4033562333097ba8659944e5f7924e04a21bc3edc6d551e202e4c543e97518f91e0cab49111029b29c3aa1bed5f35e5c90feb9d3c745953dbf859defce4537b4a09801fdc8fe6999fbde39908079811b4b992c2e8333b9f800ea0d9f0a5f53607e308942e68efef01e03d7cca6f196872bf01f436d4a8e05fc59d8fbc6b88a166f57a4e99d67ddaece844653be77819747dd2e07d581c518cb9779e9f7960c17ff0bae710ecf575b09591b013b4805c88b235df262e61a4c94f46bf9a08284611df44eadd94f44cef6225a808e211e4d3af5e96bce64a90f8013874f10749a8382a6026a855d90853440bfce31f258b3a258f7b5e659b43e702dee7c24c02d2284
" ~k:"16aedfbe554de17a3e5b83e942702bd60702d9823ba154baa6d1e7e94308324d
" ~r:"8d357b0b956fb90e8e0b9ff284cedc88a04d171a90c5997d8ee1e9bc4d0b35ff
" ~s:"ab37329c50145d146505015704fdc4fb0fd7207e0b11d8becbad934e6255c30c

" ; case ~msg:"184e599a4c1de86c4151205754df0b1912c2b3c532552c51a61c6459db98c83e59d4a40806c6a2c6b3fe74e3bb9e720d7d0a3cc11ef88959a8990c0fa057a3915fe0dd9a138aa0ec1cb1ab69d93910d8d6f9e14f3b8a135d3f031a56c76a9dc3aed1962bdf05815c2492d14f2324d2da491810d1672b633f2419da4e7ebdef24
" ~x:"a29e90d33f200b1faf61bee5d92ca8a392b1eaeeaa0817cec98b40c97e25018c
" ~y:"60159720021fd2d5a2f575b3220905788d328d0c46895a46bb985942467209ec28d8ddfdc97ec34da65b164cf48652ac475d8978959cfc4330743ed98137559391b1204da6b26b451211407e8fc77d819934c48709c8eadc620f6db2592b65483265149a324467d93c375d97230f2b1a682897cf6d280df61a34f20f0c7c729a40141958044876c44e595d2378a7d22c6cda9ab816486c294e4eddea7ada88b15eca5371da164471edafcdefc654e64a1f995068fa85dbbb5516137bc442f60717fe59c629081c234f27195d5f9c2bf85cdc1ea4cae57aa908cbff9b2a53353b13e9f6fe45daa5174cd956236d447b52011d688cd22f23018409b39a36079cb53e03b6d3a752733297fea4ca27c6395becef4081d201f41d4a00e99d95f42281dcf44b9ef6754998d94231937c82594218a78463cc837193de6bf1d3c3ec31d8dc5468cb56defc9c70d08b95b029d97aa043d557f6286b87ee4098442df495c0ad8ae4d4ae037312c5f7239032c03b088c1036fad7774b1519709242c9511e6e
" ~k:"78e781b2874ca2441e2ce74a2a2a16417b51537eca876831f6593ae25fbd796c
" ~r:"079d4df14ad703a435b21bc70a03456ca822b876c9accb018bddd674bd6392d7
" ~s:"6c7765e1f1eddf915a56a57390db45636e52f083ce440766ad4f32580f722483

" ; case ~msg:"b189dd34f58f3efa85b6f97677edfb82664cbe43a2550c336ffa08705bbda2545ef244a275014c6a265971f4c3658e5e8d6a3fafc889f3c4eda6b5616092954b15c60435efd76806e28557c05faaaa8a05c262657840865ff69c511a68d13022a712d35bde138eb7a2f8f1a87b342c7caf388c1a8b95079bc4a8003eef84b899
" ~x:"9759c24820670eaeaf92370197d0037f9f71dcc283970f341117fb56a1764001
" ~y:"05e280310810715d29ea1ca00a700378bd5979493b9803174c932b7dadb7029a9a9f9c91cf8f938af2bceaa052f2273f0de393b0f7544490d693f529a68b812e2e589cc092b83ef847c5306039aa8eaf225128926145893a51551db382fda4b63e5abc10fd076100684d4ca657c89b2265de6e0f0473f01bb222b2bc50ec1c5fcde9161831018aab3014a956033bb0a83866df11915808f9e7461645c89c6e17ab65dbf97cbf4ac1164d671a1516ca81645bc3e09913a03f30641bd0920083578ca84df71f62eb756ba445a0dc44f85a9e4f72ce5f6bf82ccbd674d2ce3c4afc300562a7dbd3e8ab838993f9decc9933dc07dc01b502fee5b390461a8c82c4e69615f121b3f9fd4f0c8b7620a25996df43d7cf355f15be09e2c82178c6f8836c36c1d3ef26ad05219fb57e85ef162c8dd8f0e55014769d53cba478a2aa66d90d8acd6cb0489d1eea46c2c41bd5495ab8def43b2cd5bb2673945c21c80a4833fd75d884c7675c09e7191fb26e92c54c7c8208d0a0e8dee75c2968e962de4493e8
" ~k:"86050bf276a649b13c18814430eadcff54edf7416f1a8b1559c6c2c808e8dc9f
" ~r:"9fd105c74a0d36973740867ccc1c731cf1c50c7935d5c09e92f574d7a569157e
" ~s:"501f50c32b0288672e02aca78f90f446acf92626365957a375550c77980c3c17

" ; case ~msg:"42c065fadd56d6a1fe68dd4e86c17efd76d0f9db87036bd7b609159d66847f46de01b8ae43590360fa324559a2d709d45cf01034f5facb7f52324e60dd464a583d42e412659d8420f7265e30cf82bbbcb2c99b0f00ca6a46d28556428789f415000dc31babbd67ccc8fbaa84a880466bca4783eaf00b7f78231c667126433e6a
" ~x:"307555893610e15549a5bfb2b446251f9595eb0c16df5fe3b784ebfc3fc30140
" ~y:"b265edfed77b3ad511e56d583129b12e5796d659d484a2fce350661f79e545dd0a06c23774c8ba2fb5101a2848c413dfc5b374a7c5ff3acc7332f0ff8bd6f5fa882c0a67689308be7154c4efc51835f349525419ed722a90bf26ddded65bc8962ba11de9e734442571affc2d42b9f3f54a46535ae9eb01361adf03fc28410abf41db3ae4113da4c40e9a368f9cd029be4d98c66d835d034e3c86544b60bcb01feb383b2add9afe7b6251a17ad4e5439a9cd2d1bf62b6cf5377c097b7268bd736cca9ceb822e5d1844a09fa69c78217c3d6737f0bf45e3236508b5a3f5c466dd0d75ace95d447f9bd7aa9ee57bd10ee3c5e8389a06c00857e699794f5cacc7dc5bb1504421dc920565618bef05dc1713b6f08bc00681c5a1c0685359729fe4b544090ccceaa82f4fefa9f1117bf1e371b99fe4ed71635dad415017a62341d704227ee7cfb64a8deae90d86c0cfd37ed363d91a4a06fd06f64dbd8142c12503f49eeb1b9a971aeb343f15cd27d279b99d4cfa51f121259b3c1b55d28d994bb3299
" ~k:"5359fe067eb9d98ec2217500de743b0dbe88e8d94552b53a0117aac4d3390083
" ~r:"6ed82af8e89e38c49a58010f0564165a16a76a2bfb348466d9b4a91e5ce53ab2
" ~s:"8c466a8b3e4c90886f29986a4d513904f31db43a68ce880311403cc755466604"
]


let private_key ~p ~q ~g ~x ~y = priv_of_hex ~p ~q ~gg:g ~x ~y

let test_rfc6979 (type a) ~priv ~msg ~(hash: a Digestif.hash) ~k ~r ~s  _ =
  let h1 = Digestif.(digest_string hash msg |> to_raw_string hash) in
  let k' =
    let module H = (val (Digestif.module_of hash)) in
    let module K = Dsa.K_gen (H) in
    K.generate ~key:priv h1 in
  assert_oct_equal
    ~msg:"computed k" k
    (Z_extra.to_octets_be ~size:(Z.numbits priv.Dsa.q // 8) k') ;
  dsa_test ~priv ~msg ~k:k' ~r ~s ~hash ()


let rfc6979_dsa_1024 =
  let priv = private_key
 ~p:"86F5CA03DCFEB225063FF830A0C769B9DD9D6153AD91D7CE27F787C43278B447
       E6533B86B18BED6E8A48B784A14C252C5BE0DBF60B86D6385BD2F12FB763ED88
       73ABFD3F5BA2E0A8C0A59082EAC056935E529DAF7C610467899C77ADEDFC846C
       881870B7B19B2B58F9BE0521A17002E3BDD6B86685EE90B3D9A1B02B782B1779"
~q:"996F967F6C8E388D9E28D01E205FBA957A5698B1"
~g:"07B0F92546150B62514BB771E2A0C0CE387F03BDA6C56B505209FF25FD3C133D
       89BBCD97E904E09114D9A7DEFDEADFC9078EA544D2E401AEECC40BB9FBBF78FD
       87995A10A1C27CB7789B594BA7EFB5C4326A9FE59A070E136DB77175464ADCA4
       17BE5DCE2F40D10A46A3A3943F26AB7FD9C0398FF8C76EE0A56826A8A88F1DBD"
~x:"411602CB19A6CCC34494D79D98EF1E7ED5AF25F7"
~y:"5DF5E01DED31D0297E274E1691C192FE5868FEF9E19A84776454B100CF16F653
       92195A38B90523E2542EE61871C0440CB87C322FC4B4D2EC5E1E7EC766E1BE8D
       4CE935437DC11C3C8FD426338933EBFE739CB3465F4D3668C5E473508253B1E6
       82F65CBDC4FAE93C2EA212390E54905A86E2223170B44EAA7DA5DD9FFCFB7F3B"
  in

  let case ~msg ~hash ~k ~r ~s =
    test_rfc6979 ~priv ~msg ~k:(vx k) ~r:(vx r) ~s:(vx s) ~hash
  in [
    case ~msg:"sample" ~hash:Digestif.sha1
    ~k:"7BDB6B0FF756E1BB5D53583EF979082F9AD5BD5B"
    ~r:"2E1A0C2562B2912CAAF89186FB0F42001585DA55"
    ~s:"29EFB6B0AFF2D7A68EB70CA313022253B9A88DF5" ;

    case ~hash:Digestif.sha224 ~msg:"sample"
   ~k:"562097C06782D60C3037BA7BE104774344687649"
   ~r:"4BC3B686AEA70145856814A6F1BB53346F02101E"
   ~s:"410697B92295D994D21EDD2F4ADA85566F6F94C1" ;

    case ~hash:Digestif.sha256 ~msg:"sample"
   ~k:"519BA0546D0C39202A7D34D7DFA5E760B318BCFB"
   ~r:"81F2F5850BE5BC123C43F71A3033E9384611C545"
   ~s:"4CDD914B65EB6C66A8AAAD27299BEE6B035F5E89" ;

    case ~hash:Digestif.sha384 ~msg:"sample"
   ~k:"95897CD7BBB944AA932DBC579C1C09EB6FCFC595"
   ~r:"07F2108557EE0E3921BC1774F1CA9B410B4CE65A"
   ~s:"54DF70456C86FAC10FAB47C1949AB83F2C6F7595" ;

    case ~hash:Digestif.sha512 ~msg:"sample"
   ~k:"09ECE7CA27D0F5A4DD4E556C9DF1D21D28104F8B"
   ~r:"16C3491F9B8C3FBBDD5E7A7B667057F0D8EE8E1B"
   ~s:"02C36A127A7B89EDBB72E4FFBC71DABC7D4FC69C" ;

    case ~hash:Digestif.sha1 ~msg:"test"
   ~k:"5C842DF4F9E344EE09F056838B42C7A17F4A6433"
   ~r:"42AB2052FD43E123F0607F115052A67DCD9C5C77"
   ~s:"183916B0230D45B9931491D4C6B0BD2FB4AAF088" ;

    case ~hash:Digestif.sha224 ~msg:"test"
   ~k:"4598B8EFC1A53BC8AECD58D1ABBB0C0C71E67297"
   ~r:"6868E9964E36C1689F6037F91F28D5F2C30610F2"
   ~s:"49CEC3ACDC83018C5BD2674ECAAD35B8CD22940F" ;

    case ~hash:Digestif.sha256 ~msg:"test"
   ~k:"5A67592E8128E03A417B0484410FB72C0B630E1A"
   ~r:"22518C127299B0F6FDC9872B282B9E70D0790812"
   ~s:"6837EC18F150D55DE95B5E29BE7AF5D01E4FE160" ;

    case ~hash:Digestif.sha384 ~msg:"test"
   ~k:"220156B761F6CA5E6C9F1B9CF9C24BE25F98CD89"
   ~r:"854CF929B58D73C3CBFDC421E8D5430CD6DB5E66"
   ~s:"91D0E0F53E22F898D158380676A871A157CDA622" ;

    case ~hash:Digestif.sha512 ~msg:"test"
   ~k:"65D2C2EEB175E370F28C75BFCDC028D22C7DBE9C"
   ~r:"8EA47E475BA8AC6F2D821DA3BD212D11A3DEB9A0"
   ~s:"7C670C7AD72B6C050C109E1790008097125433E8"
  ]

let rfc6979_dsa_2048 =
  let priv = private_key
   ~p:"9DB6FB5951B66BB6FE1E140F1D2CE5502374161FD6538DF1648218642F0B5C48
       C8F7A41AADFA187324B87674FA1822B00F1ECF8136943D7C55757264E5A1A44F
       FE012E9936E00C1D3E9310B01C7D179805D3058B2A9F4BB6F9716BFE6117C6B5
       B3CC4D9BE341104AD4A80AD6C94E005F4B993E14F091EB51743BF33050C38DE2
       35567E1B34C3D6A5C0CEAA1A0F368213C3D19843D0B4B09DCB9FC72D39C8DE41
       F1BF14D4BB4563CA28371621CAD3324B6A2D392145BEBFAC748805236F5CA2FE
       92B871CD8F9C36D3292B5509CA8CAA77A2ADFC7BFD77DDA6F71125A7456FEA15
       3E433256A2261C6A06ED3693797E7995FAD5AABBCFBE3EDA2741E375404AE25B"
   ~q:"F2C3119374CE76C9356990B465374A17F23F9ED35089BD969F61C6DDE9998C1F"
   ~g:"5C7FF6B06F8F143FE8288433493E4769C4D988ACE5BE25A0E24809670716C613
       D7B0CEE6932F8FAA7C44D2CB24523DA53FBE4F6EC3595892D1AA58C4328A06C4
       6A15662E7EAA703A1DECF8BBB2D05DBE2EB956C142A338661D10461C0D135472
       085057F3494309FFA73C611F78B32ADBB5740C361C9F35BE90997DB2014E2EF5
       AA61782F52ABEB8BD6432C4DD097BC5423B285DAFB60DC364E8161F4A2A35ACA
       3A10B1C4D203CC76A470A33AFDCBDD92959859ABD8B56E1725252D78EAC66E71
       BA9AE3F1DD2487199874393CD4D832186800654760E1E34C09E4D155179F9EC0
       DC4473F996BDCE6EED1CABED8B6F116F7AD9CF505DF0F998E34AB27514B0FFE7"
   ~x:"69C7548C21D0DFEA6B9A51C9EAD4E27C33D3B3F180316E5BCAB92C933F0E4DBC"
   ~y:"667098C654426C78D7F8201EAC6C203EF030D43605032C2F1FA937E5237DBD94
       9F34A0A2564FE126DC8B715C5141802CE0979C8246463C40E6B6BDAA2513FA61
       1728716C2E4FD53BC95B89E69949D96512E873B9C8F8DFD499CC312882561ADE
       CB31F658E934C0C197F2C4D96B05CBAD67381E7B768891E4DA3843D24D94CDFB
       5126E9B8BF21E8358EE0E0A30EF13FD6A664C0DCE3731F7FB49A4845A4FD8254
       687972A2D382599C9BAC4E0ED7998193078913032558134976410B89D2C171D1
       23AC35FD977219597AA7D15C1A9A428E59194F75C721EBCBCFAE44696A499AFA
       74E04299F132026601638CB87AB79190D4A0986315DA8EEC6561C938996BEADF"
  in

  let case ~msg ~hash ~k ~r ~s =
    test_rfc6979 ~priv ~msg ~k:(vx k) ~r:(vx r) ~s:(vx s) ~hash
  in [
    case ~hash:Digestif.sha1 ~msg:"sample"
   ~k:"888FA6F7738A41BDC9846466ABDB8174C0338250AE50CE955CA16230F9CBD53E"
   ~r:"3A1B2DBD7489D6ED7E608FD036C83AF396E290DBD602408E8677DAABD6E7445A"
   ~s:"D26FCBA19FA3E3058FFC02CA1596CDBB6E0D20CB37B06054F7E36DED0CDBBCCF" ;

   case ~hash:Digestif.sha224 ~msg:"sample"
   ~k:"BC372967702082E1AA4FCE892209F71AE4AD25A6DFD869334E6F153BD0C4D806"
   ~r:"DC9F4DEADA8D8FF588E98FED0AB690FFCE858DC8C79376450EB6B76C24537E2C"
   ~s:"A65A9C3BC7BABE286B195D5DA68616DA8D47FA0097F36DD19F517327DC848CEC" ;


   case ~hash:Digestif.sha256 ~msg:"sample"
   ~k:"8926A27C40484216F052F4427CFD5647338B7B3939BC6573AF4333569D597C52"
   ~r:"EACE8BDBBE353C432A795D9EC556C6D021F7A03F42C36E9BC87E4AC7932CC809"
   ~s:"7081E175455F9247B812B74583E9E94F9EA79BD640DC962533B0680793A38D53" ;


   case ~hash:Digestif.sha384 ~msg:"sample"
   ~k:"C345D5AB3DA0A5BCB7EC8F8FB7A7E96069E03B206371EF7D83E39068EC564920"
   ~r:"B2DA945E91858834FD9BF616EBAC151EDBC4B45D27D0DD4A7F6A22739F45C00B"
   ~s:"19048B63D9FD6BCA1D9BAE3664E1BCB97F7276C306130969F63F38FA8319021B" ;

   case ~hash:Digestif.sha512 ~msg:"sample"
   ~k:"5A12994431785485B3F5F067221517791B85A597B7A9436995C89ED0374668FC"
   ~r:"2016ED092DC5FB669B8EFB3D1F31A91EECB199879BE0CF78F02BA062CB4C942E"
   ~s:"D0C76F84B5F091E141572A639A4FB8C230807EEA7D55C8A154A224400AFF2351" ;

   case ~hash:Digestif.sha1 ~msg:"test"
   ~k:"6EEA486F9D41A037B2C640BC5645694FF8FF4B98D066A25F76BE641CCB24BA4F"
   ~r:"C18270A93CFC6063F57A4DFA86024F700D980E4CF4E2CB65A504397273D98EA0"
   ~s:"414F22E5F31A8B6D33295C7539C1C1BA3A6160D7D68D50AC0D3A5BEAC2884FAA" ;

   case ~hash:Digestif.sha224 ~msg:"test"
   ~k:"06BD4C05ED74719106223BE33F2D95DA6B3B541DAD7BFBD7AC508213B6DA6670"
   ~r:"272ABA31572F6CC55E30BF616B7A265312018DD325BE031BE0CC82AA17870EA3"
   ~s:"E9CC286A52CCE201586722D36D1E917EB96A4EBDB47932F9576AC645B3A60806" ;

   case ~hash:Digestif.sha256 ~msg:"test"
   ~k:"1D6CE6DDA1C5D37307839CD03AB0A5CBB18E60D800937D67DFB4479AAC8DEAD7"
   ~r:"8190012A1969F9957D56FCCAAD223186F423398D58EF5B3CEFD5A4146A4476F0"
   ~s:"7452A53F7075D417B4B013B278D1BB8BBD21863F5E7B1CEE679CF2188E1AB19E" ;

   case ~hash:Digestif.sha384 ~msg:"test"
   ~k:"206E61F73DBE1B2DC8BE736B22B079E9DACD974DB00EEBBC5B64CAD39CF9F91C"
   ~r:"239E66DDBE8F8C230A3D071D601B6FFBDFB5901F94D444C6AF56F732BEB954BE"
   ~s:"6BD737513D5E72FE85D1C750E0F73921FE299B945AAD1C802F15C26A43D34961" ;

   case ~hash:Digestif.sha512 ~msg:"test"
   ~k:"AFF1651E4CD6036D57AA8B2A05CCF1A9D5A40166340ECBBDC55BE10B568AA0AA"
   ~r:"89EC4BB1400ECCFF8E7D9AA515CD1DE7803F2DAFF09693EE7FD1353E90A68307"
   ~s:"C9F0BDABCC0D880BB137A994CC7F3980CE91CC10FAF529FC46565B15CEA854E1"

  ]

let suite =
  List.mapi (fun i f -> "DSA SHA1 " ^ string_of_int i >:: f) sha1_cases @
  List.mapi (fun i f -> "DSA SHA224 " ^ string_of_int i >:: f) sha224_cases @
  List.mapi (fun i f -> "DSA SHA256 " ^ string_of_int i >:: f) sha256_cases @
  List.mapi (fun i f -> "DSA SHA384 " ^ string_of_int i >:: f) sha384_cases @
  List.mapi (fun i f -> "DSA SHA512 " ^ string_of_int i >:: f) sha512_cases @
  List.mapi (fun i f -> "DSA SHA1 N224 " ^ string_of_int i >:: f) sha1_n224_cases @
  List.mapi (fun i f -> "DSA SHA224 N224 " ^ string_of_int i >:: f) sha224_n224_cases @
  List.mapi (fun i f -> "DSA SHA256 N224 " ^ string_of_int i >:: f) sha256_n224_cases @
  List.mapi (fun i f -> "DSA SHA384 N224 " ^ string_of_int i >:: f) sha384_n224_cases @
  List.mapi (fun i f -> "DSA SHA512 N224 " ^ string_of_int i >:: f) sha512_n224_cases @
  List.mapi (fun i f -> "DSA SHA1 N256 " ^ string_of_int i >:: f) sha1_n256_cases @
  List.mapi (fun i f -> "DSA SHA224 N256 " ^ string_of_int i >:: f) sha224_n256_cases @
  List.mapi (fun i f -> "DSA SHA256 N256 " ^ string_of_int i >:: f) sha256_n256_cases @
  List.mapi (fun i f -> "DSA SHA384 N256 " ^ string_of_int i >:: f) sha384_n256_cases @
  List.mapi (fun i f -> "DSA SHA512 N256 " ^ string_of_int i >:: f) sha512_n256_cases @
  List.mapi (fun i f -> "DSA SHA1 N256 2 " ^ string_of_int i >:: f) sha1_n256_cases2 @
  List.mapi (fun i f -> "DSA SHA224 N256 2 " ^ string_of_int i >:: f) sha224_n256_cases2 @
  List.mapi (fun i f -> "DSA SHA256 N256 2 " ^ string_of_int i >:: f) sha256_n256_cases2 @
  List.mapi (fun i f -> "DSA SHA384 N256 2 " ^ string_of_int i >:: f) sha384_n256_cases2 @
  List.mapi (fun i f -> "DSA SHA512 N256 2 " ^ string_of_int i >:: f) sha512_n256_cases2 @
  List.mapi (fun i f -> "RFC6979 DSA 1024 " ^ string_of_int i >:: f) rfc6979_dsa_1024 @
  List.mapi (fun i f -> "RFC6979 DSA 2048 " ^ string_of_int i >:: f) rfc6979_dsa_2048
