/*
 * Copyright (C) 2025 Red Hat Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "config.h"

#include <glib.h>

#include "core/window-private.h"
#include "meta-test/meta-context-test.h"
#include "tests/meta-test-utils.h"
#include "tests/meta-wayland-test-driver.h"
#include "tests/meta-wayland-test-utils.h"
#include "wayland/meta-wayland.h"

static MetaContext *test_context;
static MetaWaylandTestDriver *test_driver;
static MetaVirtualMonitor *virtual_monitor;

#define TEST_CLIENT_TITLE "window-mapped-inhibit-test-window"

static void
wait_for_condition (gboolean (*condition_func) (gpointer),
                    gpointer   user_data)
{
  while (!condition_func (user_data))
    g_main_context_iteration (NULL, TRUE);
}

static gboolean
window_is_not_hidden (gpointer user_data)
{
  MetaWindow *window = (MetaWindow *) user_data;
  return !meta_window_is_hidden (window);
}

static gboolean
window_is_hidden (gpointer user_data)
{
  MetaWindow *window = (MetaWindow *) user_data;
  return meta_window_is_hidden (window);
}

static void
test_window_mapped_inhibit (MetaWindowClientType client_type)
{
  g_autoptr (GError) error = NULL;
  MetaTestClient *test_client;
  MetaWindow *window;

  g_debug ("Starting window mapped inhibit test");

  test_client = meta_test_client_new (test_context,
                                      TEST_CLIENT_TITLE,
                                      client_type,
                                      &error);
  g_assert_no_error (error);

  meta_test_client_run (test_client,
                        "create " TEST_CLIENT_TITLE " csd\n"
                        "show " TEST_CLIENT_TITLE "\n");

  /* Wait for the window to be created */
  while (!(window = meta_test_client_find_window (test_client,
                                                  TEST_CLIENT_TITLE,
                                                  NULL)))
    g_main_context_iteration (NULL, TRUE);
  g_object_add_weak_pointer (G_OBJECT (window), (gpointer *) &window);

  /* Wait for the window to be properly shown */
  wait_for_condition (window_is_not_hidden, window);

  /* Initially, the window should not be mapped inhibited */
  g_assert_false (meta_window_is_mapped_inhibited (window));

  /* The window should be showing normally */
  g_assert_false (meta_window_is_hidden (window));

  /* Inhibit the window mapping */
  meta_window_inhibit_mapped (window);

  /* Now the window should be mapped inhibited */
  g_assert_true (meta_window_is_mapped_inhibited (window));

  /* Wait for the window to be processed and hidden */
  wait_for_condition (window_is_hidden, window);

  /* The window should now be hidden due to the inhibit */
  g_assert_true (meta_window_is_hidden (window));

  /* Uninhibit the window mapping */
  meta_window_uninhibit_mapped (window);

  /* The window should no longer be mapped inhibited */
  g_assert_false (meta_window_is_mapped_inhibited (window));

  /* Wait for the window to be processed and shown again */
  wait_for_condition (window_is_not_hidden, window);

  /* The window should be showing again */
  g_assert_false (meta_window_is_hidden (window));

  g_debug ("Window mapped inhibit test passed");

  meta_test_client_destroy (test_client);

  /* Wait for the window to be removed */
  while (window)
    g_main_context_iteration (NULL, TRUE);
}

static void
test_window_mapped_inhibit_multiple (MetaWindowClientType client_type)
{
  g_autoptr (GError) error = NULL;
  MetaTestClient *test_client;
  MetaWindow *window;

  g_debug ("Starting multiple window mapped inhibit test");

  test_client = meta_test_client_new (test_context,
                                      TEST_CLIENT_TITLE,
                                      client_type,
                                      &error);
  g_assert_no_error (error);

  meta_test_client_run (test_client,
                        "create " TEST_CLIENT_TITLE " csd\n"
                        "show " TEST_CLIENT_TITLE "\n");

  /* Wait for the window to be created */
  while (!(window = meta_test_client_find_window (test_client,
                                                  TEST_CLIENT_TITLE,
                                                  NULL)))
    g_main_context_iteration (NULL, TRUE);
  g_object_add_weak_pointer (G_OBJECT (window), (gpointer *) &window);

  /* Wait for the window to be properly shown */
  wait_for_condition (window_is_not_hidden, window);

  /* Initially, the window should not be mapped inhibited */
  g_assert_false (meta_window_is_mapped_inhibited (window));

  /* Inhibit the window mapping multiple times */
  meta_window_inhibit_mapped (window);
  g_assert_true (meta_window_is_mapped_inhibited (window));
  wait_for_condition (window_is_hidden, window);

  meta_window_inhibit_mapped (window);
  g_assert_true (meta_window_is_mapped_inhibited (window));
  g_assert_true (meta_window_is_hidden (window));

  meta_window_inhibit_mapped (window);
  g_assert_true (meta_window_is_mapped_inhibited (window));
  g_assert_true (meta_window_is_hidden (window));

  /* Uninhibit once - should still be inhibited */
  meta_window_uninhibit_mapped (window);
  g_assert_true (meta_window_is_mapped_inhibited (window));
  g_assert_true (meta_window_is_hidden (window));

  /* Uninhibit again - should still be inhibited */
  meta_window_uninhibit_mapped (window);
  g_assert_true (meta_window_is_mapped_inhibited (window));
  g_assert_true (meta_window_is_hidden (window));

  /* Uninhibit the final time - should no longer be inhibited */
  meta_window_uninhibit_mapped (window);
  g_assert_false (meta_window_is_mapped_inhibited (window));
  wait_for_condition (window_is_not_hidden, window);
  g_assert_false (meta_window_is_hidden (window));

  g_debug ("Multiple window mapped inhibit test passed");

  meta_test_client_destroy (test_client);

  /* Wait for the window to be removed */
  while (window)
    g_main_context_iteration (NULL, TRUE);
}

static void
test_window_mapped_inhibit_wayland (void)
{
  test_window_mapped_inhibit (META_WINDOW_CLIENT_TYPE_WAYLAND);
}

static void
test_window_mapped_inhibit_x11 (void)
{
#ifdef MUTTER_PRIVILEGED_TEST
  g_test_skip ("Running Xwayland in CI KVM doesn't work currently");
#else
  test_window_mapped_inhibit (META_WINDOW_CLIENT_TYPE_X11);
#endif
}

static void
test_window_mapped_inhibit_multiple_wayland (void)
{
  test_window_mapped_inhibit_multiple (META_WINDOW_CLIENT_TYPE_WAYLAND);
}

static void
test_window_mapped_inhibit_multiple_x11 (void)
{
#ifdef MUTTER_PRIVILEGED_TEST
  g_test_skip ("Running Xwayland in CI KVM doesn't work currently");
#else
  test_window_mapped_inhibit_multiple (META_WINDOW_CLIENT_TYPE_X11);
#endif
}

static void
on_before_tests (void)
{
  MetaWaylandCompositor *compositor =
    meta_context_get_wayland_compositor (test_context);

  test_driver = meta_wayland_test_driver_new (compositor);
  virtual_monitor = meta_create_test_monitor (test_context,
                                              400, 400, 60.0);
}

static void
on_after_tests (void)
{
  g_clear_object (&test_driver);
  g_clear_object (&virtual_monitor);
}

static void
init_tests (void)
{
  g_test_add_func ("/window/mapped-inhibit",
                   test_window_mapped_inhibit_wayland);
  g_test_add_func ("/window/mapped-inhibit/x11",
                   test_window_mapped_inhibit_x11);
  g_test_add_func ("/window/mapped-inhibit/multiple",
                   test_window_mapped_inhibit_multiple_wayland);
  g_test_add_func ("/window/mapped-inhibit/multiple/x11",
                   test_window_mapped_inhibit_multiple_x11);
}

int
main (int    argc,
      char **argv)
{
  g_autoptr (MetaContext) context = NULL;

  context = meta_create_test_context (META_CONTEXT_TEST_TYPE_HEADLESS,
#ifdef MUTTER_PRIVILEGED_TEST
                                      META_CONTEXT_TEST_FLAG_NO_X11 |
#endif
                                      META_CONTEXT_TEST_FLAG_TEST_CLIENT);
  g_assert_true (meta_context_configure (context, &argc, &argv, NULL));

  test_context = context;

  init_tests ();

  g_signal_connect (context, "before-tests",
                    G_CALLBACK (on_before_tests), NULL);
  g_signal_connect (context, "after-tests",
                    G_CALLBACK (on_after_tests), NULL);

  return meta_context_test_run_tests (META_CONTEXT_TEST (context),
                                      META_TEST_RUN_FLAG_CAN_SKIP);
}
