###############################################################################
# 
#  Copyright (2008) Alexander Stukowski
#
#  This file is part of OVITO (Open Visualization Tool).
#
#  OVITO is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  OVITO is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
###############################################################################

#
# This Python package is part of the highlevel scripting interface
# for the AtomViz plugin.
#

import math
import sys

from Scripting import *
from AtomViz import *


# The default mapping from columns to data channels that
# is used when exporting LAMMP dump files.
lammpsDefaultChannelMapping = ChannelColumnMapping()
lammpsDefaultChannelMapping.InsertColumn(0, DataChannelIdentifier.AtomIndexChannel, "Atom Index")
lammpsDefaultChannelMapping.InsertColumn(1, DataChannelIdentifier.AtomTypeChannel, "Type")
lammpsDefaultChannelMapping.InsertColumn(2, DataChannelIdentifier.PositionChannel, "Position", 0)
lammpsDefaultChannelMapping.InsertColumn(3, DataChannelIdentifier.PositionChannel, "Position", 1)
lammpsDefaultChannelMapping.InsertColumn(4, DataChannelIdentifier.PositionChannel, "Position", 2)

# The default mapping from columns to data channels that
# is used when exporting XYZ files.
xyzDefaultChannelMapping = ChannelColumnMapping()
xyzDefaultChannelMapping.InsertColumn(0, DataChannelIdentifier.AtomTypeChannel, "Type")
xyzDefaultChannelMapping.InsertColumn(1, DataChannelIdentifier.PositionChannel, "Position", 0)
xyzDefaultChannelMapping.InsertColumn(2, DataChannelIdentifier.PositionChannel, "Position", 1)
xyzDefaultChannelMapping.InsertColumn(3, DataChannelIdentifier.PositionChannel, "Position", 2)

def ExportLAMMPSDumpFile(filepath, binary = False, columns = lammpsDefaultChannelMapping, startFrame = 0, endFrame = 0, dataset = None):
	""" Exports the atoms in a scene to a LAMMPS dump file. 
		
		Parameters:
		
		filepath  -  The filename of the dump file to be written.
					 If this filename contains at least one '*' or '%' character
					 then it will be treated as a wildcard pattern string and a sequence of 
					 dump files be written with one file per animation frame. 
					 
		binary	-  A boolean flag that indicates whether the dump file is in binary
					 or text format.
					 
		columns   -  A ChannelColumnMapping object that maps data channels in the atoms object to
					 column indices in the output dump file.
					 
		startFrame - The animation frame from which on the atoms should be exported.   

		endFrame   - The animation frame up to which the atoms should be exported.   
					 
		dataset   -  The scene data set where the atoms are taken from. If not specified
					 then the current data set is used.

	"""
	
	if binary == True:
		writer = LAMMPSBinaryDumpWriter()
	else:
		writer = LAMMPSTextDumpWriter()
	
	# Setup exporter
	if "*" in filepath:
		writer.UseWildcardFilename = True
	else:
		writer.UseWildcardFilename = False
	writer.StartFrame = startFrame
	writer.EndFrame = endFrame
	# Define the column to data channel mapping
	writer.ChannelMapping = columns
		
	# Get the current data set
	if dataset == None: dataset = DataSetManager.Instance.CurrentSet
	  
	# Let the exporter write the atoms file.
	writer.ExportToFile(filepath, dataset, True)

def ExportXYZFile(filepath, columns = xyzDefaultChannelMapping, startFrame = 0, endFrame = 0, dataset = None):
	""" Exports the atoms in a scene to a XYZ file. 
		
		Parameters:
		
		filepath  -  The filename of the XYZ file to be written.
					 If this filename contains at least one '*' or '%' character
					 then it will be treated as a wildcard pattern string and a sequence of 
					 XYZ files be written with one file per animation frame. 
					 
		columns   -  A ChannelColumnMapping object that maps data channels in the atoms object to
					 column indices in the output XYZ file.
					 
		startFrame - The animation frame from which on the atoms should be exported.   

		endFrame   - The animation frame up to which the atoms should be exported.   
					 
		dataset   -  The scene data set where the atoms are taken from. If not specified
					 then the current data set is used.

	"""
	
	writer = XYZWriter()
	
	# Setup exporter
	if "*" in filepath:
		writer.UseWildcardFilename = True
	else:
		writer.UseWildcardFilename = False
	writer.StartFrame = startFrame
	writer.EndFrame = endFrame
	# Define the column to data channel mapping
	writer.ChannelMapping = columns
		
	# Get the current data set
	if dataset == None: dataset = DataSetManager.Instance.CurrentSet
	  
	# Let the exporter write the atoms file.
	writer.ExportToFile(filepath, dataset, True)

def ExportLAMMPSDataFile(filepath, dataset = None):
	""" Exports the atoms in a scene to a LAMMPS data file. 
		
		Parameters:
		
		filepath  -  The filename of the data file to be written.
					 
		dataset   -  The scene data set where the atoms are taken from. If not specified
					 then the current data set is used.

	"""
	
	writer = LAMMPSDataWriter()
	# Get the current data set
	if dataset == None: dataset = DataSetManager.Instance.CurrentSet
	writer.StartFrame = 0
	writer.EndFrame = 0
	# Let the exporter write the atoms file.
	writer.ExportToFile(filepath, dataset, True)

def ExportIMDAtomFile(filepath, dataset = None):
	""" Exports the atoms in a scene to an IMD atom file. 
		
		Parameters:
		
		filepath  -  The filename of the data file to be written.
					 
		dataset   -  The scene data set where the atoms are taken from. If not specified
					 then the current data set is used.

	"""
	
	writer = IMDAtomFileWriter()
	# Get the current data set
	if dataset == None: dataset = DataSetManager.Instance.CurrentSet
	writer.StartFrame = 0
	writer.EndFrame = 0
	# Let the exporter write the atoms file.
	writer.ExportToFile(filepath, dataset, True)
	